//                                               -*- C++ -*-
/**
 *  @file  ExponentialFactory.cxx
 *  @brief Factory for Exponential distribution
 *
 *  (C) Copyright 2005-2011 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: schueller $
 *  @date:   $LastChangedDate: 2011-05-24 19:30:41 +0200 (Tue, 24 May 2011) $
 *  Id:      $Id: ExponentialFactory.cxx 1910 2011-05-24 17:30:41Z schueller $
 */
#include "ExponentialFactory.hxx"

namespace OpenTURNS {

  namespace Uncertainty {

    namespace Distribution {

      /* Default constructor */
      ExponentialFactory::ExponentialFactory():
        DistributionImplementationFactory()
      {
        // Nothing to do
      }

      /* Virtual constructor */
      ExponentialFactory * ExponentialFactory::clone() const
      {
        return new ExponentialFactory(*this);
      }

      /* Here is the interface that all derived class must implement */

      Exponential * ExponentialFactory::build(const NumericalSample & sample) const /* throw(InvalidArgumentException, InternalException) */
      {
        const UnsignedLong size(sample.getSize());
        if (size == 0) throw InvalidArgumentException(HERE) << "Error: cannot build a Exponential distribution from an empty sample";
        if (sample.getDimension() != 1) throw InvalidArgumentException(HERE) << "Error: can build an Exponential distribution only from a sample of dimension 1, here dimension=" << sample.getDimension();
        const NumericalScalar xMin(sample.getMin()[0]);
        const NumericalScalar gamma(xMin - fabs(xMin) / (2.0 + size));
        const NumericalScalar mean(sample.computeMean()[0]);
        if (mean == gamma) throw InternalException(HERE) << "Error: can build an Exponential distribution only from a sample with mean > min, here mean=" << mean << " and min=" << gamma;
        const NumericalScalar lambda(1.0 / (mean - gamma));
        return Exponential(lambda, gamma).clone();
      }

      Exponential * ExponentialFactory::build(const NumericalPointCollection & parameters) const /* throw(InvalidArgumentException, InternalException) */
      {
        try {
          Exponential distribution;
          distribution.setParametersCollection(parameters);
          return distribution.clone();
        }
        catch (InvalidArgumentException & ex)
          {
            throw InternalException(HERE) << "Error: cannot build a Exponential distribution from the given parameters";
          }
      }

      Exponential * ExponentialFactory::build() const /* throw(InvalidArgumentException, InternalException) */
      {
        return Exponential().clone();
      }


    } /* namespace Distribution */
  } /* namespace Uncertainty */
} /* namespace OpenTURNS */
