//                                               -*- C++ -*-
/**
 *  @file  NonCentralChiSquare.cxx
 *  @brief The NonCentralChiSquare distribution
 *
 *  (C) Copyright 2005-2011 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: lebrun $
 *  @date:   $LastChangedDate: 2011-02-26 21:12:17 +0100 (sam. 26 févr. 2011) $
 *  Id:      $Id: NonCentralChiSquare.cxx 1813 2011-02-26 20:12:17Z lebrun $
 */
#include <cmath>
#include "NonCentralChiSquare.hxx"
#include "RandomGenerator.hxx"
#include "SpecFunc.hxx"
#include "DistFunc.hxx"
#include "PersistentObjectFactory.hxx"

namespace OpenTURNS {

  namespace Uncertainty {

    namespace Distribution {

      CLASSNAMEINIT(NonCentralChiSquare);

      static Base::Common::Factory<NonCentralChiSquare> RegisteredFactory("NonCentralChiSquare");

      typedef Base::Func::SpecFunc        SpecFunc;
      typedef Base::Stat::RandomGenerator RandomGenerator;

      /* Default constructor */
      NonCentralChiSquare::NonCentralChiSquare(const NumericalScalar nu,
                                               const NumericalScalar lambda)
        /* throw(InvalidArgumentException) */
        : NonEllipticalDistribution("NonCentralChiSquare"),
          nu_(0.0),
          lambda_(0.0)
      {
        // This call set also the range.
        setNuLambda(nu, lambda);
        setDimension(1);
      }

      /* Comparison operator */
      Bool NonCentralChiSquare::operator ==(const NonCentralChiSquare & other) const
      {
        if (this == &other) return true;
        return (nu_ == other.nu_) && (lambda_ == other.lambda_);
      }

      /* String converter */
      String NonCentralChiSquare::__repr__() const {
        OSS oss;
        oss << "class=" << NonCentralChiSquare::GetClassName()
            << " name=" << getName()
            << " dimension=" << getDimension()
            << " nu=" << nu_
            << " lambda=" << lambda_;
        return oss;
      }

      String NonCentralChiSquare::__str__(const String & offset) const
      {
        OSS oss;
        oss << offset << getClassName() << "(nu = " << nu_ << ", lambda = " << lambda_ << ")";
        return oss;
      }

      /* Virtual constructor */
      NonCentralChiSquare * NonCentralChiSquare::clone() const
      {
        return new NonCentralChiSquare(*this);
      }

      /* Get one realization of the distribution */
      NonCentralChiSquare::NumericalPoint NonCentralChiSquare::getRealization() const
      {
        return NumericalPoint(1, DistFunc::rNonCentralChiSquare(nu_, lambda_));
      }

      /* Get the PDF of the distribution */
      NumericalScalar NonCentralChiSquare::computePDF(const NumericalPoint & point) const
      {
        pdfEpsilon_ = DistFunc::Precision;
        return DistFunc::dNonCentralChiSquare(nu_, lambda_, point[0]);
      }

      /* Get the CDF of the distribution */
      NumericalScalar NonCentralChiSquare::computeCDF(const NumericalPoint & point,
                                                      const Bool tail) const
      {
        cdfEpsilon_ = DistFunc::Precision;
        return DistFunc::pNonCentralChiSquare(nu_, lambda_, point[0], tail);
      }

      /** Get the PDFGradient of the distribution */
      NonCentralChiSquare::NumericalPoint NonCentralChiSquare::computePDFGradient(const NumericalPoint & point) const
      {
        NumericalScalar eps(pow(DistFunc::Precision, 1.0 / 3.0));
        NumericalPoint pdfGradient(2);
        pdfGradient[0] = (DistFunc::dNonCentralChiSquare(nu_ + eps, lambda_, point[0]) - DistFunc::dNonCentralChiSquare(nu_ - eps, lambda_, point[0])) / (2.0 * eps);
        pdfGradient[1] = (DistFunc::dNonCentralChiSquare(nu_, lambda_ + eps, point[0]) - DistFunc::dNonCentralChiSquare(nu_, lambda_ - eps, point[0])) / (2.0 * eps);
        return pdfGradient;
      }

      /** Get the CDFGradient of the distribution */
      NonCentralChiSquare::NumericalPoint NonCentralChiSquare::computeCDFGradient(const NumericalPoint & point) const
      {
        NumericalScalar eps(pow(DistFunc::Precision, 1.0 / 3.0));
        NumericalPoint cdfGradient(2);
        cdfGradient[0] = (DistFunc::pNonCentralChiSquare(nu_ + eps, lambda_, point[0]) - DistFunc::pNonCentralChiSquare(nu_ - eps, lambda_, point[0])) / (2.0 * eps);
        cdfGradient[1] = (DistFunc::pNonCentralChiSquare(nu_, lambda_ + eps, point[0]) - DistFunc::pNonCentralChiSquare(nu_, lambda_ - eps, point[0])) / (2.0 * eps);
        return cdfGradient;
      }

      /* Compute the mean of the distribution */
      void NonCentralChiSquare::computeMean() const
      {
        mean_ = NumericalPoint(1, nu_ + lambda_);
        isAlreadyComputedMean_ = true;
      }

      /* Get the standard deviation of the distribution */
      NonCentralChiSquare::NumericalPoint NonCentralChiSquare::getStandardDeviation() const /* throw(NotDefinedException) */
      {
        return NumericalPoint(1, sqrt(2.0 * (nu_ + 2.0 * lambda_)));
      }

      /* Get the skewness of the distribution */
      NonCentralChiSquare::NumericalPoint NonCentralChiSquare::getSkewness() const /* throw(NotDefinedException) */
      {
        if (nu_ == 0.0) throw NotDefinedException(HERE) << "Error: the skewness is not defined for nu=0.";
        if (lambda_ == 0.0) return NumericalPoint(1, 2.0 * M_SQRT2 / sqrt(nu_));
        return NumericalPoint(1, (nu_ + 3.0 * lambda_) * pow(2.0 / (nu_ + 2.0 * lambda_), 1.5));
      }

      /* Get the kurtosis of the distribution */
      NonCentralChiSquare::NumericalPoint NonCentralChiSquare::getKurtosis() const /* throw(NotDefinedException) */
      {
        if (nu_ == 0.0) throw NotDefinedException(HERE) << "Error: the kurtosis is not defined for nu=0.";
        if (lambda_ == 0.0) return NumericalPoint(1, 3.0 + 12.0 / nu_);
        return NumericalPoint(1, 3.0 + 12.0 * (nu_ + 4.0 * lambda_) / pow(nu_ + 2.0 * lambda_, 2.0));
      }

      /* Compute the covariance of the distribution */
      void NonCentralChiSquare::computeCovariance() const
      {
        covariance_ = CovarianceMatrix(1);
        covariance_(0, 0) =  2.0 * (nu_ + 2.0 * lambda_);
        isAlreadyComputedCovariance_ = true;
      }

      /* Get the characteristic function of the distribution, i.e. phi(u) = E(exp(I*u*X)) */
      NumericalComplex NonCentralChiSquare::computeCharacteristicFunction(const NumericalScalar x,
                                                                          const Bool logScale) const
      {
        const NumericalComplex denominator(1.0, -2.0 * x);
        if (logScale) return NumericalComplex(0.0, lambda_ * x) / denominator - 0.5 * nu_ * log(denominator);
        return exp(NumericalComplex(0.0, lambda_ * x) / denominator - 0.5 * nu_ * log(denominator));
      }

      /* Parameters value and description accessor */
      NonCentralChiSquare::NumericalPointWithDescriptionCollection NonCentralChiSquare::getParametersCollection() const
      {
        NumericalPointWithDescriptionCollection parameters(1);
        NumericalPointWithDescription point(2);
        Description description(point.getDimension());
        point[0] = nu_;
        point[1] = lambda_;
        description[0] = "nu";
        description[1] = "lambda";
        point.setDescription(description);
        point.setName(getDescription()[0]);
        parameters[0] = point;
        return parameters;
      }

      void NonCentralChiSquare::setParametersCollection(const NumericalPointCollection & parametersCollection)
      {
        *this = NonCentralChiSquare(parametersCollection[0][0], parametersCollection[0][1]);
      }

      /* Nu accessor */
      void NonCentralChiSquare::setNu(const NumericalScalar nu)
      /* throw(InvalidArgumentException) */
      {
        if (nu < 0.0) throw InvalidArgumentException(HERE) << "Nu MUST be strictly positive";
        if (nu != nu_)
          {
            nu_ = nu;
            isAlreadyComputedMean_ = false;
            isAlreadyComputedCovariance_ = false;
            computeRange();
          }
      }

      void NonCentralChiSquare::setNuLambda(const NumericalScalar nu,
                                            const NumericalScalar lambda)
      /* throw(InvalidArgumentException) */
      {
        if (nu < 0.0) throw InvalidArgumentException(HERE) << "Nu MUST be strictly positive";
        if (lambda < 0.0) throw InvalidArgumentException(HERE) << "Lambda MUST be strictly positive";
        if ((nu != nu_) || (lambda != lambda_))
          {
            nu_ = nu;
            lambda_ = lambda;
            isAlreadyComputedMean_ = false;
            isAlreadyComputedCovariance_ = false;
            computeRange();
          }
      }

      /* Nu accessor */
      NumericalScalar NonCentralChiSquare::getNu() const
      {
        return nu_;
      }


      /* Lambda accessor */
      void NonCentralChiSquare::setLambda(const NumericalScalar lambda)
      {
        if (lambda < 0.0) throw InvalidArgumentException(HERE) << "Lambda MUST be strictly positive";
        if (lambda != lambda_)
          {
            lambda_ = lambda;
            isAlreadyComputedMean_ = false;
            isAlreadyComputedCovariance_ = false;
            computeRange();
          }
      }

      /* Lambda accessor */
      NumericalScalar NonCentralChiSquare::getLambda() const
      {
        return lambda_;
      }

      /* Method save() stores the object through the StorageManager */
      void NonCentralChiSquare::save(StorageManager::Advocate & adv) const
      {
        NonEllipticalDistribution::save(adv);
        adv.saveAttribute( "nu_", nu_ );
        adv.saveAttribute( "lambda_", lambda_ );
      }

      /* Method load() reloads the object from the StorageManager */
      void NonCentralChiSquare::load(StorageManager::Advocate & adv)
      {
        NonEllipticalDistribution::load(adv);
        adv.loadAttribute( "nu_", nu_ );
        adv.loadAttribute( "lambda_", lambda_ );
        computeRange();
      }

    } /* namespace Distribution */
  } /* namespace Uncertainty */
} /* namespace OpenTURNS */
