//                                               -*- C++ -*-
/**
 *  @file  NormalCopula.hxx
 *  @brief A class that implements a normal copula
 *
 *  (C) Copyright 2005-2011 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: souchaud $
 *  @date:   $LastChangedDate: 2011-07-01 10:34:36 +0200 (Fri, 01 Jul 2011) $
 *  Id:      $Id: NormalCopula.hxx 1981 2011-07-01 08:34:36Z souchaud $
 */
#ifndef OPENTURNS_NORMALCOPULA_HXX
#define OPENTURNS_NORMALCOPULA_HXX

#include "CopulaImplementation.hxx"
#include "CorrelationMatrix.hxx"
#include "Normal.hxx"
#include "Exception.hxx"

namespace OpenTURNS {

  namespace Uncertainty {

    namespace Distribution {


      /**
       * @class NormalCopula
       *
       * The class implements a normal copula
       */
      class NormalCopula
        : public Model::CopulaImplementation
      {
        CLASSNAME;

      public:
        typedef Model::CopulaImplementation                                 CopulaImplementation; // required by SWIG
        typedef CopulaImplementation::NotDefinedException                   NotDefinedException;
        typedef CopulaImplementation::NumericalPoint                        NumericalPoint;
        typedef CopulaImplementation::NumericalSample                       NumericalSample;
        typedef CopulaImplementation::Implementation                        Implementation;
        typedef CopulaImplementation::CovarianceMatrix                      CovarianceMatrix;
        typedef CopulaImplementation::IsoProbabilisticTransformation        IsoProbabilisticTransformation;
        typedef CopulaImplementation::InverseIsoProbabilisticTransformation InverseIsoProbabilisticTransformation;
        typedef CopulaImplementation::NumericalPointWithDescriptionCollection              NumericalPointWithDescriptionCollection;
        typedef CopulaImplementation::Indices                               Indices;
        typedef CopulaImplementation::Interval                              Interval;
        typedef CopulaImplementation::Description                           Description;
        typedef Base::Stat::CorrelationMatrix                               CorrelationMatrix;
        typedef Base::Common::NotSymmetricDefinitePositiveException         NotSymmetricDefinitePositiveException;
        typedef CopulaImplementation::StorageManager                        StorageManager;

        /** Default constructor */
        explicit NormalCopula(const UnsignedLong dim = 1);

        /** Default constructor */
        explicit NormalCopula(const CorrelationMatrix & correlation);


        /** Comparison operator */
        Bool operator ==(const NormalCopula & other) const;

        /** String converter */
        String __repr__() const;
        String __str__(const String & offset = "") const;

        /* Interface inherited from Distribution */

        /** Virtual constructor */
        virtual NormalCopula * clone() const;

        /** Get one realization of the NormalCopula distribution */
        NumericalPoint getRealization() const;

        /** Get the DDF of the NormalCopula distribution */
        using CopulaImplementation::computeDDF;
        NumericalPoint computeDDF(const NumericalPoint & point) const;

        /** Get the PDF of the NormalCopula distribution */
        using CopulaImplementation::computePDF;
        NumericalScalar computePDF(const NumericalPoint & point) const;

        /** Get the CDF of the NormalCopula distribution */
        using CopulaImplementation::computeCDF;
        NumericalScalar computeCDF(const NumericalPoint & point, const Bool tail = false) const;

        /** Get the probability content of an interval */
        NumericalScalar computeProbability(const Interval & interval) const;

        /** Get the PDF gradient of the distribution */
        NumericalPoint computePDFGradient(const NumericalPoint & point) const;

        /** Get the CDF gradient of the distribution */
        NumericalPoint computeCDFGradient(const NumericalPoint & point) const;

        /** Compute the PDF of Xi | X1, ..., Xi-1. x = Xi, y = (X1,...,Xi-1) */
        NumericalScalar computeConditionalPDF(const NumericalScalar x,
                                              const NumericalPoint & y) const;

        /** Compute the CDF of Xi | X1, ..., Xi-1. x = Xi, y = (X1,...,Xi-1) */
        NumericalScalar computeConditionalCDF(const NumericalScalar x,
                                              const NumericalPoint & y) const;

        /** Compute the quantile of Xi | X1, ..., Xi-1, i.e. x such that CDF(x|y) = q with x = Xi, y = (X1,...,Xi-1) */
        NumericalScalar computeConditionalQuantile(const NumericalScalar q,
                                                   const NumericalPoint & y) const;

        /** Get the distribution of the marginal distribution corresponding to indices dimensions */
        using CopulaImplementation::getMarginal;
        NormalCopula * getMarginal(const Indices & indices) const /* throw(InvalidArgumentException) */;

        /** Get the isoprobabilist transformation */
        IsoProbabilisticTransformation getIsoProbabilisticTransformation() const;

        /** Get the inverse isoprobabilist transformation */
        InverseIsoProbabilisticTransformation getInverseIsoProbabilisticTransformation() const;

        /** Tell if the distribution has independent copula */
        Bool hasIndependentCopula() const;

        /** Parameters value and description accessor */
        NumericalPointWithDescriptionCollection getParametersCollection() const;

        /** Compute the correlation matrix of a Normal Copula from its Spearman correlation matrix */
        static CorrelationMatrix GetCorrelationFromSpearmanCorrelation(const CorrelationMatrix & matrix) /* throw(NotSymmetricDefinitePositiveException) */;

        /** Compute the correlation matrix of a Normal Copula from its Kendall correlation matrix */
        static CorrelationMatrix GetCorrelationFromKendallCorrelation(const CorrelationMatrix & matrix) /* throw(NotSymmetricDefinitePositiveException) */;

        /** Method save() stores the object through the StorageManager */
        virtual void save(StorageManager::Advocate & adv) const;

        /** Method load() reloads the object from the StorageManager */
        virtual void load(StorageManager::Advocate & adv);

      protected:


      private:
        // Normal copula parameter
        CorrelationMatrix correlation_;
        // Underlying generic representative
        Normal normal_;

      }; /* class NormalCopula */


    } /* namespace Distribution */
  } /* namespace Uncertainty */
} /* namespace OpenTURNS */

#endif /* OPENTURNS_NORMALCOPULA_HXX */
