//                                               -*- C++ -*-
/**
 *  @file  Student.hxx
 *  @brief The Student distribution
 *
 *  (C) Copyright 2005-2011 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: souchaud $
 *  @date:   $LastChangedDate: 2011-07-01 10:34:36 +0200 (Fri, 01 Jul 2011) $
 *  Id:      $Id: Student.hxx 1981 2011-07-01 08:34:36Z souchaud $
 */
#ifndef OPENTURNS_STUDENT_HXX
#define OPENTURNS_STUDENT_HXX

#include "OTprivate.hxx"
#include "EllipticalDistribution.hxx"
#include "Exception.hxx"

namespace OpenTURNS {

  namespace Uncertainty {

    namespace Distribution {


      /**
       * @class Student
       *
       * The Student distribution.
       */
      class Student
        : public Model::EllipticalDistribution
      {
        // Maximum number of samples in the evaluation of the CDF
        static const UnsignedLong MaximumNumberOfPoints; // 10000000
        // Minimum number of samples in the evaluation of the CDF
        static const UnsignedLong MinimumNumberOfPoints; // 1000000
        // Maximum precision required for the CDF in the multidimensional case (dim > 3)
        static const NumericalScalar MaximumCDFEpsilon; // 5.0e-6
        // Minimum precision required for the CDF in the multidimensional case (dim > 3)
        static const NumericalScalar MinimumCDFEpsilon; // 5.0e-2

        CLASSNAME;
      public:

        typedef Base::Common::InvalidArgumentException            InvalidArgumentException;
        typedef Model::EllipticalDistribution                     EllipticalDistribution;  // required by SWIG
        typedef EllipticalDistribution::NumericalPoint            NumericalPoint;
        typedef EllipticalDistribution::NumericalSample           NumericalSample;
        typedef EllipticalDistribution::CovarianceMatrix          CovarianceMatrix;
        typedef EllipticalDistribution::Indices                   Indices;
        typedef EllipticalDistribution::Interval                  Interval;
        typedef EllipticalDistribution::NumericalPointWithDescriptionCollection  NumericalPointWithDescriptionCollection;
        typedef EllipticalDistribution::NotDefinedException       NotDefinedException;
        typedef EllipticalDistribution::StorageManager            StorageManager;


        /** Default constructor */
        Student(const NumericalScalar nu = 3.0,
                const UnsignedLong dimension = 1);

        /** Parameters constructor */
        Student(const NumericalScalar nu,
                const NumericalScalar mu,
                const NumericalScalar sigma = 1.0)
          /* throw(InvalidArgumentException) */;

        /** Parameters constructor */
        Student(const NumericalScalar nu,
                const NumericalPoint & mu,
                const NumericalPoint & sigma,
                const CorrelationMatrix & R)
          /* throw(InvalidArgumentException) */;


        /** Comparison operator */
        Bool operator ==(const Student & other) const;

        /** String converter */
        String __repr__() const;
        String __str__(const String & offset = "") const;



        /* Interface inherited from Distribution */

        /** Virtual constructor */
        virtual Student * clone() const;

        /** Get one realization of the distribution */
        NumericalPoint getRealization() const;

        /** Get the CDF of the distribution */
        using EllipticalDistribution::computeCDF;
        NumericalScalar computeCDF(const NumericalPoint & point, const Bool tail = false) const;

        /** Get the probability content of an interval */
        NumericalScalar computeProbability(const Interval & interval) const;

        /** Get the PDFGradient of the distribution */
        NumericalPoint computePDFGradient(const NumericalPoint & point) const;

        /** Get the CDFGradient of the distribution */
        NumericalPoint computeCDFGradient(const NumericalPoint & point) const;

        /** Compute the radial distribution CDF */
        NumericalScalar computeRadialDistributionCDF(const NumericalScalar radius,
                                                     const Bool tail = false) const;

        /** Compute the PDF of Xi | X1, ..., Xi-1. x = Xi, y = (X1,...,Xi-1) */
        NumericalScalar computeConditionalPDF(const NumericalScalar x, const NumericalPoint & y) const;

        /** Compute the CDF of Xi | X1, ..., Xi-1. x = Xi, y = (X1,...,Xi-1) */
        NumericalScalar computeConditionalCDF(const NumericalScalar x, const NumericalPoint & y) const;

        /** Compute the quantile of Xi | X1, ..., Xi-1, i.e. x such that CDF(x|y) = q with x = Xi, y = (X1,...,Xi-1) */
        NumericalScalar computeConditionalQuantile(const NumericalScalar q, const NumericalPoint & y) const;

        /** Get the i-th marginal distribution */
        Student * getMarginal(const UnsignedLong i) const /* throw(InvalidArgumentException) */;

        /** Get the distribution of the marginal distribution corresponding to indices dimensions */
        Student * getMarginal(const Indices & indices) const /* throw(InvalidArgumentException) */;

        /** Get the mean of the distribution */
        NumericalPoint getMean() const /* throw(NotDefinedException) */;

        /** Get the standard deviation of the distribution */
        NumericalPoint getStandardDeviation() const /* throw(NotDefinedException) */;

        /** Get the skewness of the distribution */
        NumericalPoint getSkewness() const /* throw(NotDefinedException) */;

        /** Get the kurtosis of the distribution */
        NumericalPoint getKurtosis() const /* throw(NotDefinedException) */;

        /** Get the covariance of the distribution */
        CovarianceMatrix getCovariance() const /* throw(NotDefinedException) */;

        /** Get the raw moments of the standardized distribution */
        NumericalPoint getStandardMoment(const UnsignedLong n) const;

        /** Parameters value and description accessor */
        NumericalPointWithDescriptionCollection getParametersCollection() const;
        using EllipticalDistribution::setParametersCollection;
        void setParametersCollection(const NumericalPointCollection & parametersCollection);

        /** Tell if the distribution has independent copula */
        Bool hasIndependentCopula() const;

        /* Interface specific to Student */

        /** Nu accessor */
        void setNu(const NumericalScalar nu)
          /* throw(InvalidArgumentException) */;
        NumericalScalar getNu() const;

        /** Mu accessor */
        void setMu(const NumericalScalar mu);
        NumericalScalar getMu() const;

        /** Compute the density generator of the ellipticalal generator, i.e.
         *  the function phi such that the density of the distribution can
         *  be written as p(x) = phi(t(x-mu)R^(-1)(x-mu))                      */
        NumericalScalar computeDensityGenerator(const NumericalScalar betaSquare) const;

        /** Compute the derivative of the density generator */
        NumericalScalar computeDensityGeneratorDerivative(const NumericalScalar betaSquare) const;

        /** Compute the seconde derivative of the density generator */
        NumericalScalar computeDensityGeneratorSecondDerivative(const NumericalScalar betaSquare) const;

        /** Method save() stores the object through the StorageManager */
        void save(StorageManager::Advocate & adv) const;

        /** Method load() reloads the object from the StorageManager */
        void load(StorageManager::Advocate & adv);

      protected:

      private:

        /** Quantile computation for dimension=1 */
        NumericalScalar computeScalarQuantile(const NumericalScalar prob,
                                              const Bool tail = false,
                                              const NumericalScalar precision = DefaultQuantileEpsilon) const;

        /** The nu of the Student distribution */
        NumericalScalar nu_;

        /** Specific normalization factor for the Student distribution */
        NumericalScalar studentNormalizationFactor_;
      }; /* class Student */


    } /* namespace Distribution */
  } /* namespace Uncertainty */
} /* namespace OpenTURNS */

#endif /* OPENTURNS_STUDENT_HXX */
