//                                               -*- C++ -*-
/**
 *  @file  Uniform.hxx
 *  @brief The Uniform distribution
 *
 *  (C) Copyright 2005-2011 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: souchaud $
 *  @date:   $LastChangedDate: 2011-07-01 10:34:36 +0200 (Fri, 01 Jul 2011) $
 *  Id:      $Id: Uniform.hxx 1981 2011-07-01 08:34:36Z souchaud $
 */
#ifndef OPENTURNS_UNIFORM_HXX
#define OPENTURNS_UNIFORM_HXX

#include "NonEllipticalDistribution.hxx"

namespace OpenTURNS {

  namespace Uncertainty {

    namespace Distribution {


      /**
       * @class Uniform
       *
       * The Uniform distribution.
       */
      class Uniform
        : public Model::NonEllipticalDistribution
      {
        CLASSNAME;
      public:

        typedef Base::Common::InvalidArgumentException               InvalidArgumentException;
        typedef Model::NonEllipticalDistribution                     NonEllipticalDistribution;  // required by SWIG
        typedef NonEllipticalDistribution::NumericalPoint            NumericalPoint;
        typedef NonEllipticalDistribution::NumericalSample           NumericalSample;
        typedef NonEllipticalDistribution::CovarianceMatrix          CovarianceMatrix;
        typedef NonEllipticalDistribution::NumericalPointWithDescriptionCollection  NumericalPointWithDescriptionCollection;
        typedef NonEllipticalDistribution::NotDefinedException       NotDefinedException;
        typedef NonEllipticalDistribution::StorageManager            StorageManager;

        /** Default constructor */
        Uniform();

        /** Parameters constructor */
        Uniform(const NumericalScalar a,
                const NumericalScalar b);


        /** Comparison operator */
        Bool operator ==(const Uniform & other) const;

        /** String converter */
        String __repr__() const;
        String __str__(const String & offset = "") const;



        /* Interface inherited from Distribution */

        /** Virtual constructor */
        Uniform * clone() const;

        /** Get one realization of the distribution */
        NumericalPoint getRealization() const;

        /** Get the DDF of the distribution, i.e. the gradient of its PDF w.r.t. point */
        using NonEllipticalDistribution::computeDDF;
        NumericalPoint computeDDF(const NumericalPoint & point) const;

        /** Get the PDF of the distribution, i.e. P(point < X < point+dx) = PDF(point)dx + o(dx) */
        using NonEllipticalDistribution::computePDF;
        NumericalScalar computePDF(const NumericalPoint & point) const;

        /** Get the CDF of the distribution, i.e. P(X <= point) = CDF(point) */
        using NonEllipticalDistribution::computeCDF;
        NumericalScalar computeCDF(const NumericalPoint & point, const Bool tail = false) const;

        /** Get the characteristic function of the distribution, i.e. phi(u) = E(exp(I*u*X)) */
        NumericalComplex computeCharacteristicFunction(const NumericalScalar x,
                                                       const Bool logScale = false) const;

        /** Get the gradient of the PDF w.r.t the parameters of the distribution */
        NumericalPoint computePDFGradient(const NumericalPoint & point) const;

        /** Get the gradient of the CDF w.r.t the parameters of the distribution */
        NumericalPoint computeCDFGradient(const NumericalPoint & point) const;

        /** Get the roughness, i.e. the L2-norm of the PDF */
        NumericalScalar getRoughness() const;

        /** Get the standard deviation of the distribution */
        NumericalPoint getStandardDeviation() const /* throw(NotDefinedException) */;

        /** Get the skewness of the distribution */
        NumericalPoint getSkewness() const /* throw(NotDefinedException) */;

        /** Get the kurtosis of the distribution */
        NumericalPoint getKurtosis() const /* throw(NotDefinedException) */;

        /** Get the raw moments of the standardized distribution */
        NumericalPoint getStandardMoment(const UnsignedLong n) const;

        /** Parameters value and description accessor */
        NumericalPointWithDescriptionCollection getParametersCollection() const;
        using NonEllipticalDistribution::setParametersCollection;
        void setParametersCollection(const NumericalPointCollection & parametersCollection);

        /** Check if the distribution is elliptical */
        Bool isElliptical() const;

        /* Interface specific to Uniform */

        /** A accessor */
        void setA(const NumericalScalar a);
        NumericalScalar getA() const;

        /** B accessor */
        void setB(const NumericalScalar b);
        NumericalScalar getB() const;

        /** Method save() stores the object through the StorageManager */
        void save(StorageManager::Advocate & adv) const;

        /** Method load() reloads the object from the StorageManager */
        void load(StorageManager::Advocate & adv);


      protected:

      private:

        /** Compute the mean of the distribution */
        void computeMean() const;

        /** Compute the covariance of the distribution */
        void computeCovariance() const;

        /** Get the quantile of the distribution, i.e the value Xp such that P(X <= Xp) = prob */
        NumericalScalar computeScalarQuantile(const NumericalScalar prob,
                                              const Bool tail = false,
                                              const NumericalScalar precision = DefaultQuantileEpsilon) const;

        /** Compute the numerical range of the distribution given the parameters values */
        void computeRange();

        /** The main parameter set of the distribution */
        NumericalScalar a_;
        NumericalScalar b_;

      }; /* class Uniform */


    } /* namespace Distribution */
  } /* namespace Uncertainty */
} /* namespace OpenTURNS */

#endif /* OPENTURNS_UNIFORM_HXX */
