//                                               -*- C++ -*-
/**
 *  @file  t_BarPlot_std.cxx
 *  @brief The test file of class BarPlot for standard methods
 *
 *  (C) Copyright 2005-2011 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: schueller $
 *  @date:   $LastChangedDate: 2011-05-24 19:30:41 +0200 (Tue, 24 May 2011) $
 *  Id:      $Id: t_BarPlot_std.cxx 1910 2011-05-24 17:30:41Z schueller $
 */
#include <iostream>
#include <sstream>
#include <cstdlib>
#include <cmath>

#include "OT.hxx"
#include "OTtestcode.hxx"
#include "OStream.hxx"
#include "Normal.hxx"
#include "NumericalPoint.hxx"
#include "NumericalSample.hxx"
#include "CorrelationMatrix.hxx"
#include "IdentityMatrix.hxx"
#include "Graph.hxx"
#include "BarPlot.hxx"

using namespace OT;
using namespace OT::Test;
using namespace OT::Base::Common;
using namespace OT::Base::Type;
using namespace OT::Base::Stat;
using namespace OT::Base::Graph;
using namespace OT::Uncertainty::Distribution;
using namespace std;


int main(int argc, char *argv[])
{
  TESTPREAMBLE;
  OStream fullprint(std::cout);
  setRandomGenerator();

  try {

    // Instanciate one distribution object
    UnsignedLong dim(1);
    NumericalPoint meanPoint(dim, 1.0);
    meanPoint[0] = 0.5;
    NumericalPoint sigma(dim, 1.0);
    sigma[0] = 2.0;
    CorrelationMatrix R = IdentityMatrix(dim);
    Normal distribution1(meanPoint, sigma, R);

    // Instanciate another distribution object
    meanPoint[0] = -1.5;
    sigma[0] = 4.0;
    Normal distribution2(meanPoint, sigma, R);

    // Test for sampling
    UnsignedLong size(2000);
    UnsignedLong nBars(20);
    NumericalSample sample1(distribution1.getNumericalSample( size ));
    NumericalSample sample2(distribution2.getNumericalSample( size ));


    // Construct histograms
    NumericalScalar epsilon = 0.1;
    NumericalScalar min1 = sample1.getMin()[0];
    NumericalScalar max1 = sample1.getMax()[0] + epsilon;
    NumericalScalar min2 = sample2.getMin()[0];
    NumericalScalar max2 = sample2.getMax()[0] + epsilon;
    NumericalPoint tmp(2);
    tmp[0] = (max1-min1)/nBars;
    tmp[1] = 0;
    NumericalSample data1(nBars,tmp);
    tmp[0] = (max2-min2)/nBars;
    NumericalSample data2(nBars,tmp);
    UnsignedLong index;

    for(UnsignedLong i = 0; i < size; i++)
      {
        index = UnsignedLong(floor((sample1[i][0]-min1)/(max1-min1)*nBars));
        data1[index][1]++;
        index = UnsignedLong(floor((sample2[i][0]-min2)/(max2-min2)*nBars));
        data2[index][1]++;
      }

    // Create an empty graph
    Graph myGraph("Some barplots", "y", "frequency", true,"topleft");

    // Create the first barplot
    BarPlot myBarPlot1(data1, min1, "blue", "shaded", "dashed", "histogram1");

    // Then, draw it
    myGraph.add(myBarPlot1);
    myGraph.draw("Graph_BarPlot_a_OT", 640, 480);

    // Check that the correct files have been generated by computing their checksum
    fullprint << "bitmap=" << myGraph.getBitmap() << std::endl;
    fullprint << "postscript=" << myGraph.getPostscript() << std::endl;

    // Create the second barplot
    BarPlot myBarPlot2(data2, min2, "red", "solid", "solid", "histogram2");

    // Add it to the graph and draw everything
    myGraph.add(myBarPlot2);
    myGraph.draw("Graph_BarPlot_b_OT", 640, 480);
    fullprint << "bitmap=" << myGraph.getBitmap() << std::endl;
    fullprint << "postscript=" << myGraph.getPostscript() << std::endl;
  }
  catch (TestFailed & ex) {
    std::cerr << ex << std::endl;
    return ExitCode::Error;
  }


  return ExitCode::Success;
}
