//                                               -*- C++ -*-
/**
 *  @file  t_NonCentralStudent_std.cxx
 *  @brief The test file of class NonCentralStudent for standard methods
 *
 *  (C) Copyright 2005-2011 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: souchaud $
 *  @date:   $LastChangedDate: 2011-07-01 10:34:36 +0200 (Fri, 01 Jul 2011) $
 *  Id:      $Id: t_NonCentralStudent_std.cxx 1981 2011-07-01 08:34:36Z souchaud $
 */
#include <iostream>
#include <sstream>
#include "OT.hxx"
#include "OTtestcode.hxx"
#include "OStream.hxx"
#include "NonCentralStudent.hxx"
#include "NumericalPoint.hxx"
#include "NumericalSample.hxx"

using namespace OT;
using namespace OT::Test;
using namespace OT::Base::Common;
using namespace OT::Base::Type;
using namespace OT::Base::Stat;
using namespace OT::Uncertainty::Distribution;


class TestObject : public NonCentralStudent
{
public:
  explicit TestObject() : NonCentralStudent(7.2, 4.8, -3.7) {}
  explicit TestObject(const String & name) : NonCentralStudent(7.2, 4.8, -3.7) {setName(name);}
  virtual ~TestObject() {}
};


int main(int argc, char *argv[])
{
  TESTPREAMBLE;
  OStream fullprint(std::cout);
  setRandomGenerator();

  try {
    // Test basic functionnalities
    checkClassWithClassName<TestObject>();

    // Test some extra functionnalities
    checkNameFeature<TestObject>();

    // Instanciate one distribution object
    NonCentralStudent distribution(7.2, 4.8, -3.7);
    fullprint << "Distribution " << distribution << std::endl;
    std::cout << "Distribution " << distribution << std::endl;

    // Is this distribution elliptical ?
    fullprint << "Elliptical = " << (distribution.isElliptical() ? "true" : "false") << std::endl;

    // Is this distribution continuous ?
    fullprint << "Continuous = " << (distribution.isContinuous() ? "true" : "false") << std::endl;

    // Test for realization of distribution
    NumericalPoint oneRealization = distribution.getRealization();
    fullprint << "oneRealization=" << oneRealization << std::endl;

    // Test for sampling
    UnsignedLong size = 10000;
    NumericalSample oneSample = distribution.getNumericalSample( size );
    fullprint << "oneSample first=" << oneSample[0] << " last=" << oneSample[size - 1] << std::endl;
    fullprint << "mean=" << oneSample.computeMean() << std::endl;
    fullprint << "covariance=" << oneSample.computeCovariance() << std::endl;

    // Define a point
    NumericalPoint point( distribution.getDimension(), 1.0 );
    fullprint << "Point= " << point << std::endl;

    // Show PDF and CDF of point
    NumericalScalar eps(1e-5);
    NumericalPoint DDF = distribution.computeDDF( point );
    fullprint << "ddf     =" << DDF << std::endl;
    fullprint << "ddf (FD)=" << NumericalPoint(1, (distribution.computePDF( point + NumericalPoint(1, eps) ) - distribution.computePDF( point  + NumericalPoint(1, -eps) )) / (2.0 * eps)) << std::endl;
    NumericalScalar PDF = distribution.computePDF( point );
    fullprint << "pdf     =" << PDF << std::endl;
    fullprint << "pdf (FD)=" << (distribution.computeCDF( point + NumericalPoint(1, eps) ) - distribution.computeCDF( point  + NumericalPoint(1, -eps) )) / (2.0 * eps) << std::endl;
    NumericalScalar CDF = distribution.computeCDF( point );
    fullprint << "cdf=" << CDF << std::endl;
    NumericalPoint PDFgr = distribution.computePDFGradient( point );
    fullprint << "pdf gradient     =" << PDFgr << std::endl;
    NumericalPoint PDFgrFD(3);
    PDFgrFD[0] = (NonCentralStudent(distribution.getNu() + eps, distribution.getDelta(), distribution.getGamma()).computePDF(point) -
                  NonCentralStudent(distribution.getNu() - eps, distribution.getDelta(), distribution.getGamma()).computePDF(point)) / (2.0 * eps);
    PDFgrFD[1] = (NonCentralStudent(distribution.getNu(), distribution.getDelta() + eps, distribution.getGamma()).computePDF(point) -
                  NonCentralStudent(distribution.getNu(), distribution.getDelta() - eps, distribution.getGamma()).computePDF(point)) / (2.0 * eps);
    PDFgrFD[2] = (NonCentralStudent(distribution.getNu(), distribution.getDelta(), distribution.getGamma() + eps).computePDF(point) -
                  NonCentralStudent(distribution.getNu(), distribution.getDelta(), distribution.getGamma() - eps).computePDF(point)) / (2.0 * eps);
    fullprint << "pdf gradient (FD)=" << PDFgrFD << std::endl;
    NumericalPoint CDFgr = distribution.computeCDFGradient( point );
    fullprint << "cdf gradient     =" << CDFgr << std::endl;
    NumericalPoint CDFgrFD(3);
    CDFgrFD[0] = (NonCentralStudent(distribution.getNu() + eps, distribution.getDelta(), distribution.getGamma()).computeCDF(point) -
                  NonCentralStudent(distribution.getNu() - eps, distribution.getDelta(), distribution.getGamma()).computeCDF(point)) / (2.0 * eps);
    CDFgrFD[1] = (NonCentralStudent(distribution.getNu(), distribution.getDelta() + eps, distribution.getGamma()).computeCDF(point) -
                  NonCentralStudent(distribution.getNu(), distribution.getDelta() - eps, distribution.getGamma()).computeCDF(point)) / (2.0 * eps);
    CDFgrFD[2] = (NonCentralStudent(distribution.getNu(), distribution.getDelta(), distribution.getGamma() + eps).computeCDF(point) -
                  NonCentralStudent(distribution.getNu(), distribution.getDelta(), distribution.getGamma() - eps).computeCDF(point)) / (2.0 * eps);
    fullprint << "cdf gradient (FD)=" << CDFgrFD << std::endl;
    NumericalPoint quantile = distribution.computeQuantile( 0.95 );
    fullprint << "quantile=" << quantile << std::endl;
    fullprint << "cdf(quantile)=" << distribution.computeCDF(quantile) << std::endl;
    NumericalPoint mean = distribution.getMean();
    fullprint << "mean=" << mean << std::endl;
    NumericalPoint standardDeviation = distribution.getStandardDeviation();
    fullprint << "standard deviation=" << standardDeviation << std::endl;
    NumericalPoint skewness = distribution.getSkewness();
    fullprint << "skewness=" << skewness << std::endl;
    NumericalPoint kurtosis = distribution.getKurtosis();
    fullprint << "kurtosis=" << kurtosis << std::endl;
    CovarianceMatrix covariance = distribution.getCovariance();
    fullprint << "covariance=" << covariance << std::endl;
    NonCentralStudent::NumericalPointWithDescriptionCollection parameters = distribution.getParametersCollection();
    fullprint << "parameters=" << parameters << std::endl;

  }
  catch (TestFailed & ex) {
    std::cerr << ex << std::endl;
    return ExitCode::Error;
  }


  return ExitCode::Success;
}
