//                                               -*- C++ -*-
/**
 *  @file  t_NormalCopula_std.cxx
 *  @brief The test file of class NormalCopula for standard methods
 *
 *  (C) Copyright 2005-2011 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: souchaud $
 *  @date:   $LastChangedDate: 2011-07-01 10:34:36 +0200 (Fri, 01 Jul 2011) $
 *  Id:      $Id: t_NormalCopula_std.cxx 1981 2011-07-01 08:34:36Z souchaud $
 */
#include <iostream>
#include <sstream>
#include <cmath>
#include "OT.hxx"
#include "OTtestcode.hxx"
#include "OStream.hxx"
#include "NormalCopula.hxx"
#include "Copula.hxx"
#include "NumericalPoint.hxx"
#include "NumericalSample.hxx"
#include "Distribution.hxx"
#include "CorrelationMatrix.hxx"
#include "IdentityMatrix.hxx"

using namespace OT;
using namespace OT::Test;
using namespace OT::Base::Common;
using namespace OT::Base::Type;
using namespace OT::Base::Stat;
using namespace OT::Uncertainty::Model;
using namespace OT::Uncertainty::Distribution;

int main(int argc, char *argv[])
{
  TESTPREAMBLE;
  OStream fullprint(std::cout);
  setRandomGenerator();

  try {
    // Instanciate one distribution object
    UnsignedLong dim(3);

    CorrelationMatrix R = IdentityMatrix(dim);
    for(UnsignedLong i = 0; i < dim - 1; i++)
      {
        R(i, i + 1) = 0.25;
      }
    NormalCopula copula(R);
    copula.setName("a normal copula");
    fullprint << "Copula " << copula << std::endl;
    std::cout << "Copula " << copula << std::endl;
    fullprint << "Mean " << copula.getMean() << std::endl;
    fullprint << "Covariance " << copula.getCovariance() << std::endl;
    // Is this copula an elliptical distribution?
    fullprint << "Elliptical distribution= " << (copula.isElliptical() ? "true" : "false") << std::endl;

    // Is this copula elliptical ?
    fullprint << "Elliptical copula= " << (copula.hasEllipticalCopula() ? "true" : "false") << std::endl;

    // Is this copula independent ?
    fullprint << "Independent copula= " << (copula.hasIndependentCopula() ? "true" : "false") << std::endl;

    // Test for realization of copula
    NumericalPoint oneRealization = copula.getRealization();
    fullprint << "oneRealization=" << oneRealization << std::endl;

    // Test for sampling
    UnsignedLong size = 10;
    NumericalSample oneSample = copula.getNumericalSample( size );
    fullprint << "oneSample=" << oneSample << std::endl;

    // Test for sampling
    size = 10000;
    NumericalSample anotherSample = copula.getNumericalSample( size );
    fullprint << "anotherSample mean=" << anotherSample.computeMean() << std::endl;
    fullprint << "anotherSample covariance=" << anotherSample.computeCovariance() << std::endl;

    // Define a point
    NumericalPoint point(dim, 0.2);

    // Show PDF and CDF of zero point
    NumericalScalar zeroPDF = copula.computePDF( point );
    NumericalScalar zeroCDF = copula.computeCDF( point );
    fullprint << "point= " << point
              << " pdf=" << zeroPDF
              << " cdf=" << zeroCDF
              << std::endl;
    // Get 50% quantile
    NumericalPoint quantile = copula.computeQuantile( 0.5 );
    fullprint << "Quantile=" << quantile << std::endl;
    fullprint << "CDF(quantile)=" << copula.computeCDF(quantile) << std::endl;
    // Extract the marginals
    for (UnsignedLong i = 0; i < dim; i++)
      {
        Copula margin(copula.getMarginal(i));
        fullprint << "margin=" << margin << std::endl;
        fullprint << "margin PDF=" << margin.computePDF(NumericalPoint(1,0.25)) << std::endl;
        fullprint << "margin CDF=" << margin.computeCDF(NumericalPoint(1,0.25)) << std::endl;
        fullprint << "margin quantile=" << margin.computeQuantile(0.95) << std::endl;
        fullprint << "margin realization=" << margin.getRealization() << std::endl;
      }

    // Extract a 2-D marginal
    NormalCopula::Indices indices(2, 0);
    indices[0] = 1;
    indices[1] = 0;
    fullprint << "indices=" << indices << std::endl;
    Copula margins(copula.getMarginal(indices));
    fullprint << "margins=" << margins << std::endl;
    fullprint << "margins PDF=" << margins.computePDF(NumericalPoint(2, 0.25)) << std::endl;
    fullprint << "margins CDF="<< margins.computeCDF(NumericalPoint(2, 0.25)) << std::endl;
    quantile = margins.computeQuantile(0.95);
    fullprint << "margins quantile=" << quantile << std::endl;
    fullprint << "margins CDF(quantile)=" << margins.computeCDF(quantile) << std::endl;
    fullprint << "margins realization=" << margins.getRealization() << std::endl;
    // Creation of the correlation matrix from a Spearman correlation matrix
    CorrelationMatrix spearman(dim);
    for (UnsignedLong i = 1; i < dim; i++)
      {
        spearman(i, i - 1) = 0.25;
      }
    CorrelationMatrix correlation(NormalCopula::GetCorrelationFromSpearmanCorrelation(spearman));
    fullprint << "Normal copula correlation=" << correlation << " from the Spearman correlation=" << spearman << std::endl;
  }
  catch (TestFailed & ex) {
    std::cerr << ex << std::endl;
    return ExitCode::Error;
  }


  return ExitCode::Success;
}
