//                                               -*- C++ -*-
/**
 *  @file  ProductNumericalMathGradientImplementation.cxx
 *  @brief Abstract top-level class for all distributions
 *
 *  (C) Copyright 2005-2012 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2007-05-29 16:13:02 +0200 (mar, 29 mai 2007) $
 *  Id:      $Id: ProductNumericalMathGradientImplementation.cxx 450 2007-05-29 14:13:02Z dutka $
 */
#include "ProductNumericalMathGradientImplementation.hxx"
#include "PersistentObjectFactory.hxx"

BEGIN_NAMESPACE_OPENTURNS



CLASSNAMEINIT(ProductNumericalMathGradientImplementation);

static Factory<ProductNumericalMathGradientImplementation> RegisteredFactory("ProductNumericalMathGradientImplementation");

/* Default constructor */
ProductNumericalMathGradientImplementation::ProductNumericalMathGradientImplementation(const EvaluationImplementation & p_leftEvaluation,
                                                                                       const GradientImplementation & p_leftGradient,
                                                                                       const EvaluationImplementation & p_rightEvaluation,
                                                                                       const GradientImplementation & p_rightGradient)
  : NumericalMathGradientImplementation(),
    p_leftEvaluation_(p_leftEvaluation),
    p_leftGradient_(p_leftGradient),
    p_rightEvaluation_(p_rightEvaluation),
    p_rightGradient_(p_rightGradient)
{
  // Check the compatibility of the evaluations
  if ((p_leftEvaluation_->getOutputDimension() != 1) || (p_rightEvaluation_->getOutputDimension() != 1)) throw InvalidArgumentException(HERE) << "Error: the two functions must have an output dimension equals to 1.";
  if (p_leftEvaluation_->getInputDimension() != p_rightEvaluation_->getInputDimension()) throw InvalidArgumentException(HERE) << "Error: the two functions must have the same input dimension.";

  // Check the compatibility of the gradients
  if ((p_leftGradient_->getInputDimension()  != p_rightGradient_->getInputDimension()) ||
      (p_leftGradient_->getOutputDimension() != p_rightGradient_->getOutputDimension()) ||
      (p_leftGradient_->getInputDimension()  != p_leftEvaluation_->getInputDimension()) ||
      (p_leftGradient_->getOutputDimension()  != 1)) throw InvalidArgumentException(HERE) << "Error: the gradients have incompatible dimensions.";
}

/* Virtual constructor */
ProductNumericalMathGradientImplementation * ProductNumericalMathGradientImplementation::clone() const
{
  return new ProductNumericalMathGradientImplementation(*this);
}

/* Comparison operator */
Bool ProductNumericalMathGradientImplementation::operator ==(const ProductNumericalMathGradientImplementation & other) const
{
  return true;
}

/* String converter */
String ProductNumericalMathGradientImplementation::__repr__() const {
  OSS oss;
  oss << "class=" << ProductNumericalMathGradientImplementation::GetClassName()
      << " name=" << getName()
      << " leftEvaluation=" << p_leftEvaluation_->__repr__()
      << " leftGradient=" << p_leftGradient_->__repr__()
      << " rightEvaluation=" << p_rightEvaluation_->__repr__()
      << " rightGradient=" << p_rightGradient_->__repr__();
  return oss;
}

/* Method gradient() returns the Jacobian transposed matrix of the function at point
 * h = f . g with f, g, h from R^n to R
 * grad(h) = f . grad(g) + g . grad(f)
 */
Matrix ProductNumericalMathGradientImplementation::gradient(const NumericalPoint & inP) const
{
  const UnsignedLong inputDimension(getInputDimension());
  if (inP.getDimension() != inputDimension) throw InvalidArgumentException(HERE) << "Error: the given point has an invalid dimension. Expect a dimension " << inputDimension << ", got " << inP.getDimension();
  ++callsNumber_;
  return p_leftEvaluation_->operator()(inP)[0] * p_rightGradient_->gradient(inP) + p_rightEvaluation_->operator()(inP)[0] * p_leftGradient_->gradient(inP);
}

/* Accessor for input point dimension */
UnsignedLong ProductNumericalMathGradientImplementation::getInputDimension() const
{
  return p_leftEvaluation_->getInputDimension();
}

/* Accessor for output point dimension */
UnsignedLong ProductNumericalMathGradientImplementation::getOutputDimension() const
{
  return 1;
}

/* Method save() stores the object through the StorageManager */
void ProductNumericalMathGradientImplementation::save(Advocate & adv) const
{
  NumericalMathGradientImplementation::save(adv);
  adv.saveAttribute( "leftEvaluation_", *p_leftEvaluation_ );
  adv.saveAttribute( "leftGradient_", *p_leftGradient_ );
  adv.saveAttribute( "rightEvaluation_", *p_rightEvaluation_ );
  adv.saveAttribute( "rightGradient_", *p_rightGradient_ );
}

/* Method load() reloads the object from the StorageManager */
void ProductNumericalMathGradientImplementation::load(Advocate & adv)
{
  TypedInterfaceObject<NumericalMathEvaluationImplementation> evaluationValue;
  TypedInterfaceObject<NumericalMathGradientImplementation> gradientValue;
  NumericalMathGradientImplementation::load(adv);
  adv.loadAttribute( "leftEvaluation_", evaluationValue );
  p_leftEvaluation_ = evaluationValue.getImplementation();
  adv.loadAttribute( "leftGradient_", gradientValue );
  p_leftGradient_ = gradientValue.getImplementation();
  adv.loadAttribute( "rightEvaluation_", evaluationValue );
  p_rightEvaluation_ = evaluationValue.getImplementation();
  adv.loadAttribute( "rightGradient_", gradientValue );
  p_rightGradient_ = gradientValue.getImplementation();
}

END_NAMESPACE_OPENTURNS
