//                                               -*- C++ -*-
/**
 *  @file  SimulationResult.cxx
 *  @brief SimulationResult stores the simulation result
 *
 *  (C) Copyright 2005-2012 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: schueller $
 *  @date:   $LastChangedDate: 2008-05-23 13:46:12 +0200 (ven, 23 mai 2008) $
 *  Id:      $Id: SimulationResult.cxx 821 2008-05-23 11:46:12Z schueller $
 */
#include <cmath>
#include "SimulationResult.hxx"
#include "DistFunc.hxx"

BEGIN_NAMESPACE_OPENTURNS




CLASSNAMEINIT(SimulationResult);

/* Default constructor */
SimulationResult::SimulationResult()
  : TypedInterfaceObject<SimulationResultImplementation>(SimulationResultImplementation().clone())
{
  // Nothing to do
}

/* Standard constructor */
SimulationResult::SimulationResult(const Event & event,
                                   const NumericalScalar probabilityEstimate,
                                   const NumericalScalar varianceEstimate,
                                   const UnsignedLong outerSampling,
                                   const UnsignedLong blockSize)
  : TypedInterfaceObject<SimulationResultImplementation>(new SimulationResultImplementation(event, probabilityEstimate, varianceEstimate, outerSampling, blockSize))
{
  // Nothing to do
}

/* Constructor with parameters */
SimulationResult::SimulationResult(Implementation & p_implementation)
  : TypedInterfaceObject<SimulationResultImplementation>(p_implementation)
{
  // Nothing to do
}

/* Constructor with parameters */
SimulationResult::SimulationResult(const SimulationResultImplementation & implementation)
  : TypedInterfaceObject<SimulationResultImplementation>(implementation.clone())
{
  // Nothing to do
}

/* Event accessor */
Event SimulationResult::getEvent() const
{
  return getImplementation()->getEvent();
}

/* Probability estimate accessor */
NumericalScalar SimulationResult::getProbabilityEstimate() const
{
  return getImplementation()->getProbabilityEstimate();
}

void SimulationResult::setProbabilityEstimate(const NumericalScalar probabilityEstimate)
{
  copyOnWrite();
  getImplementation()->setProbabilityEstimate(probabilityEstimate);
}

/* Variance estimate accessor */
NumericalScalar SimulationResult::getVarianceEstimate() const
{
  return getImplementation()->getVarianceEstimate();
}

void SimulationResult::setVarianceEstimate(const NumericalScalar varianceEstimate)
{
  copyOnWrite();
  getImplementation()->setVarianceEstimate(varianceEstimate);
}

/* Coefficient of variation estimate accessor */
NumericalScalar SimulationResult::getCoefficientOfVariation() const
{
  return getImplementation()->getCoefficientOfVariation();
}

/* Standard deviation estimate accessor */
NumericalScalar SimulationResult::getStandardDeviation() const
{
  return getImplementation()->getStandardDeviation();
}

/* Mean point conditioned to the event realization accessor */
NumericalPoint SimulationResult::getMeanPointInEventDomain() const
{
  return getImplementation()->getMeanPointInEventDomain();
}

/* Outer sampling accessor */
UnsignedLong SimulationResult::getOuterSampling() const
{
  return getImplementation()->getOuterSampling();
}

void SimulationResult::setOuterSampling(const UnsignedLong outerSampling)
{
  copyOnWrite();
  getImplementation()->setOuterSampling(outerSampling);
}

/* String converter */
String SimulationResult::__repr__() const
{
  return getImplementation()->__repr__();
}

/* Confidence length */
NumericalScalar SimulationResult::getConfidenceLength(const NumericalScalar level) const
{
  return getImplementation()->getConfidenceLength();
}

/* Block size accessor */
UnsignedLong SimulationResult::getBlockSize() const
{
  return getImplementation()->getBlockSize();
}

void SimulationResult::setBlockSize(const UnsignedLong blockSize)
{
  copyOnWrite();
  getImplementation()->setBlockSize(blockSize);
}

/* Importance factors accessor */
NumericalPointWithDescription SimulationResult::getImportanceFactors() const
{
  return getImplementation()->getImportanceFactors();
}

/* Draw the importance factors */
Graph SimulationResult::drawImportanceFactors() const
{
  return getImplementation()->drawImportanceFactors();
}

END_NAMESPACE_OPENTURNS
