//                                               -*- C++ -*-
/**
 *  @file  GumbelCopulaFactory.cxx
 *  @brief Factory for GumbelCopula distribution
 *
 *  (C) Copyright 2005-2012 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2007-05-10 16:43:31 +0200 (jeu, 10 mai 2007) $
 *  Id:      $Id: GumbelCopulaFactory.cxx 434 2007-05-10 14:43:31Z dutka $
 */
#include <cmath>
#include "GumbelCopulaFactory.hxx"

BEGIN_NAMESPACE_OPENTURNS



/* Default constructor */
GumbelCopulaFactory::GumbelCopulaFactory():
  DistributionImplementationFactory()
{
  // Nothing to do
}

/* Virtual constructor */
GumbelCopulaFactory * GumbelCopulaFactory::clone() const
{
  return new GumbelCopulaFactory(*this);
}

/* Here is the interface that all derived class must implement */

GumbelCopula * GumbelCopulaFactory::build(const NumericalSample & sample) const
{
  if (sample.getSize() == 0) throw InvalidArgumentException(HERE) << "Error: cannot build a GumbelCopula distribution from an empty sample";
  if (sample.getDimension() != 2) throw InvalidArgumentException(HERE) << "Error: cannot build a GumbelCopula distribution from a sample of dimension not equal to 2";
  NumericalScalar tau(sample.computeKendallTau().operator()(0, 1));
  if (tau == 1) throw InvalidArgumentException(HERE) << "Error: cannot build a GumbelCopula distribution from a sample with Kendall tau equals to 1";
  return GumbelCopula(1.0 / (1.0 - tau)).clone();
}

GumbelCopula * GumbelCopulaFactory::build(const NumericalPointCollection & parameters) const
{
  try {
    GumbelCopula copula;
    copula.setParametersCollection(parameters);
    return copula.clone();
  }
  catch (InvalidArgumentException & ex)
    {
      throw InternalException(HERE) << "Error: cannot build a GumbelCopula from the given parameters";
    }
}

GumbelCopula * GumbelCopulaFactory::build() const
{
  return GumbelCopula().clone();
}


END_NAMESPACE_OPENTURNS
