//                                               -*- C++ -*-
/**
 *  @file  t_FORM_sensitivity2.cxx
 *  @brief The test file of FORM class
 *
 *  (C) Copyright 2005-2012 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: schueller $
 *  @date:   $LastChangedDate: 2012-02-17 19:35:43 +0100 (Fri, 17 Feb 2012) $
 *  Id:      $Id: t_FORM_sensitivity2.cxx 2392 2012-02-17 18:35:43Z schueller $
 */
#include <iostream>
#include <iomanip>
#include <cmath>
#include "OT.hxx"
#include "OTtestcode.hxx"
#include "OStream.hxx"
#include "Event.hxx"
#include "RandomVector.hxx"
#include "Less.hxx"
#include "NumericalMathFunction.hxx"
#include "Simulation.hxx"
#include "FORM.hxx"
#include "IdentityMatrix.hxx"
#include "Normal.hxx"
#include "Cobyla.hxx"
#include "Pointer.hxx"

using namespace OT;
using namespace OT::Test;

String printNumericalPoint(const NumericalPoint & point, const UnsignedLong digits)
{
  OSS oss;
  oss << "[";
  NumericalScalar eps(pow(0.1, digits));
  for (UnsignedLong i = 0; i < point.getDimension(); i++)
    {
      oss << std::fixed << std::setprecision(digits) << (i==0 ? "" : ",") << ((fabs(point[i]) < eps) ? fabs(point[i]) : point[i]);
    }
  oss << "]";
  return oss;
}

int main(int argc, char *argv[])
{
  TESTPREAMBLE;
  OStream fullprint(std::cout);

  try {

    /* We create a numerical math function */
    NumericalMathFunction myFunction("poutre");

    UnsignedLong dim(myFunction.getInputDimension());
    /* We create a normal distribution point of dimension 1 */
    NumericalPoint mean(dim, 0.0);
    mean[0] = 50.0; // E
    mean[1] =  1.0; // F
    mean[2] = 10.0; // L
    mean[3] =  5.0; // I
    NumericalPoint sigma(dim, 1.0);
    IdentityMatrix R(dim);
    Normal myDistribution(mean, sigma, R);

    /* we name the components of the distribution */
    Description componentDescription(dim);
    componentDescription[0] = "E";
    componentDescription[1] = "F";
    componentDescription[2] = "L";
    componentDescription[3] = "I";
    myDistribution.setDescription(componentDescription);

    /* We create a 'usual' RandomVector from the Distribution */
    RandomVector vect(myDistribution);

    /* We create a composite random vector */
    RandomVector output(myFunction, vect);
    Description outputDescription(1);
    outputDescription[0] = "Vertical Displacement";
    output.setDescription(outputDescription);

    /* We create an Event from this RandomVector */
    Event myEvent(output, Less(), -3.0);

    /* We create a NearestPoint algorithm */
    Cobyla myCobyla;
    myCobyla.setSpecificParameters(CobylaSpecificParameters());
    myCobyla.setMaximumIterationsNumber(100);
    myCobyla.setMaximumAbsoluteError(1.0e-10);
    myCobyla.setMaximumRelativeError(1.0e-10);
    myCobyla.setMaximumResidualError(1.0e-10);
    myCobyla.setMaximumConstraintError(1.0e-10);
    fullprint << "myCobyla=" << myCobyla << std::endl << std::flush;

    /* We create a FORM algorithm */
    /* The first parameter is a NearestPointAlgorithm */
    /* The second parameter is an event */
    /* The third parameter is a starting point for the design point research */
    FORM myAlgo(myCobyla, myEvent, mean);

    fullprint << "FORM=" << myAlgo << std::endl;

    /* Perform the simulation */
    myAlgo.run();

    /* Stream out the result */
    FORMResult result(myAlgo.getResult());
    UnsignedLong digits(5);
    fullprint << "importance factors=" << printNumericalPoint(result.getImportanceFactors(), digits) << std::endl;
    fullprint << "importance factors (classical)=" << printNumericalPoint(result.getImportanceFactors(true), digits) << std::endl;
    fullprint << "Hasofer reliability index=" << std::setprecision(digits) << result.getHasoferReliabilityIndex() << std::endl;
    fullprint << "result=" << result << std::endl;

    /* Hasofer Reliability Index Sensitivity */
    Analytical::Sensitivity hasoferReliabilityIndexSensitivity(result.getHasoferReliabilityIndexSensitivity());
    fullprint << "hasoferReliabilityIndexSensitivity = " << hasoferReliabilityIndexSensitivity << std::endl;


    /* Event Probability Sensitivity */
    Analytical::Sensitivity eventProbabilitySensitivity(result.getEventProbabilitySensitivity());
    fullprint << "eventProbabilitySensitivity = " << eventProbabilitySensitivity << std::endl;

  }
  catch (TestFailed & ex) {
    std::cerr << ex << std::endl;
    return ExitCode::Error;
  }


  return ExitCode::Success;
}
