/***************************************************************************
 *   Copyright (C) 2009 by Paul Lutus                                      *
 *   lutusp@arachnoid.com                                                  *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/
/*
 * To change this template, choose Tools | Templates
 * and open the template in the editor.
 */

/*
 * OpticalRayTracer.java
 *
 * Created on Mar 16, 2009, 8:15:45 AM
 */
package opticalraytracer;

import javax.swing.*;
import java.net.*;
import java.util.*;
import java.awt.*;
import java.awt.image.*;
import java.awt.datatransfer.Clipboard;
import java.awt.datatransfer.StringSelection;
import java.awt.datatransfer.DataFlavor;
import java.awt.datatransfer.ClipboardOwner;
import java.awt.datatransfer.Transferable;
import java.text.*;

/**
 *
 * @author lutusp
 */
final public class OpticalRayTracer extends javax.swing.JFrame implements ClipboardOwner {

    final String appVersion = "2.8";
    final String appName;
    final String programName;
    boolean applet = false;
    //String appPath;
    //String fileSep;
    //String lineSep;
    //String extPath;
    //String userDir;
    //String userPath;
    //String dataPath;
    JFrame sv_programFrame;
    MyHelpPane helpPane;
    NumberFormat numberFormat;
    int sv_helpScrollPos = 0;
    InitManager initManager = null;
    //GraphicDisplay graphicDisplay;
    RayTraceComputer rayTraceComputer;
    ControlPanelManager controlPanelManager;
    Stack<UndoRedoPackage> undoStack;
    Stack<UndoRedoPackage> redoStack;
    Vector<Lens> sv_lensList;
    int sv_selectedLensIndex = -1;
    Lens selectedLens = null;
    Lens mouseTarget = null;
    GraphicDisplay gd1, gd2;
    ColorButton sv_dispHiColor;
    ColorButton sv_dispLoColor;
    ColorButton sv_beamColor;
    ColorButton sv_lensSelColor;
    ColorButton sv_intersectionColor;
    ColorButton sv_lensOutlineColor;
    ColorButton sv_yBaselineColor;
    ColorButton sv_barrierColor;
    ColorButton sv_gridColor;
    double sv_dispScale;
    double sv_xOffset;
    double sv_yOffset;
    double xCenter;
    double yCenter;
    double mousePressX, mousePressY;
    //int currentTab;
    int intersectionDotRadius;
    double yStartBeamPos;
    double yEndBeamPos;
    int dispersionBeams;
    int beamCount;
    int maxIntersections;
    double xBeamSourceRefPlane;
    double xBeamTargetRefPlane;
    double ySnap;
    double beamAngle;
    int beamWidth;
    BufferedImage image = null;
    int sv_clipboardGraphicXSize;
    int sv_clipboardGraphicYSize;
    double textFieldDoubleSensitivity;
    double textFieldIntSensitivity;
    double curvatureFactorSensitivity;
    boolean divergingSource;
    boolean antiAlias;
    boolean inverted;
    boolean drawGrid;

    // END default initializations
    double sv_lensMinThickness; // % of lens radius
    double userThickness;
    int overlappedLensSelector;
    int xSize;
    int ySize;
    int sv_decimalPlaces = 3;
    MutableDouble dx = new MutableDouble();
    MutableDouble dy = new MutableDouble();
    int popupMouseX, popupMouseY;

    /** Creates new form OpticalRayTracer */
    public OpticalRayTracer() {
        initComponents();
        // just for testing locale issues
        //Locale.setDefault(Locale.GERMANY);
        numberFormat = NumberFormat.getNumberInstance();
        sv_lensList = new Vector<Lens>();
        resetUndoRedo();
        sv_programFrame = this; // used to set screen geometry
        // Just for the first run
        int w = 800;
        int h = 650;
        Dimension ss = Toolkit.getDefaultToolkit().getScreenSize();
        setBounds(new Rectangle((ss.width - w) / 2, (ss.height - h) / 2, w, h));
        appName = getClass().getSimpleName();
        URL url = getClass().getResource(appName + ".class");
        //appPath = url.getPath().replaceFirst("(.*?)!.*", "$1");
        //appPath = appPath.replaceFirst("file:", "");
        //appPath = new File(appPath).getPath();
        programName = appName + " " + appVersion;
        setTitle(programName);
        setIconImage(new ImageIcon(getClass().getResource("icons/" + appName + ".png")).getImage());
        //userDir = System.getProperty("user.home");
        //lineSep = System.getProperty("line.separator");
        //fileSep = System.getProperty("file.separator");
        //userPath = userDir + fileSep + "." + appName;
        // FIXME no need for this reference
        rayTraceComputer = new RayTraceComputer(this);
        controlPanelManager = new ControlPanelManager(this, rayTraceComputer);
        gd1 = new GraphicDisplay(this, rayTraceComputer);
        graphicPlaceholder1.add(gd1);
        gd2 = new GraphicDisplay(this, rayTraceComputer);
        graphicPlaceholder2.add(gd2);

        setDefaults(false);

        setControlActions();

        //setupColorButtons();

        initManager = new InitManager(this);
        initManager.readConfig();
        if (sv_selectedLensIndex != -1 && sv_selectedLensIndex < sv_lensList.size()) {
            setSelectedLens(sv_lensList.get(sv_selectedLensIndex));
        }

        helpPane = new MyHelpPane(this);
        helpTab.add(helpPane);

        readControls();
        SwingUtilities.invokeLater(new Runnable() {

            public void run() {
                updateGraphicDisplay();
            }
        });
        makeDefaultLenses(false);
    }

    // debugging printers
    void p(String s) {
        System.out.println(s);
    }

    String pn(double n) {
        return String.format("%.4f", n);
    }

    void setStatus(String s) {
        statusLabel.setText(s);
    }

    void setSelectedLens(Lens p) {
        controlPanelManager.readLensValues(p);
        selectedLens = p;
    }

    void resetUndoRedo() {
        undoStack = new Stack<UndoRedoPackage>();
        redoStack = new Stack<UndoRedoPackage>();
        updateUndoRedoButtons();
    }

    void pushUndo() {
        //p("pushundo: " + selectedLens + "," + mouseTarget);
        undoStack.push(new UndoRedoPackage(sv_lensList, selectedLens));
        updateUndoRedoButtons();
    }

    void pushRedo() {
        // p("pushredo: " + selectedLens + "," + mouseTarget);
        redoStack.push(new UndoRedoPackage(sv_lensList, selectedLens));
        updateUndoRedoButtons();
    }

    void undo() {
        if (undoStack.size() > 0) {
            pushRedo();
            undoStack.pop().restore(this);
            //p("undo: " + selectedLens + "," + mouseTarget);
            updateGraphicDisplay();
            updateUndoRedoButtons();
        } else {
            beep();
        }
    }

    void redo() {
        if (redoStack.size() > 0) {
            pushUndo();
            redoStack.pop().restore(this);
            //p("redo: " + selectedLens + "," + mouseTarget);
            updateGraphicDisplay();
            updateUndoRedoButtons();
        } else {
            beep();
        }
    }

    void updateUndoRedoButtons() {
        redoButton.setEnabled(redoStack.size() > 0);
        undoButton.setEnabled(undoStack.size() > 0);
    }

    static void beep() {
        Toolkit.getDefaultToolkit().beep();
    }

    String formatNum(double v) {
        return String.format("%." + sv_decimalPlaces + "f", v);
    }

    void setControlActions() {
        new UserActionManager(textFieldDoubleSensitivity, -1e10, 1e10, lensRadiusTextField, this);
        new UserActionManager(textFieldDoubleSensitivity, -1e10, 1e10, userThicknessTextField, this);
        new UserActionManager(textFieldDoubleSensitivity, -1e10, 1e10, leftRadiusTextField, this);
        new UserActionManager(textFieldDoubleSensitivity, -1e10, 1e10, rightRadiusTextField, this);
        new UserActionManager(curvatureFactorSensitivity, 0, 0.25, leftCFTextField, this);
        new UserActionManager(curvatureFactorSensitivity, 0, 0.25, rightCFTextField, this);
        new UserActionManager(textFieldDoubleSensitivity, -1e10, 1e10, iorTextField, this);
        new UserActionManager(textFieldDoubleSensitivity, -1e10, 1e10, dispersionTextField, this);
        new UserActionManager(textFieldDoubleSensitivity, -1e10, 1e10, xPosTextField, this);
        new UserActionManager(textFieldDoubleSensitivity, -1e10, 1e10, yPosTextField, this);

        new UserActionManager(textFieldIntSensitivity, -1e10, 1e10, sv_dotRadiusTextField, this);
        new UserActionManager(textFieldDoubleSensitivity, -1e10, 1e10, sv_snapToBaseTextField, this);
        new UserActionManager(textFieldDoubleSensitivity, -1e10, 1e10, sv_beamWidthTextField, this);
        new UserActionManager(textFieldIntSensitivity, -1e10, 1e10, sv_beamCountTextField, this);
        new UserActionManager(textFieldIntSensitivity, -1e10, 1e10, sv_intersectionsTextField, this);
        new UserActionManager(textFieldDoubleSensitivity, -1e10, 1e10, sv_yStartTextField, this);
        new UserActionManager(textFieldDoubleSensitivity, -1e10, 1e10, sv_yEndTextField, this);
        new UserActionManager(textFieldDoubleSensitivity, -1e10, 1e10, sv_xSourcePlaneTextField, this);
        new UserActionManager(textFieldDoubleSensitivity, -1e10, 1e10, sv_xTargetPlaneTextField, this);
        new UserActionManager(textFieldDoubleSensitivity, -1e10, 1e10, sv_offsetAngleTextField, this);
        new UserActionManager(textFieldIntSensitivity, -1e10, 1e10, sv_dispersionBeamsTextField, this);
    }

    public void updateGraphicDisplay() {
        gd1.updateDisplay();
        gd2.updateDisplay();
    }

    public void unSelectLens() {
        pushUndo();
        selectedLens = null;
        updateGraphicDisplay();
    }

    Lens makeGenericLens(double cx, double cy, double radius, double thickness) {
        double ior = 1.52;
        double disp = 59;
        double cf = 0.03;
        boolean leftHyp = false;
        boolean rightHyp = false;
        boolean symmetrical = true;
        return new Lens(this, this.rayTraceComputer, cx, cy, 2, radius, radius, thickness, ior, cf, cf, disp, leftHyp, rightHyp, symmetrical);
    }

    // create two generic default lenses
    void makeDefaultLenses(boolean update) {
        if (sv_lensList.size() == 0) {
            sv_lensList.add(makeGenericLens(0, 0, 6, 0));
            sv_lensList.add(makeGenericLens(4, 0, -6, 0.1));
            selectedLens = null;
            if (update) {
                updateGraphicDisplay();
            }
        }
    }

    void makeNewLens() {
        pushUndo();
        double cx = 0;
        Iterator<Lens> it = sv_lensList.iterator();
        while (it.hasNext()) {
            double tx = it.next().cx;
            cx = Math.max(cx, tx);
        }
        cx += 1.0;
        Lens newLens = makeGenericLens(cx, 0, 6, 0);
        sv_lensList.add(newLens);
        setSelectedLens(newLens);
        gd1.snapLens(selectedLens);
        updateGraphicDisplay();
    }

    void makeNewLensPopup() {
        pushUndo();
        rayTraceComputer.displayToSpace(popupMouseX, popupMouseY, dx, dy);
        double cx = dx.v + sv_xOffset;
        double cy = dy.v + sv_yOffset;
        Lens lens = makeGenericLens(cx, cy, 6, 0);
        sv_lensList.add(lens);
        setSelectedLens(lens);
        gd1.snapLens(lens);
        updateGraphicDisplay();
    }

    void deleteSelectedLens() {
        if (selectedLens != null) {
            if (showMessage("Okay to delete selected lens?", "Delete Lens")) {
                sv_lensList.remove(selectedLens);
                selectedLens = null;
                updateGraphicDisplay();
            }
        }
    }

    void setupColorButtons() {
        sv_yBaselineColor = new ColorButton(this, baselineColorButton, 0x004000, "Y Baseline");
        sv_gridColor = new ColorButton(this, gridColorButton, 0xc0c0c0, "Grid");
        sv_lensOutlineColor = new ColorButton(this, lensOutlineColorButton, 0x80a0ff, "Lens outline");
        sv_intersectionColor = new ColorButton(this, intersectionColorButton, 0xaa00aa, "Beam intersection");
        sv_dispHiColor = new ColorButton(this, dispHiColorButton, 0xffffff, "High background");
        sv_dispLoColor = new ColorButton(this, dispLoColorButton, 0x000000, "Low background");
        sv_lensSelColor = new ColorButton(this, lensSelColorButton, 0x00c000, "Lens-selected");
        sv_beamColor = new ColorButton(this, beamColorButton, 0xff0000, "Tracing beams");
        sv_barrierColor = new ColorButton(this, barrierColorButton, 0x0000ff, "Domain barrier");
    }

    void setDefaults(boolean newLenses) {
        sv_dispScale = 0.2;
        sv_xOffset = 2;
        sv_yOffset = 0;
        xCenter = 0;
        yCenter = 0;
        intersectionDotRadius = 4;
        yStartBeamPos = -1.8;
        yEndBeamPos = 1.8;
        dispersionBeams = 0;
        beamCount = 6;
        maxIntersections = 64;
        xBeamSourceRefPlane = -30.0;
        xBeamTargetRefPlane = 30.0;
        ySnap = 1.0;
        beamAngle = 0;
        beamWidth = 1;
        sv_clipboardGraphicXSize = 1280;
        sv_clipboardGraphicYSize = 1024;
        textFieldDoubleSensitivity = 0.1;
        textFieldIntSensitivity = 1;
        curvatureFactorSensitivity = 0.001;
        divergingSource = false;
        antiAlias = true;
        inverted = false;
        drawGrid = true;
        sv_lensMinThickness = .0001; // % of lens radius
        userThickness = 0;
        overlappedLensSelector = 0;
        xSize = -1;
        ySize = -1;
        sv_decimalPlaces = 4;
        sv_invertedCheckBox.setSelected(false);
        sv_gridCheckBox.setSelected(true);
        sv_antiAliasCheckBox.setSelected(true);
        resetUndoRedo();
        setupColorButtons();
        if (newLenses) {
            sv_lensList = new Vector<Lens>();
            makeDefaultLenses(true);
        }
        writeControls();
    }

    double getDouble(String s) {
        double v = 0;
        double e;
        try {
            // deal with possibility of exponent
            // which numberformat cannot process
            s = s.toLowerCase();
            s = s.replaceFirst("\\s*(.*?)\\s*", "$1");
            if (!s.matches("\\(.*")) {
                String[] array = s.split("e");
                v = numberFormat.parse(array[0]).doubleValue();
                if (array.length > 1) {
                    // get signed exponent
                    e = numberFormat.parse(array[1]).doubleValue();
                    v *= Math.pow(10, e);
                }
            }
        } catch (Exception ex) {
            System.out.println(getClass().getName() + ".getDouble: Error: " + ex + ", source: " + s);
        //ex.printStackTrace();
        }
        return v;
    }

    double readJTextField(JTextField tf, double result) {
        try {
            result = getDouble(tf.getText());
        } catch (Exception e) {
            System.out.println(getClass().getName() + ": readJTextField Error: " + e);
        }
        return result;
    }

    int readJTextField(JTextField tf, int result) {
        try {
            result = (int) getDouble(tf.getText());
        } catch (Exception e) {
            System.out.println(getClass().getName() + ": readJTextField Error: " + e);
        }
        return result;
    }

    void writeJTextField(JTextField tf, int v) {
        tf.setText("" + v);
    }

    void writeJTextField(JTextField tf, double v) {
        tf.setText(formatNum(v));
    }

    void readControls() {
        intersectionDotRadius = readJTextField(sv_dotRadiusTextField, intersectionDotRadius);
        ySnap = readJTextField(sv_snapToBaseTextField, ySnap);
        beamWidth = readJTextField(sv_beamWidthTextField, beamWidth);
        beamCount = readJTextField(sv_beamCountTextField, beamCount);
        maxIntersections = readJTextField(sv_intersectionsTextField, maxIntersections);
        yStartBeamPos = readJTextField(sv_yStartTextField, yStartBeamPos);
        yEndBeamPos = readJTextField(sv_yEndTextField, yEndBeamPos);
        xBeamSourceRefPlane = readJTextField(sv_xSourcePlaneTextField, xBeamSourceRefPlane);
        xBeamTargetRefPlane = readJTextField(sv_xTargetPlaneTextField, xBeamTargetRefPlane);
        beamAngle = readJTextField(sv_offsetAngleTextField, beamAngle);
        dispersionBeams = readJTextField(sv_dispersionBeamsTextField, dispersionBeams);
        divergingSource = sv_divergingSourceCheckBox.isSelected();
        drawGrid = sv_gridCheckBox.isSelected();
        antiAlias = sv_antiAliasCheckBox.isSelected();
        inverted = sv_invertedCheckBox.isSelected();
        updateGraphicDisplay();
    }

    void writeControls() {
        writeJTextField(sv_dotRadiusTextField, intersectionDotRadius);
        writeJTextField(sv_snapToBaseTextField, ySnap);
        writeJTextField(sv_beamWidthTextField, beamWidth);
        writeJTextField(sv_beamCountTextField, beamCount);
        writeJTextField(sv_intersectionsTextField, maxIntersections);
        writeJTextField(sv_yStartTextField, yStartBeamPos);
        writeJTextField(sv_yEndTextField, yEndBeamPos);
        writeJTextField(sv_xSourcePlaneTextField, xBeamSourceRefPlane);
        writeJTextField(sv_xTargetPlaneTextField, xBeamTargetRefPlane);
        writeJTextField(sv_offsetAngleTextField, beamAngle);
        writeJTextField(sv_dispersionBeamsTextField, dispersionBeams);
        sv_divergingSourceCheckBox.setSelected(divergingSource);
        sv_gridCheckBox.setSelected(drawGrid);
        sv_antiAliasCheckBox.setSelected(antiAlias);
        sv_invertedCheckBox.setSelected(inverted);

    }

    void eraseReset() {
        if (showMessage("Okay to reset to defaults (erases all changes)?", "To Defaults")) {
            setDefaults(true);
        }
    }

    void showHideControls() {
        if (controlPanel.isVisible()) {
            controlPanel.setVisible(false);
            cPanelButton.setText("Show Controls");
        } else {
            controlPanel.setVisible(true);
            cPanelButton.setText("Hide Controls");
        }
    }

    boolean showMessage(String message, String title) {
        int reply = JOptionPane.showConfirmDialog(this, message, programName + ": " + title, JOptionPane.YES_NO_CANCEL_OPTION);
        return (reply == JOptionPane.YES_OPTION);
    }

    void clipboardCopyImage() {
        this.gd1.rayTraceProcessCore(sv_clipboardGraphicXSize, this.sv_clipboardGraphicYSize);
        ImageTransferable imt = new ImageTransferable(image);
        Clipboard clipboard = Toolkit.getDefaultToolkit().getSystemClipboard();
        clipboard.setContents(imt, null);
    }

    void clipboardCopyFullConfig() {
        String config = this.initManager.getFullConfig();
        setClipboardContents(config);
    }

    void clipboardPasteFullConfig() {
        String data = getClipboardContents();
        if (data != null) {
            if (showMessage("Okay to read full configuration (erases all current settings)?", "Read Full Configuration")) {
                initManager.setConfig(data);
                readControls();
            }
        }
    }

    void setClipboardContents(String s) {
        StringSelection stringSelection = new StringSelection(s);
        Clipboard clipboard = Toolkit.getDefaultToolkit().getSystemClipboard();
        clipboard.setContents(stringSelection, this);
    }

    String getClipboardContents() {
        String s = null;
        Clipboard clipboard = Toolkit.getDefaultToolkit().getSystemClipboard();
        Transferable content = clipboard.getContents(this);
        if (content != null) {
            try {
                s = (String) content.getTransferData(DataFlavor.stringFlavor);
            } catch (Exception e) {
                //System.out.println("getClipboardContents: " + e);
            }
        }
        return s;
    }

    void clipboardCutLens() {
        Lens lens;
        if ((lens = clipboardCopyLens()) != null) {
            pushUndo();
            sv_lensList.remove(lens);
            selectedLens = null;
            updateGraphicDisplay();
        }
    }

    Lens clipboardCopyLens() {
        Lens lens = null;
        if (selectedLens != null) {
            lens = selectedLens;
            setClipboardContents(lens.toString());
        }
        return lens;
    }

    void clipboardPasteLens(boolean toMouse) {
        String s = getClipboardContents();
        if (s != null) {
            if (s.matches("(?ms).*(,|\\s).*")) {
                decodeMultiLensString(s);
            } else {
                Lens newLens = new Lens(this, rayTraceComputer, s);
                if (newLens.valid) {
                    pushUndo();
                    sv_lensList.add(newLens);
                    if (toMouse) {
                        // position lens at mouse cursor
                        rayTraceComputer.displayToSpace(popupMouseX, popupMouseY, dx, dy);
                        newLens.cx = dx.v + sv_xOffset;
                        newLens.cy = dy.v + sv_yOffset;
                    }
                    updateGraphicDisplay();
                    // select this lens
                    setSelectedLens(newLens);
                    gd1.snapLens(selectedLens);
                }
            }
        }
    }

    void decodeMultiLensString(String s) {
        s = s.replaceFirst("(?ms)^(\\s|\\[|\"|')*(.*?)(\\s|\\]|\"|')*$", "$2");
        s = s.replaceAll("(\\s|,)+", ",");
        String[] array = s.split(",");
        for (int i = 0; i < array.length; i++) {
            sv_lensList.add(new Lens(this, rayTraceComputer, array[i]));
        }
        updateGraphicDisplay();
    }

    public void lostOwnership(Clipboard aClipboard, Transferable aContents) {
        //do nothing
    }

    void processTabClick() {
        Component comp = sv_mainTabbedPane.getSelectedComponent();
        //comp.requestFocus();
        if (comp == helpTab) {
            helpPane.sv_findTextField.requestFocusInWindow();
        }
    }

    public void close() {
        this.sv_helpScrollPos = helpPane.getScrollPos();
        sv_selectedLensIndex = sv_lensList.indexOf(selectedLens);
        initManager.writeConfig();
        setVisible(false);
        System.exit(0);
    }

    /** This method is called from within the constructor to
     * initialize the form.
     * WARNING: Do NOT modify this code. The content of this method is
     * always regenerated by the Form Editor.
     */
    @SuppressWarnings("unchecked")
    // <editor-fold defaultstate="collapsed" desc="Generated Code">//GEN-BEGIN:initComponents
    private void initComponents() {
        java.awt.GridBagConstraints gridBagConstraints;

        sv_mainTabbedPane = new javax.swing.JTabbedPane();
        displayTab = new javax.swing.JPanel();
        graphicPanel = new javax.swing.JPanel();
        graphicPlaceholder1 = new javax.swing.JPanel();
        buttonPanel = new javax.swing.JPanel();
        sv_antiAliasCheckBox = new javax.swing.JCheckBox();
        sv_invertedCheckBox = new javax.swing.JCheckBox();
        sv_gridCheckBox = new javax.swing.JCheckBox();
        unselectButton = new javax.swing.JButton();
        undoButton = new javax.swing.JButton();
        redoButton = new javax.swing.JButton();
        newLensButton = new javax.swing.JButton();
        resetButton = new javax.swing.JButton();
        cPanelButton = new javax.swing.JButton();
        copyImageButton = new javax.swing.JButton();
        copyFullConfigButton = new javax.swing.JButton();
        pasteFullConfigButton = new javax.swing.JButton();
        quitButton = new javax.swing.JButton();
        controlPanel = new javax.swing.JPanel();
        leftLabel = new javax.swing.JLabel();
        rightLabel = new javax.swing.JLabel();
        sphereRadiusLabel = new javax.swing.JLabel();
        cfLabel = new javax.swing.JLabel();
        leftRadiusTextField = new javax.swing.JTextField();
        rightRadiusTextField = new javax.swing.JTextField();
        leftCFTextField = new javax.swing.JTextField();
        rightCFTextField = new javax.swing.JTextField();
        leftHypCheckBox = new javax.swing.JCheckBox();
        rightHypCheckBox = new javax.swing.JCheckBox();
        hypLabel = new javax.swing.JLabel();
        controlSubPanel = new javax.swing.JPanel();
        iorLabel = new javax.swing.JLabel();
        iorTextField = new javax.swing.JTextField();
        dispersionLabel = new javax.swing.JLabel();
        dispersionTextField = new javax.swing.JTextField();
        lensRadiusLabel = new javax.swing.JLabel();
        lensRadiusTextField = new javax.swing.JTextField();
        thicknessLabel1 = new javax.swing.JLabel();
        userThicknessTextField = new javax.swing.JTextField();
        posLabel = new javax.swing.JLabel();
        xPosLabel = new javax.swing.JLabel();
        xPosTextField = new javax.swing.JTextField();
        yPosLabel = new javax.swing.JLabel();
        yPosTextField = new javax.swing.JTextField();
        symmCheckBox = new javax.swing.JCheckBox();
        configurationTab = new javax.swing.JPanel();
        graphicPlaceholder2 = new javax.swing.JPanel();
        colorPanel = new javax.swing.JPanel();
        baselineColorButton = new javax.swing.JButton();
        gridColorButton = new javax.swing.JButton();
        lensOutlineColorButton = new javax.swing.JButton();
        beamColorButton = new javax.swing.JButton();
        intersectionColorButton = new javax.swing.JButton();
        lensSelColorButton = new javax.swing.JButton();
        dispHiColorButton = new javax.swing.JButton();
        dispLoColorButton = new javax.swing.JButton();
        jLabel1 = new javax.swing.JLabel();
        barrierColorButton = new javax.swing.JButton();
        numericPanel = new javax.swing.JPanel();
        jLabel2 = new javax.swing.JLabel();
        sv_dotRadiusTextField = new javax.swing.JTextField();
        jLabel3 = new javax.swing.JLabel();
        sv_snapToBaseTextField = new javax.swing.JTextField();
        jLabel4 = new javax.swing.JLabel();
        sv_beamWidthTextField = new javax.swing.JTextField();
        jLabel5 = new javax.swing.JLabel();
        sv_beamCountTextField = new javax.swing.JTextField();
        jLabel6 = new javax.swing.JLabel();
        sv_intersectionsTextField = new javax.swing.JTextField();
        jLabel7 = new javax.swing.JLabel();
        sv_yStartTextField = new javax.swing.JTextField();
        jLabel8 = new javax.swing.JLabel();
        sv_yEndTextField = new javax.swing.JTextField();
        jLabel9 = new javax.swing.JLabel();
        sv_xSourcePlaneTextField = new javax.swing.JTextField();
        jLabel10 = new javax.swing.JLabel();
        sv_xTargetPlaneTextField = new javax.swing.JTextField();
        jLabel11 = new javax.swing.JLabel();
        sv_offsetAngleTextField = new javax.swing.JTextField();
        jLabel12 = new javax.swing.JLabel();
        sv_dispersionBeamsTextField = new javax.swing.JTextField();
        sv_divergingSourceCheckBox = new javax.swing.JCheckBox();
        helpTab = new javax.swing.JPanel();
        statusLabel = new javax.swing.JLabel();

        setDefaultCloseOperation(javax.swing.WindowConstants.EXIT_ON_CLOSE);
        addWindowListener(new java.awt.event.WindowAdapter() {
            public void windowClosing(java.awt.event.WindowEvent evt) {
                formWindowClosing(evt);
            }
        });
        getContentPane().setLayout(new java.awt.GridBagLayout());

        sv_mainTabbedPane.setTabPlacement(javax.swing.JTabbedPane.BOTTOM);
        sv_mainTabbedPane.addMouseListener(new java.awt.event.MouseAdapter() {
            public void mouseClicked(java.awt.event.MouseEvent evt) {
                sv_mainTabbedPaneMouseClicked(evt);
            }
        });

        displayTab.setLayout(new java.awt.GridBagLayout());

        graphicPanel.setLayout(new java.awt.GridBagLayout());

        graphicPlaceholder1.setBorder(javax.swing.BorderFactory.createLineBorder(new java.awt.Color(0, 0, 0)));
        graphicPlaceholder1.setLayout(new java.awt.BorderLayout());
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.fill = java.awt.GridBagConstraints.BOTH;
        gridBagConstraints.weightx = 1.0;
        gridBagConstraints.weighty = 1.0;
        gridBagConstraints.insets = new java.awt.Insets(2, 2, 2, 2);
        graphicPanel.add(graphicPlaceholder1, gridBagConstraints);

        buttonPanel.setBackground(new java.awt.Color(247, 221, 250));
        buttonPanel.setBorder(javax.swing.BorderFactory.createLineBorder(new java.awt.Color(0, 0, 0)));
        buttonPanel.setLayout(new java.awt.GridBagLayout());

        sv_antiAliasCheckBox.setBackground(new java.awt.Color(247, 221, 250));
        sv_antiAliasCheckBox.setSelected(true);
        sv_antiAliasCheckBox.setText("Antialias");
        sv_antiAliasCheckBox.setToolTipText("Best appearance, slower drawing");
        sv_antiAliasCheckBox.addMouseListener(new java.awt.event.MouseAdapter() {
            public void mouseClicked(java.awt.event.MouseEvent evt) {
                sv_antiAliasCheckBoxMouseClicked(evt);
            }
        });
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 10;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.weightx = 1.0;
        gridBagConstraints.insets = new java.awt.Insets(0, 2, 0, 2);
        buttonPanel.add(sv_antiAliasCheckBox, gridBagConstraints);

        sv_invertedCheckBox.setBackground(new java.awt.Color(247, 221, 250));
        sv_invertedCheckBox.setText("Inverted");
        sv_invertedCheckBox.setToolTipText("Dark background");
        sv_invertedCheckBox.addMouseListener(new java.awt.event.MouseAdapter() {
            public void mouseClicked(java.awt.event.MouseEvent evt) {
                sv_invertedCheckBoxMouseClicked(evt);
            }
        });
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 8;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.weightx = 1.0;
        gridBagConstraints.insets = new java.awt.Insets(0, 2, 0, 2);
        buttonPanel.add(sv_invertedCheckBox, gridBagConstraints);

        sv_gridCheckBox.setBackground(new java.awt.Color(247, 221, 250));
        sv_gridCheckBox.setSelected(true);
        sv_gridCheckBox.setText("Grid");
        sv_gridCheckBox.setToolTipText("Show grid lines");
        sv_gridCheckBox.addMouseListener(new java.awt.event.MouseAdapter() {
            public void mouseClicked(java.awt.event.MouseEvent evt) {
                sv_gridCheckBoxMouseClicked(evt);
            }
        });
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 9;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.weightx = 1.0;
        gridBagConstraints.insets = new java.awt.Insets(0, 2, 0, 2);
        buttonPanel.add(sv_gridCheckBox, gridBagConstraints);

        unselectButton.setIcon(new javax.swing.ImageIcon(getClass().getResource("/opticalraytracer/icons/edit-clear.png"))); // NOI18N
        unselectButton.setToolTipText("Clear present lens selection");
        unselectButton.addMouseListener(new java.awt.event.MouseAdapter() {
            public void mouseClicked(java.awt.event.MouseEvent evt) {
                unselectButtonMouseClicked(evt);
            }
        });
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 5;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.EAST;
        gridBagConstraints.weightx = 1.0;
        gridBagConstraints.insets = new java.awt.Insets(1, 1, 1, 1);
        buttonPanel.add(unselectButton, gridBagConstraints);

        undoButton.setIcon(new javax.swing.ImageIcon(getClass().getResource("/opticalraytracer/icons/edit-undo.png"))); // NOI18N
        undoButton.setToolTipText("Undo Prior Action");
        undoButton.addMouseListener(new java.awt.event.MouseAdapter() {
            public void mouseClicked(java.awt.event.MouseEvent evt) {
                undoButtonMouseClicked(evt);
            }
        });
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 6;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.weightx = 1.0;
        gridBagConstraints.insets = new java.awt.Insets(1, 1, 1, 1);
        buttonPanel.add(undoButton, gridBagConstraints);

        redoButton.setIcon(new javax.swing.ImageIcon(getClass().getResource("/opticalraytracer/icons/edit-redo.png"))); // NOI18N
        redoButton.setToolTipText("Redo Undone Action");
        redoButton.addMouseListener(new java.awt.event.MouseAdapter() {
            public void mouseClicked(java.awt.event.MouseEvent evt) {
                redoButtonMouseClicked(evt);
            }
        });
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 7;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.weightx = 1.0;
        gridBagConstraints.insets = new java.awt.Insets(1, 1, 1, 1);
        buttonPanel.add(redoButton, gridBagConstraints);

        newLensButton.setIcon(new javax.swing.ImageIcon(getClass().getResource("/opticalraytracer/icons/document-new.png"))); // NOI18N
        newLensButton.setToolTipText("Create new Lens to right of existing lenses");
        newLensButton.addMouseListener(new java.awt.event.MouseAdapter() {
            public void mouseClicked(java.awt.event.MouseEvent evt) {
                newLensButtonMouseClicked(evt);
            }
        });
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.weightx = 1.0;
        gridBagConstraints.insets = new java.awt.Insets(1, 1, 1, 1);
        buttonPanel.add(newLensButton, gridBagConstraints);

        resetButton.setIcon(new javax.swing.ImageIcon(getClass().getResource("/opticalraytracer/icons/process-stop.png"))); // NOI18N
        resetButton.setToolTipText("Erase & reset all settings");
        resetButton.addMouseListener(new java.awt.event.MouseAdapter() {
            public void mouseClicked(java.awt.event.MouseEvent evt) {
                resetButtonMouseClicked(evt);
            }
        });
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 1;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.weightx = 1.0;
        gridBagConstraints.insets = new java.awt.Insets(1, 1, 1, 1);
        buttonPanel.add(resetButton, gridBagConstraints);

        cPanelButton.setText("Hide Controls");
        cPanelButton.setToolTipText("Hide control panel below");
        cPanelButton.addMouseListener(new java.awt.event.MouseAdapter() {
            public void mouseClicked(java.awt.event.MouseEvent evt) {
                cPanelButtonMouseClicked(evt);
            }
        });
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 11;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.weightx = 1.0;
        gridBagConstraints.insets = new java.awt.Insets(0, 2, 0, 2);
        buttonPanel.add(cPanelButton, gridBagConstraints);

        copyImageButton.setIcon(new javax.swing.ImageIcon(getClass().getResource("/opticalraytracer/icons/applications-multimedia.png"))); // NOI18N
        copyImageButton.setToolTipText("Copy workspace graphic image to clipboard");
        copyImageButton.addMouseListener(new java.awt.event.MouseAdapter() {
            public void mouseClicked(java.awt.event.MouseEvent evt) {
                copyImageButtonMouseClicked(evt);
            }
        });
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 2;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.weightx = 1.0;
        gridBagConstraints.insets = new java.awt.Insets(1, 1, 1, 1);
        buttonPanel.add(copyImageButton, gridBagConstraints);

        copyFullConfigButton.setIcon(new javax.swing.ImageIcon(getClass().getResource("/opticalraytracer/icons/edit-copy.png"))); // NOI18N
        copyFullConfigButton.setToolTipText("Copy full configuration to clipboard");
        copyFullConfigButton.addMouseListener(new java.awt.event.MouseAdapter() {
            public void mouseClicked(java.awt.event.MouseEvent evt) {
                copyFullConfigButtonMouseClicked(evt);
            }
        });
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 3;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.weightx = 1.0;
        gridBagConstraints.insets = new java.awt.Insets(1, 1, 1, 1);
        buttonPanel.add(copyFullConfigButton, gridBagConstraints);

        pasteFullConfigButton.setIcon(new javax.swing.ImageIcon(getClass().getResource("/opticalraytracer/icons/edit-paste.png"))); // NOI18N
        pasteFullConfigButton.setToolTipText("Paste full configuration from clipboard");
        pasteFullConfigButton.addMouseListener(new java.awt.event.MouseAdapter() {
            public void mouseClicked(java.awt.event.MouseEvent evt) {
                pasteFullConfigButtonMouseClicked(evt);
            }
        });
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 4;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.weightx = 1.0;
        gridBagConstraints.insets = new java.awt.Insets(1, 1, 1, 1);
        buttonPanel.add(pasteFullConfigButton, gridBagConstraints);

        quitButton.setIcon(new javax.swing.ImageIcon(getClass().getResource("/opticalraytracer/icons/application-exit.png"))); // NOI18N
        quitButton.setToolTipText("Quit OpticalRayTracer");
        quitButton.addMouseListener(new java.awt.event.MouseAdapter() {
            public void mouseClicked(java.awt.event.MouseEvent evt) {
                quitButtonMouseClicked(evt);
            }
        });
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 12;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.weightx = 1.0;
        gridBagConstraints.insets = new java.awt.Insets(0, 2, 0, 2);
        buttonPanel.add(quitButton, gridBagConstraints);

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 1;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.weightx = 1.0;
        gridBagConstraints.insets = new java.awt.Insets(2, 2, 2, 2);
        graphicPanel.add(buttonPanel, gridBagConstraints);

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.fill = java.awt.GridBagConstraints.BOTH;
        gridBagConstraints.weightx = 1.0;
        gridBagConstraints.weighty = 1.0;
        displayTab.add(graphicPanel, gridBagConstraints);

        controlPanel.setBackground(new java.awt.Color(198, 238, 254));
        controlPanel.setBorder(javax.swing.BorderFactory.createLineBorder(new java.awt.Color(0, 0, 0)));
        controlPanel.setLayout(new java.awt.GridBagLayout());

        leftLabel.setFont(new java.awt.Font("DejaVu Sans", 1, 14));
        leftLabel.setText("Left");
        leftLabel.setMinimumSize(new java.awt.Dimension(31, 27));
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 1;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        gridBagConstraints.weightx = 1.0;
        gridBagConstraints.weighty = 1.0;
        gridBagConstraints.insets = new java.awt.Insets(0, 4, 0, 0);
        controlPanel.add(leftLabel, gridBagConstraints);

        rightLabel.setFont(new java.awt.Font("DejaVu Sans", 1, 14));
        rightLabel.setText("Right");
        rightLabel.setMinimumSize(new java.awt.Dimension(40, 27));
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 2;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        gridBagConstraints.insets = new java.awt.Insets(0, 4, 0, 0);
        controlPanel.add(rightLabel, gridBagConstraints);

        sphereRadiusLabel.setFont(new java.awt.Font("DejaVu Sans", 1, 14));
        sphereRadiusLabel.setHorizontalAlignment(javax.swing.SwingConstants.CENTER);
        sphereRadiusLabel.setText("Sphere Radius");
        sphereRadiusLabel.setHorizontalTextPosition(javax.swing.SwingConstants.CENTER);
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 1;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        gridBagConstraints.weightx = 1.0;
        gridBagConstraints.insets = new java.awt.Insets(2, 0, 0, 0);
        controlPanel.add(sphereRadiusLabel, gridBagConstraints);

        cfLabel.setFont(new java.awt.Font("DejaVu Sans", 1, 14));
        cfLabel.setHorizontalAlignment(javax.swing.SwingConstants.CENTER);
        cfLabel.setText("Curvature Factor");
        cfLabel.setHorizontalTextPosition(javax.swing.SwingConstants.CENTER);
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 2;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        gridBagConstraints.weightx = 1.0;
        gridBagConstraints.insets = new java.awt.Insets(2, 0, 0, 0);
        controlPanel.add(cfLabel, gridBagConstraints);

        leftRadiusTextField.setFont(new java.awt.Font("Monospaced", 0, 14));
        leftRadiusTextField.setHorizontalAlignment(javax.swing.JTextField.RIGHT);
        leftRadiusTextField.setText("0.0");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 1;
        gridBagConstraints.gridy = 1;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        gridBagConstraints.weightx = 1.0;
        gridBagConstraints.insets = new java.awt.Insets(1, 1, 1, 1);
        controlPanel.add(leftRadiusTextField, gridBagConstraints);

        rightRadiusTextField.setFont(new java.awt.Font("Monospaced", 0, 14));
        rightRadiusTextField.setHorizontalAlignment(javax.swing.JTextField.RIGHT);
        rightRadiusTextField.setText("0.0");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 1;
        gridBagConstraints.gridy = 2;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        gridBagConstraints.weightx = 1.0;
        gridBagConstraints.insets = new java.awt.Insets(1, 1, 1, 1);
        controlPanel.add(rightRadiusTextField, gridBagConstraints);

        leftCFTextField.setFont(new java.awt.Font("Monospaced", 0, 14));
        leftCFTextField.setHorizontalAlignment(javax.swing.JTextField.RIGHT);
        leftCFTextField.setText("0.0");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 2;
        gridBagConstraints.gridy = 1;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        gridBagConstraints.weightx = 1.0;
        gridBagConstraints.insets = new java.awt.Insets(1, 1, 1, 1);
        controlPanel.add(leftCFTextField, gridBagConstraints);

        rightCFTextField.setFont(new java.awt.Font("Monospaced", 0, 14));
        rightCFTextField.setHorizontalAlignment(javax.swing.JTextField.RIGHT);
        rightCFTextField.setText("0.0");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 2;
        gridBagConstraints.gridy = 2;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        gridBagConstraints.weightx = 1.0;
        gridBagConstraints.insets = new java.awt.Insets(1, 1, 1, 1);
        controlPanel.add(rightCFTextField, gridBagConstraints);

        leftHypCheckBox.setBackground(new java.awt.Color(198, 238, 254));
        leftHypCheckBox.setAlignmentX(0.5F);
        leftHypCheckBox.setHorizontalAlignment(javax.swing.SwingConstants.CENTER);
        leftHypCheckBox.addMouseListener(new java.awt.event.MouseAdapter() {
            public void mouseClicked(java.awt.event.MouseEvent evt) {
                leftHypCheckBoxMouseClicked(evt);
            }
        });
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 3;
        gridBagConstraints.gridy = 1;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.weightx = 1.0;
        controlPanel.add(leftHypCheckBox, gridBagConstraints);

        rightHypCheckBox.setBackground(new java.awt.Color(198, 238, 254));
        rightHypCheckBox.setAlignmentX(0.5F);
        rightHypCheckBox.setHorizontalAlignment(javax.swing.SwingConstants.CENTER);
        rightHypCheckBox.addMouseListener(new java.awt.event.MouseAdapter() {
            public void mouseClicked(java.awt.event.MouseEvent evt) {
                rightHypCheckBoxMouseClicked(evt);
            }
        });
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 3;
        gridBagConstraints.gridy = 2;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.weightx = 1.0;
        controlPanel.add(rightHypCheckBox, gridBagConstraints);

        hypLabel.setFont(new java.awt.Font("DejaVu Sans", 1, 14)); // NOI18N
        hypLabel.setHorizontalAlignment(javax.swing.SwingConstants.CENTER);
        hypLabel.setText("Hyperboloid");
        hypLabel.setHorizontalTextPosition(javax.swing.SwingConstants.CENTER);
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 3;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        gridBagConstraints.weightx = 1.0;
        gridBagConstraints.insets = new java.awt.Insets(2, 0, 0, 0);
        controlPanel.add(hypLabel, gridBagConstraints);

        controlSubPanel.setOpaque(false);
        controlSubPanel.setLayout(new java.awt.GridBagLayout());

        iorLabel.setHorizontalAlignment(javax.swing.SwingConstants.CENTER);
        iorLabel.setText("IOR");
        iorLabel.setToolTipText("Index of Refraction");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 4;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        gridBagConstraints.weightx = 0.25;
        gridBagConstraints.insets = new java.awt.Insets(2, 4, 2, 4);
        controlSubPanel.add(iorLabel, gridBagConstraints);

        iorTextField.setFont(new java.awt.Font("Monospaced", 0, 14));
        iorTextField.setHorizontalAlignment(javax.swing.JTextField.RIGHT);
        iorTextField.setText("0.0");
        iorTextField.setToolTipText("Index of refraction");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 5;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        gridBagConstraints.weightx = 1.0;
        gridBagConstraints.insets = new java.awt.Insets(1, 1, 1, 1);
        controlSubPanel.add(iorTextField, gridBagConstraints);

        dispersionLabel.setHorizontalAlignment(javax.swing.SwingConstants.CENTER);
        dispersionLabel.setText("Dispersion");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 6;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        gridBagConstraints.weightx = 0.25;
        gridBagConstraints.insets = new java.awt.Insets(2, 4, 2, 4);
        controlSubPanel.add(dispersionLabel, gridBagConstraints);

        dispersionTextField.setFont(new java.awt.Font("Monospaced", 0, 14));
        dispersionTextField.setHorizontalAlignment(javax.swing.JTextField.RIGHT);
        dispersionTextField.setText("0.0");
        dispersionTextField.setToolTipText("A wavelength-sensitive property of some media");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 7;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.EAST;
        gridBagConstraints.weightx = 1.0;
        gridBagConstraints.insets = new java.awt.Insets(1, 1, 1, 1);
        controlSubPanel.add(dispersionTextField, gridBagConstraints);

        lensRadiusLabel.setHorizontalAlignment(javax.swing.SwingConstants.CENTER);
        lensRadiusLabel.setText("Lens Radius");
        lensRadiusLabel.setMinimumSize(new java.awt.Dimension(84, 27));
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.weightx = 0.25;
        gridBagConstraints.insets = new java.awt.Insets(0, 4, 0, 0);
        controlSubPanel.add(lensRadiusLabel, gridBagConstraints);

        lensRadiusTextField.setHorizontalAlignment(javax.swing.JTextField.RIGHT);
        lensRadiusTextField.setText("0.0");
        lensRadiusTextField.setToolTipText("Center-to-edge radius");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 1;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        gridBagConstraints.weightx = 1.0;
        gridBagConstraints.insets = new java.awt.Insets(1, 1, 1, 1);
        controlSubPanel.add(lensRadiusTextField, gridBagConstraints);

        thicknessLabel1.setHorizontalAlignment(javax.swing.SwingConstants.CENTER);
        thicknessLabel1.setText("Thickness");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 2;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.insets = new java.awt.Insets(2, 4, 2, 4);
        controlSubPanel.add(thicknessLabel1, gridBagConstraints);

        userThicknessTextField.setHorizontalAlignment(javax.swing.JTextField.RIGHT);
        userThicknessTextField.setText("0.0");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 3;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.weightx = 1.0;
        gridBagConstraints.insets = new java.awt.Insets(1, 1, 1, 1);
        controlSubPanel.add(userThicknessTextField, gridBagConstraints);

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 3;
        gridBagConstraints.gridwidth = 6;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.weightx = 1.0;
        controlPanel.add(controlSubPanel, gridBagConstraints);

        posLabel.setFont(new java.awt.Font("DejaVu Sans", 1, 14)); // NOI18N
        posLabel.setHorizontalAlignment(javax.swing.SwingConstants.CENTER);
        posLabel.setText("Position");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 4;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.gridwidth = 2;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.weightx = 1.0;
        gridBagConstraints.insets = new java.awt.Insets(2, 0, 0, 0);
        controlPanel.add(posLabel, gridBagConstraints);

        xPosLabel.setFont(new java.awt.Font("DejaVu Sans", 1, 14));
        xPosLabel.setText("X");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 4;
        gridBagConstraints.gridy = 1;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        gridBagConstraints.weightx = 0.25;
        gridBagConstraints.insets = new java.awt.Insets(2, 4, 2, 4);
        controlPanel.add(xPosLabel, gridBagConstraints);

        xPosTextField.setFont(new java.awt.Font("Monospaced", 0, 14));
        xPosTextField.setHorizontalAlignment(javax.swing.JTextField.RIGHT);
        xPosTextField.setText("0.0");
        xPosTextField.setToolTipText("Lens X coordinate");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 5;
        gridBagConstraints.gridy = 1;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        gridBagConstraints.weightx = 2.0;
        gridBagConstraints.insets = new java.awt.Insets(1, 1, 1, 1);
        controlPanel.add(xPosTextField, gridBagConstraints);

        yPosLabel.setFont(new java.awt.Font("DejaVu Sans", 1, 14));
        yPosLabel.setText("Y");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 4;
        gridBagConstraints.gridy = 2;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        gridBagConstraints.weightx = 0.25;
        gridBagConstraints.insets = new java.awt.Insets(2, 4, 2, 4);
        controlPanel.add(yPosLabel, gridBagConstraints);

        yPosTextField.setFont(new java.awt.Font("Monospaced", 0, 14));
        yPosTextField.setHorizontalAlignment(javax.swing.JTextField.RIGHT);
        yPosTextField.setText("0.0");
        yPosTextField.setToolTipText("Lens Y coordinate");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 5;
        gridBagConstraints.gridy = 2;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        gridBagConstraints.weightx = 2.0;
        gridBagConstraints.insets = new java.awt.Insets(1, 1, 1, 1);
        controlPanel.add(yPosTextField, gridBagConstraints);

        symmCheckBox.setBackground(new java.awt.Color(198, 238, 254));
        symmCheckBox.setText("Symmetrical");
        symmCheckBox.addMouseListener(new java.awt.event.MouseAdapter() {
            public void mouseClicked(java.awt.event.MouseEvent evt) {
                symmCheckBoxMouseClicked(evt);
            }
        });
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        gridBagConstraints.weightx = 0.25;
        gridBagConstraints.insets = new java.awt.Insets(0, 4, 0, 0);
        controlPanel.add(symmCheckBox, gridBagConstraints);

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 1;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.weightx = 1.0;
        gridBagConstraints.insets = new java.awt.Insets(2, 2, 2, 2);
        displayTab.add(controlPanel, gridBagConstraints);

        sv_mainTabbedPane.addTab("Design", new javax.swing.ImageIcon(getClass().getResource("/opticalraytracer/icons/applications-graphics.png")), displayTab); // NOI18N

        configurationTab.setLayout(new java.awt.GridBagLayout());

        graphicPlaceholder2.setBorder(javax.swing.BorderFactory.createLineBorder(new java.awt.Color(0, 0, 0)));
        graphicPlaceholder2.setLayout(new java.awt.BorderLayout());
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.fill = java.awt.GridBagConstraints.BOTH;
        gridBagConstraints.weightx = 1.0;
        gridBagConstraints.weighty = 1.0;
        gridBagConstraints.insets = new java.awt.Insets(2, 2, 2, 2);
        configurationTab.add(graphicPlaceholder2, gridBagConstraints);

        colorPanel.setBackground(new java.awt.Color(254, 250, 221));
        colorPanel.setBorder(javax.swing.BorderFactory.createLineBorder(new java.awt.Color(0, 0, 0)));
        colorPanel.setLayout(new java.awt.GridBagLayout());

        baselineColorButton.setText("<html>&nbsp;&nbsp;&nbsp;&nbsp;</html>");
        baselineColorButton.setMinimumSize(new java.awt.Dimension(64, 32));
        baselineColorButton.setPreferredSize(new java.awt.Dimension(48, 24));
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 2;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.weightx = 1.0;
        gridBagConstraints.insets = new java.awt.Insets(2, 2, 2, 2);
        colorPanel.add(baselineColorButton, gridBagConstraints);

        gridColorButton.setPreferredSize(new java.awt.Dimension(48, 24));
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 3;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.weightx = 1.0;
        gridBagConstraints.insets = new java.awt.Insets(2, 2, 2, 2);
        colorPanel.add(gridColorButton, gridBagConstraints);

        lensOutlineColorButton.setPreferredSize(new java.awt.Dimension(48, 24));
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 4;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.weightx = 1.0;
        gridBagConstraints.insets = new java.awt.Insets(2, 2, 2, 2);
        colorPanel.add(lensOutlineColorButton, gridBagConstraints);

        beamColorButton.setPreferredSize(new java.awt.Dimension(48, 24));
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 5;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.weightx = 1.0;
        gridBagConstraints.insets = new java.awt.Insets(2, 2, 2, 2);
        colorPanel.add(beamColorButton, gridBagConstraints);

        intersectionColorButton.setPreferredSize(new java.awt.Dimension(48, 24));
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 6;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.weightx = 1.0;
        gridBagConstraints.insets = new java.awt.Insets(2, 2, 2, 2);
        colorPanel.add(intersectionColorButton, gridBagConstraints);

        lensSelColorButton.setPreferredSize(new java.awt.Dimension(48, 24));
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 7;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.weightx = 1.0;
        gridBagConstraints.insets = new java.awt.Insets(2, 2, 2, 2);
        colorPanel.add(lensSelColorButton, gridBagConstraints);

        dispHiColorButton.setPreferredSize(new java.awt.Dimension(48, 24));
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 8;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.weightx = 1.0;
        gridBagConstraints.insets = new java.awt.Insets(2, 2, 2, 2);
        colorPanel.add(dispHiColorButton, gridBagConstraints);

        dispLoColorButton.setPreferredSize(new java.awt.Dimension(48, 24));
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 9;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.weightx = 1.0;
        gridBagConstraints.insets = new java.awt.Insets(2, 2, 2, 2);
        colorPanel.add(dispLoColorButton, gridBagConstraints);

        jLabel1.setHorizontalAlignment(javax.swing.SwingConstants.CENTER);
        jLabel1.setText("Choose colors:");
        jLabel1.setToolTipText("Hover over buttons for names");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.weightx = 1.0;
        gridBagConstraints.insets = new java.awt.Insets(0, 4, 0, 4);
        colorPanel.add(jLabel1, gridBagConstraints);

        barrierColorButton.setText("<html>&nbsp;&nbsp;&nbsp;&nbsp;</html>");
        barrierColorButton.setMinimumSize(new java.awt.Dimension(64, 32));
        barrierColorButton.setPreferredSize(new java.awt.Dimension(48, 24));
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 1;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.weightx = 1.0;
        gridBagConstraints.insets = new java.awt.Insets(2, 2, 2, 2);
        colorPanel.add(barrierColorButton, gridBagConstraints);

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 1;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.weightx = 1.0;
        gridBagConstraints.insets = new java.awt.Insets(2, 2, 2, 2);
        configurationTab.add(colorPanel, gridBagConstraints);

        numericPanel.setBackground(new java.awt.Color(209, 246, 234));
        numericPanel.setBorder(javax.swing.BorderFactory.createLineBorder(new java.awt.Color(0, 0, 0)));
        numericPanel.setLayout(new java.awt.GridBagLayout());

        jLabel2.setHorizontalAlignment(javax.swing.SwingConstants.RIGHT);
        jLabel2.setText("Intersection dot size");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.weightx = 0.25;
        gridBagConstraints.insets = new java.awt.Insets(0, 4, 0, 4);
        numericPanel.add(jLabel2, gridBagConstraints);

        sv_dotRadiusTextField.setHorizontalAlignment(javax.swing.JTextField.RIGHT);
        sv_dotRadiusTextField.setText("0.0");
        sv_dotRadiusTextField.setToolTipText("Negative radii = solid dots");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.weightx = 1.0;
        gridBagConstraints.insets = new java.awt.Insets(1, 1, 1, 1);
        numericPanel.add(sv_dotRadiusTextField, gridBagConstraints);

        jLabel3.setHorizontalAlignment(javax.swing.SwingConstants.RIGHT);
        jLabel3.setText("Snap-to-base band");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.weightx = 0.25;
        gridBagConstraints.insets = new java.awt.Insets(0, 4, 0, 4);
        numericPanel.add(jLabel3, gridBagConstraints);

        sv_snapToBaseTextField.setHorizontalAlignment(javax.swing.JTextField.RIGHT);
        sv_snapToBaseTextField.setText("0.0");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.weightx = 1.0;
        gridBagConstraints.insets = new java.awt.Insets(1, 1, 1, 1);
        numericPanel.add(sv_snapToBaseTextField, gridBagConstraints);

        jLabel4.setHorizontalAlignment(javax.swing.SwingConstants.RIGHT);
        jLabel4.setText("Beam width");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.weightx = 0.25;
        gridBagConstraints.insets = new java.awt.Insets(0, 4, 0, 4);
        numericPanel.add(jLabel4, gridBagConstraints);

        sv_beamWidthTextField.setHorizontalAlignment(javax.swing.JTextField.RIGHT);
        sv_beamWidthTextField.setText("0.0");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.weightx = 1.0;
        gridBagConstraints.insets = new java.awt.Insets(1, 1, 1, 1);
        numericPanel.add(sv_beamWidthTextField, gridBagConstraints);

        jLabel5.setHorizontalAlignment(javax.swing.SwingConstants.RIGHT);
        jLabel5.setText("Beam count");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 1;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.weightx = 0.25;
        gridBagConstraints.insets = new java.awt.Insets(0, 4, 0, 4);
        numericPanel.add(jLabel5, gridBagConstraints);

        sv_beamCountTextField.setHorizontalAlignment(javax.swing.JTextField.RIGHT);
        sv_beamCountTextField.setText("0.0");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 1;
        gridBagConstraints.gridy = 1;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.weightx = 1.0;
        gridBagConstraints.insets = new java.awt.Insets(1, 1, 1, 1);
        numericPanel.add(sv_beamCountTextField, gridBagConstraints);

        jLabel6.setHorizontalAlignment(javax.swing.SwingConstants.RIGHT);
        jLabel6.setText("Interactions per beam");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 2;
        gridBagConstraints.gridy = 1;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.weightx = 0.25;
        gridBagConstraints.insets = new java.awt.Insets(0, 4, 0, 4);
        numericPanel.add(jLabel6, gridBagConstraints);

        sv_intersectionsTextField.setHorizontalAlignment(javax.swing.JTextField.RIGHT);
        sv_intersectionsTextField.setText("0.0");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 3;
        gridBagConstraints.gridy = 1;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.weightx = 1.0;
        gridBagConstraints.insets = new java.awt.Insets(1, 1, 1, 1);
        numericPanel.add(sv_intersectionsTextField, gridBagConstraints);

        jLabel7.setHorizontalAlignment(javax.swing.SwingConstants.RIGHT);
        jLabel7.setText("Source Y start");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 4;
        gridBagConstraints.gridy = 1;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.weightx = 0.25;
        gridBagConstraints.insets = new java.awt.Insets(0, 4, 0, 4);
        numericPanel.add(jLabel7, gridBagConstraints);

        sv_yStartTextField.setHorizontalAlignment(javax.swing.JTextField.RIGHT);
        sv_yStartTextField.setText("0.0");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 5;
        gridBagConstraints.gridy = 1;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.weightx = 1.0;
        gridBagConstraints.insets = new java.awt.Insets(1, 1, 1, 1);
        numericPanel.add(sv_yStartTextField, gridBagConstraints);

        jLabel8.setHorizontalAlignment(javax.swing.SwingConstants.RIGHT);
        jLabel8.setText("Source Y end");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 2;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.weightx = 0.25;
        gridBagConstraints.insets = new java.awt.Insets(0, 4, 0, 4);
        numericPanel.add(jLabel8, gridBagConstraints);

        sv_yEndTextField.setHorizontalAlignment(javax.swing.JTextField.RIGHT);
        sv_yEndTextField.setText("0.0");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 1;
        gridBagConstraints.gridy = 2;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.weightx = 1.0;
        gridBagConstraints.insets = new java.awt.Insets(1, 1, 1, 1);
        numericPanel.add(sv_yEndTextField, gridBagConstraints);

        jLabel9.setHorizontalAlignment(javax.swing.SwingConstants.RIGHT);
        jLabel9.setText("X source plane");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 2;
        gridBagConstraints.gridy = 2;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.weightx = 0.25;
        gridBagConstraints.insets = new java.awt.Insets(0, 4, 0, 4);
        numericPanel.add(jLabel9, gridBagConstraints);

        sv_xSourcePlaneTextField.setHorizontalAlignment(javax.swing.JTextField.RIGHT);
        sv_xSourcePlaneTextField.setText("0.0");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 3;
        gridBagConstraints.gridy = 2;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.weightx = 1.0;
        gridBagConstraints.insets = new java.awt.Insets(1, 1, 1, 1);
        numericPanel.add(sv_xSourcePlaneTextField, gridBagConstraints);

        jLabel10.setHorizontalAlignment(javax.swing.SwingConstants.RIGHT);
        jLabel10.setText("X target plane");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 4;
        gridBagConstraints.gridy = 2;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.weightx = 0.25;
        gridBagConstraints.insets = new java.awt.Insets(0, 4, 0, 4);
        numericPanel.add(jLabel10, gridBagConstraints);

        sv_xTargetPlaneTextField.setHorizontalAlignment(javax.swing.JTextField.RIGHT);
        sv_xTargetPlaneTextField.setText("0.0");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 5;
        gridBagConstraints.gridy = 2;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.weightx = 1.0;
        gridBagConstraints.insets = new java.awt.Insets(1, 1, 1, 1);
        numericPanel.add(sv_xTargetPlaneTextField, gridBagConstraints);

        jLabel11.setHorizontalAlignment(javax.swing.SwingConstants.RIGHT);
        jLabel11.setText("Beam offset angle");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 3;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.weightx = 0.25;
        gridBagConstraints.insets = new java.awt.Insets(0, 4, 0, 4);
        numericPanel.add(jLabel11, gridBagConstraints);

        sv_offsetAngleTextField.setHorizontalAlignment(javax.swing.JTextField.RIGHT);
        sv_offsetAngleTextField.setText("0.0");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 1;
        gridBagConstraints.gridy = 3;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.weightx = 1.0;
        gridBagConstraints.insets = new java.awt.Insets(1, 1, 1, 1);
        numericPanel.add(sv_offsetAngleTextField, gridBagConstraints);

        jLabel12.setHorizontalAlignment(javax.swing.SwingConstants.RIGHT);
        jLabel12.setText("Dispersion beam count");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 2;
        gridBagConstraints.gridy = 3;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.weightx = 0.25;
        gridBagConstraints.insets = new java.awt.Insets(0, 4, 0, 4);
        numericPanel.add(jLabel12, gridBagConstraints);

        sv_dispersionBeamsTextField.setHorizontalAlignment(javax.swing.JTextField.RIGHT);
        sv_dispersionBeamsTextField.setText("0.0");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 3;
        gridBagConstraints.gridy = 3;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.weightx = 1.0;
        gridBagConstraints.insets = new java.awt.Insets(1, 1, 1, 1);
        numericPanel.add(sv_dispersionBeamsTextField, gridBagConstraints);

        sv_divergingSourceCheckBox.setBackground(new java.awt.Color(209, 246, 234));
        sv_divergingSourceCheckBox.setText("Diverging beams");
        sv_divergingSourceCheckBox.setHorizontalAlignment(javax.swing.SwingConstants.CENTER);
        sv_divergingSourceCheckBox.addMouseListener(new java.awt.event.MouseAdapter() {
            public void mouseClicked(java.awt.event.MouseEvent evt) {
                sv_divergingSourceCheckBoxMouseClicked(evt);
            }
        });
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 4;
        gridBagConstraints.gridy = 3;
        gridBagConstraints.gridwidth = 2;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.weightx = 1.0;
        gridBagConstraints.insets = new java.awt.Insets(2, 2, 2, 2);
        numericPanel.add(sv_divergingSourceCheckBox, gridBagConstraints);

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 2;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.weightx = 1.0;
        gridBagConstraints.insets = new java.awt.Insets(2, 2, 2, 2);
        configurationTab.add(numericPanel, gridBagConstraints);

        sv_mainTabbedPane.addTab("Configure", new javax.swing.ImageIcon(getClass().getResource("/opticalraytracer/icons/applications-accessories.png")), configurationTab); // NOI18N

        helpTab.setLayout(new java.awt.BorderLayout());
        sv_mainTabbedPane.addTab("Help", new javax.swing.ImageIcon(getClass().getResource("/opticalraytracer/icons/system-help.png")), helpTab); // NOI18N

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.fill = java.awt.GridBagConstraints.BOTH;
        gridBagConstraints.weightx = 1.0;
        gridBagConstraints.weighty = 1.0;
        gridBagConstraints.insets = new java.awt.Insets(1, 1, 1, 1);
        getContentPane().add(sv_mainTabbedPane, gridBagConstraints);

        statusLabel.setFont(new java.awt.Font("Monospaced", 1, 14));
        statusLabel.setText("Cursor Position");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 1;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.weightx = 1.0;
        gridBagConstraints.insets = new java.awt.Insets(2, 4, 2, 0);
        getContentPane().add(statusLabel, gridBagConstraints);

        pack();
    }// </editor-fold>//GEN-END:initComponents

    private void formWindowClosing(java.awt.event.WindowEvent evt) {//GEN-FIRST:event_formWindowClosing
        // TODO add your handling code here:
        close();
    }//GEN-LAST:event_formWindowClosing

    private void symmCheckBoxMouseClicked(java.awt.event.MouseEvent evt) {//GEN-FIRST:event_symmCheckBoxMouseClicked
        // TODO add your handling code here:
        updateGraphicDisplay();
    }//GEN-LAST:event_symmCheckBoxMouseClicked

    private void leftHypCheckBoxMouseClicked(java.awt.event.MouseEvent evt) {//GEN-FIRST:event_leftHypCheckBoxMouseClicked
        // TODO add your handling code here:
        updateGraphicDisplay();
}//GEN-LAST:event_leftHypCheckBoxMouseClicked

    private void rightHypCheckBoxMouseClicked(java.awt.event.MouseEvent evt) {//GEN-FIRST:event_rightHypCheckBoxMouseClicked
        // TODO add your handling code here:
        updateGraphicDisplay();
    }//GEN-LAST:event_rightHypCheckBoxMouseClicked

    private void sv_antiAliasCheckBoxMouseClicked(java.awt.event.MouseEvent evt) {//GEN-FIRST:event_sv_antiAliasCheckBoxMouseClicked
        // TODO add your handling code here:
        readControls();
}//GEN-LAST:event_sv_antiAliasCheckBoxMouseClicked

    private void sv_invertedCheckBoxMouseClicked(java.awt.event.MouseEvent evt) {//GEN-FIRST:event_sv_invertedCheckBoxMouseClicked
        // TODO add your handling code here:
        readControls();
}//GEN-LAST:event_sv_invertedCheckBoxMouseClicked

    private void sv_gridCheckBoxMouseClicked(java.awt.event.MouseEvent evt) {//GEN-FIRST:event_sv_gridCheckBoxMouseClicked
        // TODO add your handling code here:
        readControls();
}//GEN-LAST:event_sv_gridCheckBoxMouseClicked

    private void unselectButtonMouseClicked(java.awt.event.MouseEvent evt) {//GEN-FIRST:event_unselectButtonMouseClicked
        // TODO add your handling code here:
        unSelectLens();
    }//GEN-LAST:event_unselectButtonMouseClicked

    private void undoButtonMouseClicked(java.awt.event.MouseEvent evt) {//GEN-FIRST:event_undoButtonMouseClicked
        // TODO add your handling code here:
        undo();
    }//GEN-LAST:event_undoButtonMouseClicked

    private void redoButtonMouseClicked(java.awt.event.MouseEvent evt) {//GEN-FIRST:event_redoButtonMouseClicked
        // TODO add your handling code here:
        redo();
    }//GEN-LAST:event_redoButtonMouseClicked

    private void newLensButtonMouseClicked(java.awt.event.MouseEvent evt) {//GEN-FIRST:event_newLensButtonMouseClicked
        // TODO add your handling code here:
        makeNewLens();
    }//GEN-LAST:event_newLensButtonMouseClicked

    private void resetButtonMouseClicked(java.awt.event.MouseEvent evt) {//GEN-FIRST:event_resetButtonMouseClicked
        // TODO add your handling code here:
        eraseReset();
    }//GEN-LAST:event_resetButtonMouseClicked

    private void cPanelButtonMouseClicked(java.awt.event.MouseEvent evt) {//GEN-FIRST:event_cPanelButtonMouseClicked
        // TODO add your handling code here:
        showHideControls();
    }//GEN-LAST:event_cPanelButtonMouseClicked

    private void sv_divergingSourceCheckBoxMouseClicked(java.awt.event.MouseEvent evt) {//GEN-FIRST:event_sv_divergingSourceCheckBoxMouseClicked
        // TODO add your handling code here:
        readControls();
    }//GEN-LAST:event_sv_divergingSourceCheckBoxMouseClicked

    private void copyImageButtonMouseClicked(java.awt.event.MouseEvent evt) {//GEN-FIRST:event_copyImageButtonMouseClicked
        // TODO add your handling code here:
        clipboardCopyImage();
    }//GEN-LAST:event_copyImageButtonMouseClicked

    private void copyFullConfigButtonMouseClicked(java.awt.event.MouseEvent evt) {//GEN-FIRST:event_copyFullConfigButtonMouseClicked
        // TODO add your handling code here:
        clipboardCopyFullConfig();
    }//GEN-LAST:event_copyFullConfigButtonMouseClicked

    private void pasteFullConfigButtonMouseClicked(java.awt.event.MouseEvent evt) {//GEN-FIRST:event_pasteFullConfigButtonMouseClicked
        // TODO add your handling code here:
        clipboardPasteFullConfig();
    }//GEN-LAST:event_pasteFullConfigButtonMouseClicked

    private void quitButtonMouseClicked(java.awt.event.MouseEvent evt) {//GEN-FIRST:event_quitButtonMouseClicked
        // TODO add your handling code here:
        close();
    }//GEN-LAST:event_quitButtonMouseClicked

    private void sv_mainTabbedPaneMouseClicked(java.awt.event.MouseEvent evt) {//GEN-FIRST:event_sv_mainTabbedPaneMouseClicked
        // TODO add your handling code here:
        processTabClick();
    }//GEN-LAST:event_sv_mainTabbedPaneMouseClicked

    /**
     * @param args the command line arguments
     */
    public static void main(String args[]) {

        try {
            // Default to system-specific L&F
            UIManager.setLookAndFeel(UIManager.getSystemLookAndFeelClassName());
        } catch (Exception e) {
            // handle exception
        }

        java.awt.EventQueue.invokeLater(new Runnable() {

            public void run() {
                new OpticalRayTracer().setVisible(true);
            }
        });
    }

    // Variables declaration - do not modify//GEN-BEGIN:variables
    private javax.swing.JButton barrierColorButton;
    private javax.swing.JButton baselineColorButton;
    private javax.swing.JButton beamColorButton;
    private javax.swing.JPanel buttonPanel;
    private javax.swing.JButton cPanelButton;
    private javax.swing.JLabel cfLabel;
    private javax.swing.JPanel colorPanel;
    private javax.swing.JPanel configurationTab;
    private javax.swing.JPanel controlPanel;
    private javax.swing.JPanel controlSubPanel;
    private javax.swing.JButton copyFullConfigButton;
    private javax.swing.JButton copyImageButton;
    private javax.swing.JButton dispHiColorButton;
    private javax.swing.JButton dispLoColorButton;
    private javax.swing.JLabel dispersionLabel;
    protected javax.swing.JTextField dispersionTextField;
    private javax.swing.JPanel displayTab;
    private javax.swing.JPanel graphicPanel;
    private javax.swing.JPanel graphicPlaceholder1;
    private javax.swing.JPanel graphicPlaceholder2;
    private javax.swing.JButton gridColorButton;
    private javax.swing.JPanel helpTab;
    private javax.swing.JLabel hypLabel;
    private javax.swing.JButton intersectionColorButton;
    private javax.swing.JLabel iorLabel;
    protected javax.swing.JTextField iorTextField;
    private javax.swing.JLabel jLabel1;
    private javax.swing.JLabel jLabel10;
    private javax.swing.JLabel jLabel11;
    private javax.swing.JLabel jLabel12;
    private javax.swing.JLabel jLabel2;
    private javax.swing.JLabel jLabel3;
    private javax.swing.JLabel jLabel4;
    private javax.swing.JLabel jLabel5;
    private javax.swing.JLabel jLabel6;
    private javax.swing.JLabel jLabel7;
    private javax.swing.JLabel jLabel8;
    private javax.swing.JLabel jLabel9;
    protected javax.swing.JTextField leftCFTextField;
    protected javax.swing.JCheckBox leftHypCheckBox;
    private javax.swing.JLabel leftLabel;
    protected javax.swing.JTextField leftRadiusTextField;
    private javax.swing.JButton lensOutlineColorButton;
    private javax.swing.JLabel lensRadiusLabel;
    protected javax.swing.JTextField lensRadiusTextField;
    private javax.swing.JButton lensSelColorButton;
    private javax.swing.JButton newLensButton;
    private javax.swing.JPanel numericPanel;
    private javax.swing.JButton pasteFullConfigButton;
    private javax.swing.JLabel posLabel;
    private javax.swing.JButton quitButton;
    private javax.swing.JButton redoButton;
    private javax.swing.JButton resetButton;
    protected javax.swing.JTextField rightCFTextField;
    protected javax.swing.JCheckBox rightHypCheckBox;
    private javax.swing.JLabel rightLabel;
    protected javax.swing.JTextField rightRadiusTextField;
    private javax.swing.JLabel sphereRadiusLabel;
    protected javax.swing.JLabel statusLabel;
    protected javax.swing.JCheckBox sv_antiAliasCheckBox;
    protected javax.swing.JTextField sv_beamCountTextField;
    protected javax.swing.JTextField sv_beamWidthTextField;
    protected javax.swing.JTextField sv_dispersionBeamsTextField;
    protected javax.swing.JCheckBox sv_divergingSourceCheckBox;
    protected javax.swing.JTextField sv_dotRadiusTextField;
    protected javax.swing.JCheckBox sv_gridCheckBox;
    protected javax.swing.JTextField sv_intersectionsTextField;
    protected javax.swing.JCheckBox sv_invertedCheckBox;
    protected javax.swing.JTabbedPane sv_mainTabbedPane;
    protected javax.swing.JTextField sv_offsetAngleTextField;
    protected javax.swing.JTextField sv_snapToBaseTextField;
    protected javax.swing.JTextField sv_xSourcePlaneTextField;
    protected javax.swing.JTextField sv_xTargetPlaneTextField;
    protected javax.swing.JTextField sv_yEndTextField;
    protected javax.swing.JTextField sv_yStartTextField;
    protected javax.swing.JCheckBox symmCheckBox;
    private javax.swing.JLabel thicknessLabel1;
    private javax.swing.JButton undoButton;
    protected javax.swing.JButton unselectButton;
    protected javax.swing.JTextField userThicknessTextField;
    private javax.swing.JLabel xPosLabel;
    protected javax.swing.JTextField xPosTextField;
    private javax.swing.JLabel yPosLabel;
    protected javax.swing.JTextField yPosTextField;
    // End of variables declaration//GEN-END:variables
}
