
/*
 * Copyright (c) Abraham vd Merwe <abz@blio.com>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *	  notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *	  notice, this list of conditions and the following disclaimer in the
 *	  documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the author nor the names of other contributors
 *	  may be used to endorse or promote products derived from this software
 *	  without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <pwd.h>
#include <grp.h>
#include <sys/types.h>

#include <debug/memory.h>
#include <debug/log.h>
#include <config/parse.h>

#include "bprintf.h"
#include "config.h"

/*
 * free all memory allocated by config_parse() for the specified
 * config structure.
 */
void config_destroy (config_t *config)
{
   if (config != NULL)
	 {
		mem_free (config->database);
		mem_free (config->htdocs);
		mem_free (config->documentroot);
		mem_free (config->pidfile);

		if (config->n)
		  {
			 int i;

			 for (i = 0; i < config->n; i++)
			   {
				  mem_free (config->router[i].hostname);
				  if (config->router[i].service != NULL) mem_free (config->router[i].service);
				  mem_free (config->router[i].community);

				  if (config->router[i].n_ignore)
					{
					   int j;
					   for (j = 0; j < config->router[i].n_ignore; j++) mem_free (config->router[i].ignore[j]);
					   mem_free (config->router[i].ignore);
					}

				  if (config->router[i].n_order)
					{
					   int j;
					   for (j = 0; j < config->router[i].n_order; j++) mem_free (config->router[i].order[j]);
					   mem_free (config->router[i].order);
					}

				  if (config->router[i].n_alias)
					{
					   int j;
					   for (j = 0; j < config->router[i].n_alias; j++) mem_free (config->router[i].alias[j]);
					   mem_free (config->router[i].alias);
					}
			   }

			 mem_free (config->router);
		  }
	 }
}

static stmt_t *find_statement (stmt_t *s,const char *name)
{
   stmt_t *tmp,*prev = s;

   if (s == NULL) return (NULL);

   do
	 {
		tmp = s->next;
		if (!strcmp (s->name,name)) return (s);
		s = tmp;
	 }
   while (prev != s);

   return (NULL);
}

static int get_port (char **str,const char *configfile,stmt_t *stmt)
{
   *str = NULL;

   if ((stmt = find_statement (stmt,"port")) != NULL)
	 {
		if (stmt->n != 1)
		  {
			 type_mismatch:
			 log_printf (LOG_ERROR,"%s: ``port'' value is neither a string nor an integer!\n",configfile);
			 return (-1);
		  }

		switch (stmt->type)
		  {
		   case TOK_STRING:
			 if ((*str = (char *) mem_alloc ((strlen (stmt->args->string) + 1) * sizeof (char))) == NULL)
			   {
				  out_of_memory:
				  log_printf (LOG_ERROR,"Couldn't allocate memory for temporary storage: %s\n",strerror (errno));
				  return (-1);
			   }
			 strcpy (*str,stmt->args->string);
			 break;

		   case TOK_INTEGER:
			 if (stmt->args->integer < 1 || stmt->args->integer > 65535)
			   {
				  log_printf (LOG_ERROR,"%s: ``port'' value out of range!\n",configfile);
				  return (-1);
			   }
			 if ((*str = bprintf ("%ld",stmt->args->integer)) == NULL)
			   goto out_of_memory;
			 break;

		   default:
			 goto type_mismatch;
		  }
	 }

   return (0);
}

static int get_string_optional (char **result,const char *configfile,stmt_t *stmt,const char *name)
{
   char *str;

   if ((stmt = find_statement (stmt,name)) == NULL)
	 {
		*result = NULL;
		return (0);
	 }

   if (stmt->type != TOK_STRING || stmt->n != 1)
	 {
		log_printf (LOG_ERROR,"%s: ``%s'' value is not a string!\n",configfile,name);
		return (-1);
	 }

   if ((str = (char *) mem_alloc ((strlen (stmt->args->string) + 1) * sizeof (char))) == NULL)
	 {
		log_printf (LOG_ERROR,"Couldn't allocate memory for temporary storage: %s\n",strerror (errno));
		return (-1);
	 }

   strcpy (str,stmt->args->string);

   *result = str;

   return (0);
}

static char *get_string (const char *configfile,stmt_t *stmt,const char *name)
{
   char *str;

   if ((stmt = find_statement (stmt,name)) == NULL)
	 {
		log_printf (LOG_ERROR,"%s: No ``%s'' statement found!\n",configfile,name);
		return (NULL);
	 }

   if (stmt->type != TOK_STRING || stmt->n != 1)
	 {
		log_printf (LOG_ERROR,"%s: ``%s'' value is not a string!\n",configfile,name);
		return (NULL);
	 }

   if ((str = (char *) mem_alloc ((strlen (stmt->args->string) + 1) * sizeof (char))) == NULL)
	 {
		log_printf (LOG_ERROR,"Couldn't allocate memory for temporary storage: %s\n",strerror (errno));
		return (NULL);
	 }

   strcpy (str,stmt->args->string);

   return (str);
}

static int get_user_id (uid_t *uid,const char *configfile,stmt_t *stmt)
{
   if ((stmt = find_statement (stmt,"user")) == NULL)
	 {
		log_printf (LOG_ERROR,"%s: No ``user'' statement found!\n",configfile);
		return (-1);
	 }

   if (stmt->type == TOK_STRING && stmt->n == 1)
	 {
		struct passwd *pw;

		if ((pw = getpwnam (stmt->args->string)) == NULL)
		  {
			 log_printf (LOG_ERROR,"%s: Couldn't convert user name %s to a uid: %s\n",configfile,stmt->args->string,strerror (errno));
			 return (-1);
		  }

		*uid = pw->pw_uid;
	 }
   else if (stmt->type == TOK_INTEGER && stmt->n == 1)
	 {
		if (stmt->args->integer < 0)
		  {
			 log_printf (LOG_ERROR,"%s: ``user'' value out of range!\n",configfile);
			 return (-1);
		  }
		*uid = stmt->args->integer;
	 }
   else
	 {
		log_printf (LOG_ERROR,"%s: ``user'' value is not a string, nor an integer!\n",configfile);
		return (-1);
	 }

   return (0);
}

static int get_group_id (gid_t *gid,const char *configfile,stmt_t *stmt)
{
   if ((stmt = find_statement (stmt,"group")) == NULL)
	 {
		log_printf (LOG_ERROR,"%s: No ``group'' statement found!\n",configfile);
		return (-1);
	 }

   if (stmt->type == TOK_STRING && stmt->n == 1)
	 {
		struct group *gr;

		if ((gr = getgrnam (stmt->args->string)) == NULL)
		  {
			 log_printf (LOG_ERROR,"%s: Couldn't convert group name %s to a gid: %s\n",configfile,stmt->args->string,strerror (errno));
			 return (-1);
		  }

		*gid = gr->gr_gid;
	 }
   else if (stmt->type == TOK_INTEGER && stmt->n == 1)
	 {
		if (stmt->args->integer < 0)
		  {
			 log_printf (LOG_ERROR,"%s: ``group'' value out of range!\n",configfile);
			 return (-1);
		  }
		*gid = stmt->args->integer;
	 }
   else
	 {
		log_printf (LOG_ERROR,"%s: ``group'' value is not a string, nor an integer!\n",configfile);
		return (-1);
	 }

   return (0);
}

static int get_timeout (uint32_t *timeout,const char *configfile,stmt_t *stmt)
{
   if ((stmt = find_statement (stmt,"timeout")) != NULL)
	 {
		if (stmt->type != TOK_INTEGER || stmt->n != 1)
		  {
			 log_printf (LOG_ERROR,"%s: ``timeout'' value is not an integer!\n",configfile);
			 return (-1);
		  }

		if (stmt->args->integer < 1 || stmt->args->integer > 4294967)
		  {
			 log_printf (LOG_ERROR,"%s: ``timeout'' value out of range!\n",configfile);
			 return (-1);
		  }

		*timeout = (uint32_t) stmt->args->integer * 1000;
	 }

   return (0);
}

static int get_retries (uint16_t *retries,const char *configfile,stmt_t *stmt)
{
   if ((stmt = find_statement (stmt,"retries")) != NULL)
	 {
		if (stmt->type != TOK_INTEGER || stmt->n != 1)
		  {
			 log_printf (LOG_ERROR,"%s: ``retries'' value is not an integer!\n",configfile);
			 return (-1);
		  }

		if (stmt->args->integer < 1 || stmt->args->integer > 65535)
		  {
			 log_printf (LOG_ERROR,"%s: ``retries'' value out of range!\n",configfile);
			 return (-1);
		  }

		*retries = (uint16_t) stmt->args->integer;
	 }

   return (0);
}

static int get_ignore_admin_down (int *ignore_admin_down,const char *configfile,stmt_t *stmt)
{
   if ((stmt = find_statement (stmt,"ignore_admin_down")) != NULL)
	 {
		if (stmt->type != TOK_BOOLEAN || stmt->n != 1)
		  {
			 log_printf (LOG_ERROR,"%s: ``ignore_admin_down'' is not a boolean value!\n",configfile);
			 return (-1);
		  }

		*ignore_admin_down = stmt->args->boolean;
	 }

   return (0);
}

static int get_router (config_router_t *router,const char *configfile,section_t *section,uint32_t timeout,uint16_t retries,int ignore_admin_down)
{
   stmt_t *stmt;

   if ((router->hostname = get_string (configfile,section->stmt,"host")) == NULL)
	 {
		error0:
		log_printf (LOG_ERROR,"%s: Parse error in router section %s!\n",configfile,section->name);
		return (-1);
	 }

   if ((router->community = get_string (configfile,section->stmt,"community")) == NULL)
	 {
		mem_free (router->hostname);
		goto error0;
	 }

   if (get_port (&router->service,configfile,section->stmt) < 0)
	 {
		mem_free (router->hostname);
		mem_free (router->community);
		goto error0;
	 }

   router->timeout = router->retries = 0;
   router->ignore_admin_down = -1;

   if (get_timeout (&router->timeout,configfile,section->stmt) < 0 ||
	   get_retries (&router->retries,configfile,section->stmt) < 0 ||
	   get_ignore_admin_down (&router->ignore_admin_down,configfile,section->stmt) < 0)
	 {
		error1:
		mem_free (router->hostname);
		mem_free (router->community);
		if (router->service != NULL) mem_free (router->service);
		goto error0;
	 }

   if (!router->timeout) router->timeout = timeout;
   if (!router->retries) router->retries = retries;
   if (router->ignore_admin_down < 0) router->ignore_admin_down = ignore_admin_down;

   if (!router->timeout)
	 {
		log_printf (LOG_ERROR,
					"%s: No global ``timeout'' specified and ``timeout'' missing in router section %s!\n",
					configfile,section->name);
		goto error1;
	 }

   if (!router->retries)
	 {
		log_printf (LOG_ERROR,
					"%s: No global ``retries'' specified and ``retries'' missing in router section %s!\n",
					configfile,section->name);
		goto error1;
	 }

   if (router->ignore_admin_down < 0)
	 {
		log_printf (LOG_ERROR,
					"%s: No global ``ignore_admin_down'' specified and ``ignore_admin_down'' missing in router section %s!\n",
					configfile,section->name);
		goto error1;
	 }

   router->n_ignore = 0;

   if ((stmt = find_statement (section->stmt,"ignore")) != NULL)
	 {
		router->n_ignore = stmt->n;
		if (router->n_ignore)
		  {
			 int i;

			 if (stmt->type != TOK_STRING)
			   {
				  log_printf (LOG_ERROR,"%s: ``ignore'' value is not a string array!\n",configfile);
				  router->n_ignore = 0;
				  goto error1;
			   }

			 if ((router->ignore = (char **) mem_alloc (sizeof (char *) * router->n_ignore)) == NULL)
			   {
				  error2:
				  log_printf (LOG_ERROR,"Couldn't allocate memory for temporary storage: %s\n",strerror (errno));
				  router->n_ignore = 0;
				  goto error1;
			   }

			 for (i = 0; i < router->n_ignore; i++)
			   {
				  if ((router->ignore[i] = (char *) mem_alloc (sizeof (char) * (strlen (stmt->args[i].string) + 1))) == NULL)
					{
					   int j,saved = errno;
					   for (j = 0; j < i; j++) mem_free (router->ignore[j]);
					   mem_free (router->ignore);
					   errno = saved;
					   goto error2;
					}
				  strcpy (router->ignore[i],stmt->args[i].string);
			   }
		  }
	 }

   router->n_order = 0;

   if ((stmt = find_statement (section->stmt,"order")) != NULL)
	 {
		router->n_order = stmt->n;
		if (router->n_order)
		  {
			 int i,j,saved = EINVAL;

			 if (stmt->type != TOK_STRING)
			   {
				  log_printf (LOG_ERROR,"%s: ``order'' value is not a string array!\n",configfile);
				  error4:
				  for (i = 0; i < router->n_ignore; i++) mem_free (router->ignore[i]);
				  mem_free (router->ignore);
				  router->n_order = router->n_ignore = 0;
				  goto error1;
			   }

			 if ((router->order = (char **) mem_alloc (sizeof (char *) * router->n_order)) == NULL)
			   {
				  saved = errno;
				  error3:
				  for (i = 0; i < router->n_ignore; i++) mem_free (router->ignore[i]);
				  mem_free (router->ignore);
				  router->n_order = 0;
				  errno = saved;
				  goto error2;
			   }

			 for (i = 0; i < router->n_order; i++)
			   {
				  if ((router->order[i] = (char *) mem_alloc (sizeof (char) * (strlen (stmt->args[i].string) + 1))) == NULL)
					{
					   saved = errno;
					   for (j = 0; j < i; j++) mem_free (router->order[j]);
					   mem_free (router->order);
					   router->n_order = 0;
					   goto error3;
					}
				  strcpy (router->order[i],stmt->args[i].string);
			   }
		  }
	 }

   router->n_alias = 0;

   if ((stmt = find_statement (section->stmt,"alias")) != NULL)
	 {
		router->n_alias = stmt->n;
		if (router->n_alias)
		  {
			 int i,j,saved;

			 if (stmt->type != TOK_STRING)
			   {
				  log_printf (LOG_ERROR,"%s: ``alias'' value is not a string array!\n",configfile);
				  for (i = 0; i < router->n_order; i++) mem_free (router->order[i]);
				  mem_free (router->order);
				  router->n_alias = 0;
				  goto error4;
			   }

			 if ((router->alias = (char **) mem_alloc (sizeof (char *) * router->n_alias)) == NULL)
			   {
				  saved = errno;
				  error5:
				  for (i = 0; i < router->n_order; i++) mem_free (router->order[i]);
				  mem_free (router->order);
				  router->n_alias = 0;
				  goto error3;
			   }

			 for (i = 0; i < router->n_alias; i++)
			   {
				  if ((router->alias[i] = (char *) mem_alloc (sizeof (char) * (strlen (stmt->args[i].string) + 1))) == NULL)
					{
					   saved = errno;
					   for (j = 0; j < i; j++) mem_free (router->alias[j]);
					   mem_free (router->alias);
					   router->n_alias = 0;
					   goto error5;
					}
				  strcpy (router->alias[i],stmt->args[i].string);
			   }
		  }
	 }

   return (0);
}

/*
 * parse configuration file. return 0 if successful, -1 if some
 * error occurred. the function automatically logs all errors.
 */
int config_parse (config_t *config,const char *filename)
{
   section_t *section;
   uint32_t timeout = 0;
   uint16_t retries = 0;
   int ignore_admin_down = -1;
   int i;

   if ((section = parse (filename)) == NULL)
	 return (-1);

   log_printf (LOG_VERBOSE,"Analyzing configuration file data\n");

   if (!section->n)
	 {
		log_printf (LOG_ERROR,"%s: No router sections found!\n",filename);
		parse_destroy (&section);
		return (-1);
	 }

   config->n = section->n;

   if ((config->database = get_string (filename,section->stmt,"database")) == NULL)
	 {
		parse_destroy (&section);
		return (-1);
	 }

   if ((config->htdocs = get_string (filename,section->stmt,"htdocs")) == NULL)
	 {
		parse_destroy (&section);
		mem_free (config->database);
		return (-1);
	 }

   if (get_string_optional (&config->documentroot,filename,section->stmt,"documentroot") < 0)
	 {
		parse_destroy (&section);
		mem_free (config->database);
		mem_free (config->htdocs);
		return (-1);
	 }

   if (config->documentroot == NULL)
	 {
		if ((config->documentroot = (char *) mem_alloc (sizeof (char))) == NULL)
		  {
			 log_printf (LOG_ERROR,"Unable to allocate memory for temporary storage: %s!\n",strerror (errno));
			 parse_destroy (&section);
			 mem_free (config->database);
			 mem_free (config->htdocs);
			 return (-1);
		  }
		config->documentroot[0] = '\0';
	 }

   if ((config->pidfile = get_string (filename,section->stmt,"pidfile")) == NULL)
	 {
		parse_destroy (&section);
		mem_free (config->database);
		mem_free (config->htdocs);
		mem_free (config->documentroot);
		return (-1);
	 }

   if (get_timeout (&timeout,filename,section->stmt) < 0 ||
	   get_retries (&retries,filename,section->stmt) < 0 ||
	   get_ignore_admin_down (&ignore_admin_down,filename,section->stmt) < 0 ||
	   get_user_id (&config->uid,filename,section->stmt) < 0 ||
	   get_group_id (&config->gid,filename,section->stmt) < 0)
	 {
		parse_destroy (&section);
		mem_free (config->database);
		mem_free (config->htdocs);
		mem_free (config->documentroot);
		mem_free (config->pidfile);
		return (-1);
	 }

   if ((config->router = (config_router_t *) mem_alloc (sizeof (config_router_t) * config->n)) == NULL)
	 {
		log_printf (LOG_ERROR,"Couldn't allocate memory for temporary storage: %s\n",strerror (errno));
		parse_destroy (&section);
		mem_free (config->database);
		mem_free (config->htdocs);
		mem_free (config->documentroot);
		mem_free (config->pidfile);
		return (-1);
	 }

   for (i = 0; i < config->n; i++)
	 if (get_router (&config->router[i],filename,section->child[i],timeout,retries,ignore_admin_down) < 0)
	   {
		  parse_destroy (&section);
		  config->n = i;
		  config_destroy (config);
		  return (-1);
	   }

   parse_destroy (&section);

   return (0);
}

