/*
 *
 * Copyright (C) 2004 Mekensleep
 *
 *	Mekensleep
 *	24 rue vieille du temple
 *	75004 Paris
 *       licensing@mekensleep.com
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * Authors:
 *  Cedric PINSON <cpinson@freesheep.org>
 *  Loic Dachary <loic@gnu.org>
 *
 */

#ifdef WIN32
#  include <windows.h>
#endif // WIN32

#include <cal3d/cal3d.h>

#include <osgCal/SubMeshSoftware>

#include <iostream>

using namespace osgCal;

SubMeshSoftware::SubMeshSoftware() :
  _supportsPrimitiveFunctor(true),
  _invisible(false),
  _calSubmesh(0),
  _calRenderer(0)
{
}

SubMeshSoftware::SubMeshSoftware(CalSubmesh* calSubmesh, CalRenderer* calRenderer) :
  _supportsPrimitiveFunctor(true),
  _invisible(false),
  _calSubmesh(calSubmesh),
  _calRenderer(calRenderer)
{
  setSupportsDisplayList(false);
}

SubMeshSoftware::SubMeshSoftware(const SubMeshSoftware& submesh, const osg::CopyOp& copyop) :
  osg::Geometry(submesh, copyop),
  _supportsPrimitiveFunctor(submesh._supportsPrimitiveFunctor),
  _invisible(submesh._invisible),
  _calSubmesh(submesh._calSubmesh),
  _calRenderer(submesh._calRenderer)
{
}

SubMeshSoftware::~SubMeshSoftware() {}

void SubMeshSoftware::update(void)
{
	CalPhysique * pPhysique = _calRenderer->m_pModel->getPhysique();

	// get bone vector of the skeleton
	std::vector<CalBone *>& vectorBone = pPhysique->m_pModel->getSkeleton()->getVectorBone();

	// get vertex vector of the core submesh
	std::vector<CalCoreSubmesh::Vertex>& vectorVertex = _calSubmesh->getCoreSubmesh()->getVectorVertex();

	//
	// get vertex buffer pointer
	float * pVertexBuffer = (float*)(dynamic_cast<osg::Vec3Array*>(getVertexArray())->getDataPointer());

	//
	// get normal buffer pointer
	float * pNormalBuffer = NULL;
	if(!_invisible)
	{
		pNormalBuffer = (float*)(dynamic_cast<osg::Vec3Array*>(getNormalArray())->getDataPointer());
	}

	memset(pVertexBuffer, 0, _calSubmesh->getVertexCount() * 3 * sizeof(float));
	if (NULL != pNormalBuffer)
	{
		memset(pNormalBuffer, 0, _calSubmesh->getVertexCount() * 3 * sizeof(float));
	}

	// calculate all submesh vertices
	for (int vertexId = 0; vertexId < _calSubmesh->getVertexCount() ; ++vertexId)
	{
		// get the vertex
		CalCoreSubmesh::Vertex& vertex = vectorVertex[vertexId];

		// blend together all vertex influences
		for(unsigned int influenceId = 0; influenceId < vertex.vectorInfluence.size(); ++influenceId)
		{
			// get the influence
			const CalCoreSubmesh::Influence & influence = vertex.vectorInfluence[influenceId];

			// get the bone of the influence vertex
			CalBone * pBone = vectorBone[influence.boneId];

			// transform vertex with current state of the bone
			const CalMatrix & tm = pBone->getTransformMatrix();
			const CalVector & trans = pBone->getTranslationBoneSpace();

			const float & vposx = vertex.position.x;
			const float & vposy = vertex.position.y;
			const float & vposz = vertex.position.z;

			pVertexBuffer[0] += (tm.dxdx * vposx + tm.dxdy * vposy + tm.dxdz * vposz + trans.x) * influence.weight;
			pVertexBuffer[1] += (tm.dydx * vposx + tm.dydy * vposy + tm.dydz * vposz + trans.y) * influence.weight;
			pVertexBuffer[2] += (tm.dzdx * vposx + tm.dzdy * vposy + tm.dzdz * vposz + trans.z) * influence.weight;

			if (NULL != pNormalBuffer)
			{
				// transform normal with current state of the bone
				const float & vnx = vertex.normal.x;
				const float & vny = vertex.normal.y;
				const float & vnz = vertex.normal.z;
				pNormalBuffer[0] += (tm.dxdx * vnx + tm.dxdy * vny + tm.dxdz * vnz) * influence.weight;
				pNormalBuffer[1] += (tm.dydx * vnx + tm.dydy * vny + tm.dydz * vnz) * influence.weight;
				pNormalBuffer[2] += (tm.dzdx * vnx + tm.dzdy * vny + tm.dzdz * vnz) * influence.weight;
			}
		}

		// save vertex position
		pVertexBuffer += 3;

		if (NULL != pNormalBuffer)
		{
			pNormalBuffer += 3;
		}
	}
	dirtyBound();
	dirtyDisplayList();
}

void SubMeshSoftware::drawImplementation(osg::State& state) const 
{
  if(!_invisible) Geometry::drawImplementation(state);
}

bool SubMeshSoftware::computeBound() const
{
  if(_supportsPrimitiveFunctor) {
    return Geometry::computeBound();
  } else {
    _bbox=_staticbbox;
    _bbox_computed = true;
    return true;
  }
}

void SubMeshSoftware::create(void)
{
  _calRenderer->m_pSelectedSubmesh = _calSubmesh;

  // vertexes
  {
    int count = _calRenderer->getVertexCount();
    osg::Vec3Array* array = new osg::Vec3Array(count);
    _calRenderer->getVertices((float*)array->getDataPointer());
    setVertexArray(array);
  }

  // normals
  if(!_invisible) {
    int count = _calRenderer->getVertexCount();
    osg::Vec3Array* array = new osg::Vec3Array(count);
    _calRenderer->getNormals((float*)array->getDataPointer());
    setNormalArray(array);
    setNormalBinding(osg::Geometry::BIND_PER_VERTEX);
  }

  // texture coordinates
  if(!_invisible) {
    int count = _calRenderer->getVertexCount();
    for(int mapId = 0; mapId < _calRenderer->getMapCount(); mapId++) {
      osg::Vec2Array* array = new osg::Vec2Array(count);
      _calRenderer->getTextureCoordinates(mapId, (float*)array->getDataPointer());
      setTexCoordArray(mapId, array);
    }
  }

  // faces
  {
    int count = _calRenderer->getFaceCount();
    osg::PrimitiveSet* primitive;
    if(sizeof(CalIndex) == sizeof(GLushort)) {
      osg::DrawElementsUShort* primitiveUShort = new osg::DrawElementsUShort(osg::PrimitiveSet::TRIANGLES, count*3);
      primitive = primitiveUShort;
      _calRenderer->getFaces((CalIndex*)&(primitiveUShort->front()));
    } else {
      osg::DrawElementsUInt* primitiveUInt = new osg::DrawElementsUInt(osg::PrimitiveSet::TRIANGLES, count*3);
      primitive = primitiveUInt;
      _calRenderer->getFaces((CalIndex*)&(primitiveUInt->front()));
    }
    addPrimitiveSet(primitive);
  }
}

