//
//
//****************************************************************************************//
// Copyright (c) 2002-2010, The MITRE Corporation
// All rights reserved.
//
// Redistribution and use in source and binary forms, with or without modification, are
// permitted provided that the following conditions are met:
//
//     * Redistributions of source code must retain the above copyright notice, this list
//       of conditions and the following disclaimer.
//     * Redistributions in binary form must reproduce the above copyright notice, this 
//       list of conditions and the following disclaimer in the documentation and/or other
//       materials provided with the distribution.
//     * Neither the name of The MITRE Corporation nor the names of its contributors may be
//       used to endorse or promote products derived from this software without specific 
//       prior written permission.
//
// THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY 
// EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
// OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT 
// SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
// SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT
// OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
// HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR
// TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
// EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
//
//****************************************************************************************//

#ifndef OVALDATACOLLECTOR_H
#define OVALDATACOLLECTOR_H

//	other includes
#include <string>
#include <vector>
#include <iostream>
#include <stdlib.h>

//	include common classes
#include "DocumentManager.h"
#include "Common.h"
#include "Version.h"
#include "AbsObjectCollector.h"
#include "CollectedObject.h"
#include "AbsVariable.h"

//	namespaces
XERCES_CPP_NAMESPACE_USE

class AbsObjectCollector; // forward declaration needed to allow successful compilation

/**
	This class acts as a base class for all data collectors. Doing so provides some common
	functionality to all child classes and simplifies the creation of a new data collector. 
	Collectors are family specific. That is each time a new family is added to the OVAL
	schema a new collector class should be created to support it. Each family specific
	collector is responsible for making calls to the probes used to gather data for that
	families tests. Look at the Run() method in the DataCollector class for guidance.
*/
class AbsDataCollector {

public:
	AbsDataCollector();
	virtual ~AbsDataCollector();

	/** Return a reference to the singleton datacollector instance. */
	static AbsDataCollector* Instance();
	/** Add the specified schmema location to the document root. 
		Ensure that the specified schema location is not already added.
	*/
	void AddSchemaLocation(std::string newSchemaLocation);
	/**	Add the specified xmlns to the document root. 
		Ensure that the specified xmlns is not already added.	
	*/
	void AddXmlns(std::string newXmlnsAlias, std::string newXmlnsUri);
	/** Loop through all objects in the provided oval definitions document.
		Get the object reference from the test and call the object collector.
	*/
	void Run();

	/** Return a reference to the collected_objects element in the sc document. */
	DOMElement* GetSCCollectedObjectsElm();
	/** Return a reference to the system data element in the sc document. */
	DOMElement* GetSCSystemDataElm();

	/** Initilaize the base AbsDataCollector.
		This function intialized the oval system characteristics document that is generated by the data collector.
	*/
	void InitBase(AbsObjectCollector*);

	/** Return true if the data collector is running. */
	static bool GetIsRunning();

protected:

	/** Write the generator element to the oval system characteristics document. 
		Note that the values are all hard coded except for the time stamp value.
	*/
	void WriteGenerator();
	/** Write the system info element to the oval system characteristics document.
		Each concrete data collector implements this method since different api calls are utilized to fetchthe data.
	*/
	virtual void WriteSystemInfo() = 0;

	DOMElement* collectedObjectsElm;
	
	DOMElement* systemDataElm;

	AbsObjectCollector *objectCollector;

	/** The singleton instance of a concrete DataCollector in the application. */
	static AbsDataCollector* instance;
	/** A flag used to indicate that the concrete DataCollector is running. 
		The flag is used by the ObjectComponent class so that it can determine whether to a concrete data collector is running or not. 
	*/
	static bool isRunning;
};

/** 
	This class represents an Exception that occured while running the Data collector.
*/
class AbsDataCollectorException : public Exception {
public:
	AbsDataCollectorException(std::string errMsgIn = "", int severity = ERROR_FATAL, Exception *ex = NULL);
	~AbsDataCollectorException();
};

#endif
