///////////////////////////////////////////////////////////////////////////////
//
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

#include <core/Core.h>
#include <core/actions/ActionManager.h>
#include <core/actions/ApplyModifierAction.h>
#include <core/plugins/PluginManager.h>
#include <core/scene/objects/Modifier.h>
#include "stdactions/StdActions.h"

namespace Core {

/// The singleton instance of this class.
ActionManager* ActionManager::_singletonInstance = NULL;

/******************************************************************************
* Returns the proxy for the action with the given ID.
******************************************************************************/
ActionProxy* ActionManager::findActionProxy(const QString& actionId)
{
	return findChild<ActionProxy*>(actionId);
}

/******************************************************************************
* Returns a list of all registered action proxies.
******************************************************************************/
QList<ActionProxy*> ActionManager::getActionProxies()
{
	return findChildren<ActionProxy*>();
}

/******************************************************************************
* Returns an action with the given ID.
******************************************************************************/
Action::SmartPtr ActionManager::getAction(const QString& actionId)
{
	ActionProxy* proxy = findActionProxy(actionId);
	if(!proxy) return NULL;
	return proxy->action();
}

/******************************************************************************
* Invokes the command action with the given ID.
******************************************************************************/
void ActionManager::invokeAction(const QString& actionId)
{
	ActionProxy* proxy = findActionProxy(actionId);
	if(!proxy) throw Exception(tr("Action with id '%1' is not defined.").arg(actionId));
	proxy->trigger();
}

/******************************************************************************
* Registers a new action proxy with the ActionManager.
******************************************************************************/
void ActionManager::addAction(ActionProxy* actionProxy)
{
	CHECK_POINTER(actionProxy);
	OVITO_ASSERT_MSG(findActionProxy(actionProxy->objectName()) == NULL, "ActionManager::addAction", "There is already an action with the same ID.");

	// Make it a child of this manager.
	actionProxy->setParent(this);
}

/******************************************************************************
* Registers a new action with the ActionManager.
******************************************************************************/
ActionProxy* ActionManager::addAction(const Action::SmartPtr& action)
{
	CHECK_OBJECT_POINTER(action);
	ActionProxy* proxy = new ActionProxy(action);
	addAction(proxy);
	return proxy;
}

/******************************************************************************
* Creates and registers the standard actions with the action manager.
******************************************************************************/
void ActionManager::registerStandardActions()
{
	static FileActionsHandler fileActions;
	static AnimationActionsHandler animationActions;
	static ViewportActionsHandler viewportActions;
	static EditingActionsHandler editingActions;
	static RenderingActionsHandler renderingActions;
	static OptionsActionsHandler optionsActions;
	addAction(new XFormSystemChooser());

	// Register ApplyModifierActions for all installed modifier classes.
	//Q_FOREACH(PluginClassDescriptor* modifierClass, PLUGIN_MANAGER.listClasses(PLUGINCLASSINFO(Modifier)))
	//	addAction(new ApplyModifierAction(modifierClass));
}

};
