
/*
 * Copyright (C) 2002-2003 Stefan Holst
 * Copyright (C) 2004-2005 Maximilian Schwerin
 *
 * This file is part of oxine a free media player.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * $Id: mediamarks.c 1372 2006-10-16 10:39:45Z mschwerin $
 *
 */
#include "config.h"

#include <errno.h>
#include <fcntl.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <sys/stat.h>
#include <sys/types.h>

#include "codeset.h"
#include "environment.h"
#include "filelist.h"
#include "heap.h"
#include "i18n.h"
#include "logger.h"
#include "mediamarks.h"
#include "xmlparser.h"

static void
read_entry (const char *xml_mrl, filelist_t * filelist, xml_node_t * node)
{
    char *title = NULL;
    char *mrl = NULL;
    char *description = NULL;
    char *thumbnail_mrl = NULL;
    fileitem_type_t type = FILE_TYPE_UNKNOWN;
    int listened_count = 0;

    char *codeset = get_system_encoding ();
    recode_t *xr = recode_init ("utf8", codeset);

    while (node) {
        if (strcasecmp (node->name, "title") == 0) {
            title = recode (xr, node->data);
        }

        else if (strcasecmp (node->name, "description") == 0) {
            description = recode (xr, node->data);
        }

        else if (strcasecmp (node->name, "thumbnail") == 0) {
            thumbnail_mrl =
                recode (xr, xml_parser_get_property (node, "href"));
        }

        else if (strcasecmp (node->name, "mrl") == 0) {
            mrl = recode (xr, xml_parser_get_property (node, "href"));

            char *entry_type = xml_parser_get_property (node, "type");
            if (entry_type) {
                if (strcasecmp (entry_type, "regular") == 0) {
                    type = FILE_TYPE_REGULAR;
                }

                else if (strcasecmp (entry_type, "directory") == 0) {
                    type = FILE_TYPE_DIRECTORY;
                }

                else if (strcasecmp (entry_type, "mountpoint") == 0) {
                    type = FILE_TYPE_MOUNTPOINT;
                }

                else if (strcasecmp (entry_type, "mediamarks") == 0) {
                    type = FILE_TYPE_MEDIAMARKS;
                }

                else if (strcasecmp (entry_type, "playlist_m3u") == 0) {
                    type = FILE_TYPE_M3U;
                }
            }
        }

        else if (strcasecmp (node->name, "listened_count") == 0) {
            listened_count = atoi (node->data);
        }

        else {
            error (_("Unknown name '%s' for node!"), node->name);
        }

        node = node->next;
    }

    recode_done (xr);
    ho_free (codeset);

    if (mrl) {
        fileitem_t *fileitem = filelist_add (filelist, title, mrl, type);
        fileitem->listened_count = listened_count;
        if (description)
            fileitem->description = ho_strdup (description);
        if (thumbnail_mrl)
            fileitem->thumbnail_mrl = ho_strdup (thumbnail_mrl);
    } else {
        error (_("Found an entry that did not contain a MRL!"));
    }

    ho_free (mrl);
    ho_free (title);
    ho_free (description);
    ho_free (thumbnail_mrl);
}


static void
read_nodes (const char *xml_mrl, filelist_t * filelist, xml_node_t * node)
{
    while (node) {
        if (!strcasecmp (node->name, "entry")) {
            read_entry (xml_mrl, filelist, node->child);
        }

        else {
            error (_("Unknown name '%s' for node!"), node->name);
        }
        node = node->next;
    }
}


static int
_mediamarks_read (filelist_t * filelist, bool sort, const char *rootnode)
{
    int size;
    char *file = read_entire_file (filelist->mrl, &size);
    xml_node_t *node;

    if (!file)
        return 0;

    xml_parser_init (file, strlen (file), XML_PARSER_CASE_INSENSITIVE);

    if (xml_parser_build_tree (&node) < 0) {
        error (_("Parsing '%s' failed!"), filelist->mrl);
        ho_free (file);
        return 0;
    }

    if (strcasecmp (node->name, rootnode)) {
        error (_("Root node of '%s' must be '%s'!"), filelist->mrl, rootnode);
        xml_parser_free_tree (node);
        ho_free (file);
        return 0;
    }

    char *str_sort = xml_parser_get_property (node, "sort");
    if (str_sort && (strcasecmp (str_sort, "false") == 0))
        sort = false;

    read_nodes (filelist->mrl, filelist, node->child);
    xml_parser_free_tree (node);
    ho_free (file);

    if (sort)
        filelist_sort (filelist, NULL);

    return 1;
}


int
mediamarks_read (filelist_t * filelist)
{
    return _mediamarks_read (filelist, "true", "oxine_mediamarks");
}


int
favorites_read (filelist_t * filelist)
{
    return _mediamarks_read (filelist, "false", "oxine_favorites");
}
