
/*
 * Copyright (C) 2004-2005 Maximilian Schwerin
 *
 * This file is part of oxine a free media player.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * $Id: settings_menu.c 1571 2006-11-19 17:54:55Z mschwerin $
 *
 */
#include "config.h"

#include <assert.h>
#include <stdio.h>
#include <xine.h>

#include "environment.h"
#include "gui_utils.h"
#include "heap.h"
#include "i18n.h"
#include "logger.h"
#include "main_menu.h"
#include "oxine.h"
#include "settings_menu.h"

#define NAME_X          20
#define NAME_W          420
#define EDIT_X          (NAME_X + NAME_W + 10)
#define EDIT_W          (780 - EDIT_X)

#define TOP_Y           100
#define SPACING_Y       60
#define NUM_PER_PAGE    (425 / SPACING_Y)

extern oxine_t *oxine;

static char *cur_page_first_key = NULL;
static char *cur_page_last_key = NULL;

static l_list_t *cur_page_entries = NULL;

#define SETTINGS_BUTTONS_NUM 2
static otk_widget_t *settings_menu_buttons[SETTINGS_BUTTONS_NUM];

static bool changes_to_save = false;
static bool focus_next_page = false;
static bool multipage_section = false;

static char *section = NULL;
static char *sections[] = {
    "gui",
#ifdef HAVE_WEATHER
    "weather",
#endif
#ifdef HAVE_VDR
    "vdr",
#endif
    "playlist",
    "shutdown",
    "visual_anim",
    "audio",
    "video",
    "media",
    "decoder",
    "subtitles",
    "effects",
    "engine",
    "misc",
    NULL
};

static void settings_menu_gettext (void);
static void show_settings_section_menu_cb (void *oxine_p);

static void
free_config_dups (void *entry_p)
{
    xine_cfg_entry_t *entry = (xine_cfg_entry_t *) entry_p;
    ho_free (entry);
}


static xine_cfg_entry_t *
ho_config_dup (xine_cfg_entry_t * entry)
{
    xine_cfg_entry_t *entry_cpy = ho_new (xine_cfg_entry_t);
    memcpy (entry_cpy, entry, sizeof (xine_cfg_entry_t));

    if (!cur_page_entries)
        cur_page_entries = l_list_new ();
    l_list_append (cur_page_entries, entry_cpy);

    return entry_cpy;
}


static void
config_update_entry (xine_cfg_entry_t * entry)
{
    xine_config_update_entry (oxine->xine, entry);
    changes_to_save = true;
}


static void
display_config_description (oxine_t * oxine, xine_cfg_entry_t * entry, int y)
{
    otk_label_new (oxine->otk, NAME_X, y + 20, NAME_W,
                   OTK_ALIGN_LEFT | OTK_ALIGN_VCENTER, entry->description);
}


static void
config_string_change_cb (void *entry_p, char *text)
{
    xine_cfg_entry_t *entry = (xine_cfg_entry_t *) entry_p;

    if (!text)
        return;

    entry->str_value = text;
    config_update_entry (entry);
    ho_free (text);
}


static void
display_config_string (oxine_t * oxine, xine_cfg_entry_t * entry, int y)
{
    display_config_description (oxine, entry, y);

    xine_cfg_entry_t *entry_cpy = ho_config_dup (entry);
    otk_widget_t *editbox = otk_editbox_new (oxine->otk, EDIT_X, y,
                                             EDIT_W, 40, 1024,
                                             config_string_change_cb,
                                             entry_cpy);
    otk_editbox_set_text (editbox, entry->str_value);
}


static int
config_range_get_cb (void *entry_p)
{
    xine_cfg_entry_t *entry = (xine_cfg_entry_t *) entry_p;

    return entry->num_value;
}


static int
config_range_change_cb (void *entry_p, int value)
{
    xine_cfg_entry_t *entry = (xine_cfg_entry_t *) entry_p;

    entry->num_value = value;
    if (entry->num_value > entry->range_max)
        entry->num_value = entry->range_max;
    if (entry->num_value < entry->range_min)
        entry->num_value = entry->range_min;

    config_update_entry (entry);

    return value;
}


static void
display_config_range (oxine_t * oxine, xine_cfg_entry_t * entry, int y)
{
    display_config_description (oxine, entry, y);

    xine_cfg_entry_t *entry_cpy = ho_config_dup (entry);

    char *format = "%d";
    if ((entry->range_min == 0) && (entry->range_max == 100))
        format = "%d%%";

    otk_widget_t *s;
    s = otk_slider_new (oxine->otk, EDIT_X, y, EDIT_W, 40,
                        5, 36, true, OTK_SLIDER_HORIZONTAL, format,
                        true, true, entry->num_default,
                        entry->range_min, entry->range_max, 1,
                        config_range_get_cb, entry_cpy,
                        config_range_change_cb, entry_cpy);
    otk_widget_set_update (s, true);
}


static int
config_enum_change_cb (void *entry_p, int num_current)
{
    xine_cfg_entry_t *entry = (xine_cfg_entry_t *) entry_p;

    entry->num_value = num_current;
    config_update_entry (entry);

    return num_current;
}


static void
display_config_enum (oxine_t * oxine, xine_cfg_entry_t * entry, int y)
{
    display_config_description (oxine, entry, y);

    xine_cfg_entry_t *entry_cpy = ho_config_dup (entry);
    otk_widget_t *list = otk_selector_new (oxine->otk, EDIT_X, y, EDIT_W,
                                           config_enum_change_cb, entry_cpy);
    int i = 0;
    char **enum_values = entry->enum_values;
    while (enum_values[i]) {
        otk_listentry_new (list, _(enum_values[i]), NULL, NULL,
                           NULL, NULL, NULL, NULL);
        i++;
    }
    otk_list_set_pos (list, entry->num_value);
}


static void
config_num_change_cb (void *entry_p, char *text)
{
    xine_cfg_entry_t *entry = (xine_cfg_entry_t *) entry_p;

    if (!text)
        return;

    entry->num_value = atoi (text);
    config_update_entry (entry);

    ho_free (text);
}


static void
display_config_num (oxine_t * oxine, xine_cfg_entry_t * entry, int y)
{
    display_config_description (oxine, entry, y);

    char str[10];
    snprintf (str, 10, "%d", entry->num_value);
    xine_cfg_entry_t *entry_cpy = ho_config_dup (entry);
    otk_widget_t *editbox = otk_editbox_new (oxine->otk, EDIT_X, y,
                                             EDIT_W, 40, 100,
                                             config_num_change_cb, entry_cpy);
    otk_editbox_set_text (editbox, str);
}


static void
config_bool_change_cb (void *entry_p)
{
    xine_cfg_entry_t *entry = (xine_cfg_entry_t *) entry_p;

    entry->num_value = (entry->num_value + 1) % 2;
    config_update_entry (entry);
}


static void
display_config_bool (oxine_t * oxine, xine_cfg_entry_t * entry, int y)
{
    xine_cfg_entry_t *entry_cpy = ho_config_dup (entry);
    otk_checkbox_new (oxine->otk, NAME_X, y + 5, 30, entry->num_value,
                      config_bool_change_cb, entry_cpy);

    otk_label_new (oxine->otk, NAME_X + 40, y + 20, 780 - NAME_X - 40,
                   OTK_ALIGN_LEFT | OTK_ALIGN_VCENTER, entry->description);
}


static void
save_settings (void)
{
    if (changes_to_save) {
        xine_config_save (oxine->xine, get_file_config ());
        changes_to_save = false;
    }
}


static void
save_settings_cb (void *oxine_p)
{
    if (changes_to_save)
        show_message_dialog (show_settings_menu_cb, oxine, NULL, NULL,
                             DIALOG_OK, OXINE_BACKGROUNDS "/settingsmenu.png",
                             _("It may be necessary to restart oxine\n"
                               "for the changes to take effect!"));
    else
        show_settings_menu_cb (oxine);

    save_settings ();
}


static void
change_section_cb (void *section_p)
{
    if (cur_page_entries)
        l_list_free (cur_page_entries, free_config_dups);
    cur_page_entries = NULL;

    multipage_section = false;
    focus_next_page = false;

    section = (char *) section_p;
    ho_free (cur_page_first_key);
    cur_page_first_key = NULL;
    ho_free (cur_page_last_key);
    cur_page_last_key = NULL;

    show_settings_section_menu_cb (oxine);
}


static void
change_page_cb (void *section_p)
{
    if (cur_page_entries)
        l_list_free (cur_page_entries, free_config_dups);
    cur_page_entries = NULL;

    focus_next_page = true;

    ho_free (cur_page_first_key);
    cur_page_first_key = ho_strdup (cur_page_last_key);
    ho_free (cur_page_last_key);
    cur_page_last_key = NULL;

    show_settings_section_menu_cb (oxine);
}


static bool
is_from_correct_section (xine_cfg_entry_t * entry)
{
    return (strncmp (entry->key, section, strlen (section)) == 0);
}


static bool
is_ok_to_show (xine_cfg_entry_t * entry)
{
    if (strcmp (entry->key, "shutdown.last") == 0)
        return false;
    else if (strcmp (entry->key, "media.dvb.last_channel") == 0)
        return false;

    return true;
}


static void
settings_menu_event_handler (void *oxine_p, oxine_event_t * event)
{
    oxine_t *oxine = (oxine_t *) oxine_p;

    if ((oxine->current_menu == show_settings_menu_cb)
        || (oxine->current_menu == show_settings_section_menu_cb)) {
        if (event->type == OXINE_EVENT_KEY) {
            switch (event->source.key) {
            case OXINE_KEY_BACK:
                if (oxine->backto_menu == show_settings_section_menu_cb)
                    save_settings_cb (oxine);
                else
                    show_main_menu_cb (oxine);
                event->source.key = OXINE_KEY_NULL;
            case OXINE_KEY_MENU_MAIN:
            case OXINE_KEY_PLAY_DVB:
            case OXINE_KEY_PLAY_V4L:
            case OXINE_KEY_MENU_VIDEO:
            case OXINE_KEY_MENU_MUSIC:
            case OXINE_KEY_QUIT:
            case OXINE_KEY_SHUTDOWN:
                save_settings ();
                settings_menu_free ();
                break;
            default:
                break;
            }
        }
    }
}


static void
show_settings_section_menu_cb (void *oxine_p)
{
    hide_user_interface (oxine);

    oxine->current_menu = show_settings_section_menu_cb;
    oxine->backto_menu = show_settings_section_menu_cb;

    create_new_window (false, true);

    int err = 0;
    xine_cfg_entry_t entry;
    if (cur_page_first_key)
        err = xine_config_lookup_entry (oxine->xine, cur_page_first_key,
                                        &entry);
    else
        err = xine_config_get_first_entry (oxine->xine, &entry);

    int num = 0;
    while ((err || num == 0) && (num < NUM_PER_PAGE)) {
        if (entry.description && is_from_correct_section (&entry)
            && is_ok_to_show (&entry)) {

            if (!cur_page_first_key)
                cur_page_first_key = ho_strdup (entry.key);

            int y = TOP_Y + num * SPACING_Y;
            switch (entry.type) {
            case XINE_CONFIG_TYPE_STRING:
                display_config_string (oxine, &entry, y);
                break;
            case XINE_CONFIG_TYPE_RANGE:
                display_config_range (oxine, &entry, y);
                break;
            case XINE_CONFIG_TYPE_ENUM:
                display_config_enum (oxine, &entry, y);
                break;
            case XINE_CONFIG_TYPE_NUM:
                display_config_num (oxine, &entry, y);
                break;
            case XINE_CONFIG_TYPE_BOOL:
                display_config_bool (oxine, &entry, y);
                break;
            }
            num += 1;
        }
        err = xine_config_get_next_entry (oxine->xine, &entry);
    }
    ho_free (cur_page_last_key);
    cur_page_last_key = ho_strdup (entry.key);

    if (num >= NUM_PER_PAGE) {
        multipage_section = true;
    }

    int x = 20;
    int y = 545;

    settings_menu_buttons[0] =
        otk_text_button_new (oxine->otk, x, y, 180, 35,
                             _("Next Page"), change_page_cb, section);
    otk_widget_set_alignment (settings_menu_buttons[0], OTK_ALIGN_CENTER);
    x += 190;
    settings_menu_buttons[1] =
        otk_text_button_new (oxine->otk, x, y, 180, 35,
                             _("Back"), save_settings_cb, oxine);
    otk_widget_set_alignment (settings_menu_buttons[1], OTK_ALIGN_CENTER);

    otk_widget_set_enabled (settings_menu_buttons[0], multipage_section);
    if (multipage_section && focus_next_page)
        otk_widget_set_focus (settings_menu_buttons[0]);
    else
        otk_widget_set_focus (settings_menu_buttons[1]);

    show_user_interface (oxine);
}


void
show_settings_menu_cb (void *oxine_p)
{
    hide_user_interface (oxine);

    oxine->current_menu = show_settings_menu_cb;
    oxine->backto_menu = show_settings_menu_cb;

    show_menu_background (OXINE_BACKGROUNDS "/settingsmenu.png");

    odk_add_event_handler (oxine->odk, settings_menu_event_handler, oxine,
                           EVENT_HANDLER_PRIORITY_NORMAL);

    static otk_widget_t *settings_menu_list = NULL;
    static otk_widget_t *settings_menu_window = NULL;
    if (settings_menu_window) {
        otk_window_set_current (oxine->otk, settings_menu_window);
        otk_list_select_all (settings_menu_list);
        show_user_interface (oxine);
        return;
    }
    settings_menu_window = create_new_window (true, true);

    char *section_names[] = {
        _("User Interface"),
#ifdef HAVE_WEATHER
        _("Weather"),
#endif
#ifdef HAVE_VDR
        _("Video Disc Recorder"),
#endif
        _("Playlist"),
        _("Shutdown"),
        _("Audio Visualization"),
        _("Audio"),
        _("Video"),
        _("Media"),
        _("Decoder"),
        _("Subtitles"),
        _("Effects"),
        _("Engine"),
        _("Miscellaneous")
    };

    settings_menu_list = otk_list_new (oxine->otk, 215, 150, 370, 400,
                                       45, 55, false, false,
                                       OTK_LIST_MULTIPLE_SELECTION, oxine);
    int i = 0;
    for (; sections[i]; i++) {
        otk_widget_t *e =
            otk_listentry_new (settings_menu_list, section_names[i],
                               change_section_cb, sections[i],
                               change_section_cb, sections[i],
                               NULL, NULL);
        otk_widget_set_font (e, "sans", 30);
        otk_widget_set_alignment (e, OTK_ALIGN_CENTER | OTK_ALIGN_VCENTER);
    }
    {
        otk_widget_t *e =
            otk_listentry_new (settings_menu_list, _("Mainmenu"),
                               show_main_menu_cb, oxine,
                               show_main_menu_cb, oxine,
                               NULL, NULL);
        otk_widget_set_font (e, "sans", 30);
        otk_widget_set_alignment (e, OTK_ALIGN_CENTER | OTK_ALIGN_VCENTER);
    }
    otk_list_select_all (settings_menu_list);

    show_user_interface (oxine);
}


void
settings_menu_free (void)
{
    if (cur_page_entries)
        l_list_free (cur_page_entries, free_config_dups);
    cur_page_entries = NULL;

    ho_free (cur_page_first_key);
    cur_page_first_key = NULL;
    ho_free (cur_page_last_key);
    cur_page_last_key = NULL;

    if (false)
        settings_menu_gettext ();
}


/* Here are all the strings used in enumerations. They are here so gettext can
 * find them. This is necessary as they are not translated by xine-lib which
 * is rather sad :-(. To update these run: 
 * grep -B 1 "# {" ~/.oxine/config | grep -v "^--$" | sed -e 's/, default.*$//' -e 's/# { /_("/' -e 's/  /");_("/g' -e 's/ }$/");/' | sed -e 's/^# /    \/\/ /' >> settings_menu.c
 */
static void
settings_menu_gettext (void)
{
    // palette (foreground-border-background) to use for subtitles and OSD
    _("white-black-transparent");
    _("white-none-transparent");
    _("white-none-translucid");
    _("yellow-black-transparent");
    // speaker arrangement
    _("Mono 1.0");
    _("Stereo 2.0");
    _("Headphones 2.0");
    _("Stereo 2.1");
    _("Surround 3.0");
    _("Surround 4.0");
    _("Surround 4.1");
    _("Surround 5.0");
    _("Surround 5.1");
    _("Surround 6.0");
    _("Surround 6.1");
    _("Surround 7.1");
    _("Pass Through");
    // method to sync audio and video
    _("metronom feedback");
    _("resample");
    // enable resampling
    _("auto");
    _("off");
    _("on");
    // OpenGL renderer
    _("2D_Tex_Fragprog");
    _("2D_Tex");
    _("2D_Tex_Tiled");
    _("Image_Pipeline");
    _("Cylinder");
    _("Env_Mapped_Torus");
    // play mode when title/chapter is given
    _("entire dvd");
    _("one chapter");
    // unit for seeking
    _("seek in program chain");
    _("seek in program");
    // unit for the skip action
    _("skip program");
    _("skip part");
    _("skip title");
    // network bandwidth
    _("14.4 Kbps (Modem)");
    _("19.2 Kbps (Modem)");
    _("28.8 Kbps (Modem)");
    _("33.6 Kbps (Modem)");
    _("34.4 Kbps (Modem)");
    _("57.6 Kbps (Modem)");
    _("115.2 Kbps (ISDN)");
    _("262.2 Kbps (Cable/DSL)");
    _("393.2 Kbps (Cable/DSL)");
    _("524.3 Kbps (Cable/DSL)");
    _("1.5 Mbps (T1)");
    _("10.5 Mbps (LAN)");
    // MMS protocol
    _("auto");
    _("TCP");
    _("HTTP");
    // VCD default type to use on autoplay
    _("MPEG track");
    _("entry");
    _("segment");
    _("playback-control item");
    // VCD position slider range
    _("auto");
    _("track");
    _("entry");
    // subtitle size
    _("tiny");
    _("small");
    _("normal");
    _("large");
    _("very large");
    _("huge");
    // colorspace conversion method
    _("Fast but not photorealistic");
    _("Slow but looks better");
    // media format detection strategy
    _("default");
    _("reverse");
    _("content");
    _("extension");
    // memcopy method used by xine
    _("probe");
    _("libc");
    _("kernel");
    _("mmx");
    _("mmxext");
    _("sse");
    // audio visualization type
    _("Background Image");
    _("Background Animation");
    _("Post Plugin");
    // post plugin
    _("oscope");
    _("fftscope");
    _("fftgraph");
    _("goom");
}
