
/*
 * Copyright (C) 2004-2005 Maximilian Schwerin
 *
 * This file is part of oxine a free media player.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * $Id: event.c 2661 2007-08-22 15:08:37Z mschwerin $
 *
 */
#include "config.h"

#include <assert.h>
#include <stdlib.h>

#include "environment.h"
#include "event.h"
#include "heap.h"
#include "i18n.h"
#include "logger.h"
#include "oxine.h"
#include "utils.h"
#include "scheduler.h"

#include "menu_base.h"
#include "menu_help.h"
#include "menu_filelist.h"
#include "menu_main.h"
#include "menu_playback.h"
#include "menu_playlist.h"
#include "menu_settings.h"
#include "menu_weather.h"

extern oxine_t *oxine;

static void
do_not_restart_playlist (void)
{
    if (oxine->current_playlist == oxine->rw_playlist) {
        playlist_set_current (oxine->rw_playlist, NULL);
    }
}


static void
show_menu_playback_job (void *p)
{
    if (odk_current_is_playback_mode (oxine->odk)) {
        oxine_event_t ev;
        ev.type = OXINE_EVENT_KEY;
        ev.source.key = OXINE_KEY_MENU_PLAYBACK;

        odk_oxine_event_send (oxine->odk, &ev);
    }
}


/**
 * This event handler is responsible for handling DVD related keyboard events.
 *
 * @return                      If the keyboard event is consumed by this
 *                              function TRUE is returned.
 */
static bool
oxine_dvd_key_handler (oxine_t * oxine, oxine_event_t * ev)
{
    if (ev->source.key == OXINE_KEY_NULL) {
        return true;
    }
    else if (!odk_current_is_dvd (oxine->odk)) {
        return false;
    }

    /* we backup the key */
    oxine_key_id_t key_save = ev->source.key;
    /* and consume this event */
    ev->source.key = OXINE_KEY_NULL;

    switch (key_save) {
    case OXINE_KEY_PREV:
        /* This is necessary because we don't automatically forward events to
         * oxine when the GUI is visible. */
        if (!odk_get_forward_events_to_xine (oxine->odk)
            && odk_current_has_chapters (oxine->odk)) {
            odk_xine_event_send (oxine->odk, true, XINE_EVENT_INPUT_PREVIOUS);
        }
        break;
    case OXINE_KEY_NEXT:
        /* This is necessary because we don't automatically forward events to
         * oxine when the GUI is visible. */
        if (!odk_get_forward_events_to_xine (oxine->odk)
            && odk_current_has_chapters (oxine->odk)) {
            odk_xine_event_send (oxine->odk, true, XINE_EVENT_INPUT_NEXT);
        }
        break;
    default:
        /* As this was not an event we want to handle, we restore the original
         * key code. */
        ev->source.key = key_save;
        break;
    }

    return (ev->source.key == OXINE_KEY_NULL);
}


#ifdef HAVE_VDR

/**
 * This event handler is responsible for handling VDR related keyboard events.
 *
 * @return                      If the keyboard event is consumed by this
 *                              function TRUE is returned.
 */
static bool
oxine_vdr_key_handler (oxine_t * oxine, oxine_event_t * ev)
{
    if (ev->source.key == OXINE_KEY_NULL) {
        return true;
    }
    else if (!odk_current_is_vdr (oxine->odk)) {
        return false;
    }

    /* we backup the key */
    oxine_key_id_t key_save = ev->source.key;
    /* and consume this event */
    ev->source.key = OXINE_KEY_NULL;

    switch (key_save) {
    case OXINE_KEY_ACTIVATE:
    case OXINE_KEY_MENU_HELP:
    case OXINE_KEY_TELEVISION:
    case OXINE_KEY_SKIP:
    case OXINE_KEY_SEEK:
    case OXINE_KEY_SPEED:
    case OXINE_KEY_PLAYMODE:
    case OXINE_KEY_SPU_CHANNEL:
    case OXINE_KEY_SPU_OFFSET:
    case OXINE_KEY_AUDIO_OFFSET:
    case OXINE_KEY_ASPECT_RATIO:
    case OXINE_KEY_EJECT:
        /* These keys are ignored when in VDR mode. */
        break;
    case OXINE_KEY_SELECT:
    case OXINE_KEY_PPLAY:
        odk_xine_event_send (oxine->odk, false, XINE_EVENT_VDR_PAUSE);
        break;
    case OXINE_KEY_PLAY:
        odk_xine_event_send (oxine->odk, false, XINE_EVENT_VDR_PLAY);
        break;
    case OXINE_KEY_PAUSE:
        odk_xine_event_send (oxine->odk, false, XINE_EVENT_VDR_PAUSE);
        break;
    case OXINE_KEY_STOP:
        odk_xine_event_send (oxine->odk, false, XINE_EVENT_VDR_STOP);
        break;
    case OXINE_KEY_FASTRWD:
        odk_xine_event_send (oxine->odk, false, XINE_EVENT_VDR_FASTREW);
        break;
    case OXINE_KEY_FASTFWD:
        odk_xine_event_send (oxine->odk, false, XINE_EVENT_VDR_FASTFWD);
        break;
    case OXINE_KEY_PREV:
        odk_xine_event_send (oxine->odk, false, XINE_EVENT_VDR_CHANNELMINUS);
        break;
    case OXINE_KEY_NEXT:
        odk_xine_event_send (oxine->odk, false, XINE_EVENT_VDR_CHANNELPLUS);
        break;
    case OXINE_KEY_BACK:
        odk_xine_event_send (oxine->odk, false, XINE_EVENT_VDR_BACK);
        break;
    case OXINE_KEY_VOLUME:
        if (ev->how > 0) {
            odk_xine_event_send (oxine->odk, false, XINE_EVENT_VDR_VOLPLUS);
        }
        else {
            odk_xine_event_send (oxine->odk, false, XINE_EVENT_VDR_VOLMINUS);
        }
        break;
    case OXINE_KEY_VOLUME_MUTE:
        odk_change_stream_param (oxine->odk,
                                 ODK_PARAM_AO_VOLUME_MUTE, 1,
                                 ODK_AO_VOLUME_MUTE_OFF,
                                 ODK_AO_VOLUME_MUTE_ON);
        break;
    case OXINE_KEY_MENU_PLAYBACK:
    case OXINE_KEY_SHOW_OSD:
        odk_xine_event_send (oxine->odk, false, XINE_EVENT_INPUT_MENU1);
        break;
    case OXINE_KEY_AUDIO_CHANNEL:
        odk_xine_event_send (oxine->odk, false, XINE_EVENT_VDR_AUDIO);
        break;
    case OXINE_KEY_DEINTERLACE:
        odk_change_stream_param (oxine->odk, ODK_PARAM_VO_DEINTERLACE,
                                 1, 0, 1);
        break;
    case OXINE_KEY_MENU_MAIN:
    case OXINE_KEY_MENU_VIDEO:
    case OXINE_KEY_MENU_IMAGE:
    case OXINE_KEY_MENU_MUSIC:
    case OXINE_KEY_MENU_WEATHER:
    case OXINE_KEY_MENU_SETTINGS:
    case OXINE_KEY_MENU_PLAYLIST:
    case OXINE_KEY_QUIT:
    case OXINE_KEY_SHUTDOWN:
        /* As we do not want to use the oxine OSD over VDR, we always stop TV
         * playback before showing our menus. */
        odk_xine_event_send (oxine->odk, false, XINE_EVENT_VDR_STOP);
        odk_stop_stream (oxine->odk);
    default:
        /* As this was not an event we want to pass to VDR, we restore the
         * original key code. */
        ev->source.key = key_save;
        break;
    }

    return (ev->source.key == OXINE_KEY_NULL);
}
#endif


/**
 * This event handler is responsible for handling stream speed related
 * keyboard events.
 *
 * @return                      If the keyboard event is consumed by this
 *                              function TRUE is returned.
 */
static bool
oxine_stream_speed_key_handler (oxine_t * oxine, oxine_event_t * ev)
{
    if (ev->source.key == OXINE_KEY_NULL) {
        return true;
    }
    else if (!odk_current_is_playback_mode (oxine->odk)) {
        return false;
    }

    int cur_speed = odk_get_stream_param (oxine->odk, ODK_PARAM_SPEED);
    int new_speed = cur_speed;

    if (is_current_menu (hide_user_interface)
        || is_current_menu (show_menu_playback)
        || is_current_menu (show_playback_controls)
        || is_current_menu (show_playback_info)) {
        switch (ev->source.key) {
        case OXINE_KEY_SELECT:
            if (cur_speed == ODK_SPEED_PAUSE) {
                new_speed = ODK_SPEED_NORMAL;
            }
            else {
                new_speed = ODK_SPEED_PAUSE;
            }
            break;
        case OXINE_KEY_SPEED:
            if (ev->how > 0) {
                if (cur_speed < ODK_SPEED_FAST_4) {
                    new_speed <<= 1;
                }
            }
            else {
                if (cur_speed > ODK_SPEED_SLOW_4) {
                    new_speed >>= 1;
                }
            }
            break;
        default:
            break;
        }
    }

    {
        switch (ev->source.key) {
        case OXINE_KEY_PPLAY:
            if (cur_speed == ODK_SPEED_PAUSE) {
                new_speed = ODK_SPEED_NORMAL;
            }
            else {
                new_speed = ODK_SPEED_PAUSE;
            }
            break;
        case OXINE_KEY_PLAY:
            new_speed = ODK_SPEED_NORMAL;
            break;
        case OXINE_KEY_PAUSE:
            new_speed = ODK_SPEED_PAUSE;
            break;
        default:
            break;
        }
    }

    if (cur_speed != new_speed) {
        odk_set_stream_param (oxine->odk, ODK_PARAM_SPEED, new_speed);
        show_stream_parameter (ODK_PARAM_SPEED);
    }

    return (ev->source.key == OXINE_KEY_NULL);
}


/**
 * This event handler is responsible for handling image playback related
 * keyboard events.
 *
 * @return                      If the keyboard event is consumed by this
 *                              function TRUE is returned.
 */
#ifdef HAVE_IMAGE_ROTATION
static bool
oxine_image_key_handler (oxine_t * oxine, oxine_event_t * ev)
{
    if (ev->source.key == OXINE_KEY_NULL) {
        return true;
    }
    else if (!odk_current_is_playback_mode (oxine->odk)) {
        return false;
    }
    else if (!odk_current_is_image (oxine->odk)) {
        return false;
    }

    /* we backup the key */
    oxine_key_id_t key_save = ev->source.key;
    /* and consume this event */
    ev->source.key = OXINE_KEY_NULL;

    /* As we cannot overlay an OSD over the images we have to stop
     * playback and return to logo mode for the menus. */
    switch (key_save) {
    case OXINE_KEY_RED:
        ev->how = -90;
    case OXINE_KEY_ROTATE_LEFT:
        if (ev->how > 0) {
            ev->how *= -1;
        }
        odk_rotate_current_image (oxine->odk, ev->how);
        break;
    case OXINE_KEY_BLUE:
        ev->how = +90;
    case OXINE_KEY_ROTATE_RIGHT:
        if (ev->how < 0) {
            ev->how *= -1;
        }
        odk_rotate_current_image (oxine->odk, ev->how);
        break;
    default:
        /* As this was not an event we want to handle, we restore the original
         * key code. */
        ev->source.key = key_save;
        break;
    }

    return (ev->source.key == OXINE_KEY_NULL);
}
#endif /* HAVE_IMAGE_ROTATION */


/**
 * This event handler is responsible for handling playback related keyboard
 * events.
 *
 * @return                      If the keyboard event is consumed by this
 *                              function TRUE is returned.
 */
static bool
oxine_playback_key_handler (oxine_t * oxine, oxine_event_t * ev)
{
    if (ev->source.key == OXINE_KEY_NULL) {
        return true;
    }
    else if (!odk_current_is_playback_mode (oxine->odk)) {
        return false;
    }
#ifdef HAVE_IMAGE_ROTATION
    else if (oxine_image_key_handler (oxine, ev)) {
        return true;
    }
#endif
#ifdef HAVE_VDR
    else if (oxine_vdr_key_handler (oxine, ev)) {
        return true;
    }
#endif
    else if (oxine_dvd_key_handler (oxine, ev)) {
        return true;
    }
    else if (oxine_stream_speed_key_handler (oxine, ev)) {
        return true;
    }

    switch (ev->source.key) {
    case OXINE_KEY_STOP:
        odk_stop_stream (oxine->odk);
        do_not_restart_playlist ();
        break;
    case OXINE_KEY_PREV:
        if (playlist_get_current_pos (oxine->current_playlist) > 0) {
            playlist_play_prev ();
        }
        break;
    case OXINE_KEY_NEXT:
        if (!playlist_current_is_last (oxine->current_playlist)) {
            playlist_play_next ();
        }
        break;
    case OXINE_KEY_PLAYMODE:
        switch (playlist_get_playmode (oxine->current_playlist)) {
        case PLAYBACK_MODE_NORMAL:
            playlist_set_playmode (oxine->current_playlist,
                                   PLAYBACK_MODE_REPEAT);
            break;
        case PLAYBACK_MODE_REPEAT:
            playlist_set_playmode (oxine->current_playlist,
                                   PLAYBACK_MODE_RANDOM);
            break;
        case PLAYBACK_MODE_RANDOM:
            playlist_set_playmode (oxine->current_playlist,
                                   PLAYBACK_MODE_NORMAL);
            break;
        }
    default:
        break;
    }

    switch (ev->source.key) {
    case OXINE_KEY_MENU_PLAYBACK:
        if (is_current_menu (show_menu_playback)) {
            if (!odk_current_is_audio (oxine->odk)) {
                hide_user_interface (oxine);
                set_backto_menu (hide_user_interface, NULL);
            }
        }
        else {
            show_menu_playback (oxine);
        }
        break;
    case OXINE_KEY_VOLUME:
        odk_change_stream_param (oxine->odk, ODK_PARAM_AO_VOLUME,
                                 ev->how, 0, 100);
        show_stream_parameter (ODK_PARAM_AO_VOLUME);
        break;
    case OXINE_KEY_VOLUME_MUTE:
        odk_change_stream_param (oxine->odk,
                                 ODK_PARAM_AO_VOLUME_MUTE, 1,
                                 ODK_AO_VOLUME_MUTE_OFF,
                                 ODK_AO_VOLUME_MUTE_ON);
        show_stream_parameter (ODK_PARAM_AO_VOLUME_MUTE);
        break;
    case OXINE_KEY_AUDIO_CHANNEL:
        {
            int max_channels = odk_get_stream_info (oxine->odk,
                                                    ODK_STREAM_INFO_MAX_AUDIO_CHANNEL);
            if (max_channels > 1) {
                odk_change_stream_param (oxine->odk,
                                         ODK_PARAM_AUDIO_CHANNEL,
                                         ev->how, -2, max_channels - 1);
                show_stream_parameter (ODK_PARAM_AUDIO_CHANNEL);
            }
        }
        break;
    case OXINE_KEY_AUDIO_OFFSET:
        odk_change_stream_param (oxine->odk,
                                 ODK_PARAM_AUDIO_OFFSET,
                                 ev->how, -90000, 90000);
        show_stream_parameter (ODK_PARAM_AUDIO_OFFSET);
        break;
    case OXINE_KEY_FASTFWD:
        odk_set_stream_param (oxine->odk, ODK_PARAM_SPEED, ODK_SPEED_FAST_2);
        show_stream_parameter (ODK_PARAM_SPEED);
        break;
    case OXINE_KEY_FASTRWD:
        warn ("xine-lib does not yet support fast rewind.");
        break;
    case OXINE_KEY_SKIP:
        odk_set_stream_param (oxine->odk, ODK_PARAM_SPEED, ODK_SPEED_NORMAL);
        odk_change_stream_param (oxine->odk, ODK_PARAM_POSITION,
                                 ev->how, 0, 0);
        show_stream_parameter (ODK_PARAM_POSITION);
        break;
    case OXINE_KEY_SEEK:
        odk_set_stream_param (oxine->odk, ODK_PARAM_SPEED, ODK_SPEED_NORMAL);
        odk_set_stream_param (oxine->odk, ODK_PARAM_POSITION, ev->how);
        show_stream_parameter (ODK_PARAM_POSITION);
        break;
    case OXINE_KEY_SATURATION:
        odk_change_stream_param (oxine->odk, ODK_PARAM_VO_SATURATION,
                                 ev->how, 0, 65535);
        show_stream_parameter (ODK_PARAM_VO_SATURATION);
        break;
    case OXINE_KEY_BRIGHTNESS:
        odk_change_stream_param (oxine->odk, ODK_PARAM_VO_BRIGHTNESS,
                                 ev->how, 0, 65535);
        show_stream_parameter (ODK_PARAM_VO_BRIGHTNESS);
        break;
    case OXINE_KEY_CONTRAST:
        odk_change_stream_param (oxine->odk, ODK_PARAM_VO_CONTRAST,
                                 ev->how, 0, 65535);
        show_stream_parameter (ODK_PARAM_VO_CONTRAST);
        break;
    case OXINE_KEY_HUE:
        odk_change_stream_param (oxine->odk, ODK_PARAM_VO_HUE,
                                 ev->how, 0, 65535);
        show_stream_parameter (ODK_PARAM_VO_HUE);
        break;
    default:
        break;
    }

    /* If the current stream does not have a video component we quit at this
     * point as none of the following events are relevant for streams
     * without video. */
    if (odk_current_is_audio (oxine->odk)) {
        return (ev->source.key == OXINE_KEY_NULL);
    }

    switch (ev->source.key) {
    case OXINE_KEY_SHOW_OSD:
        if (!oxine->user_interface_is_visible
            || is_current_menu (show_playback_info)
            || is_current_menu (show_playback_controls)) {
            show_menu_playback (oxine);
            ev->source.key = OXINE_KEY_NULL;
        }
        else {
            hide_user_interface (oxine);
            set_backto_menu (hide_user_interface, NULL);
            ev->source.key = OXINE_KEY_NULL;
        }
        break;
    case OXINE_KEY_BACK:
        if (!oxine->user_interface_is_visible
            || is_current_menu (show_playback_info)
            || is_current_menu (show_playback_controls)) {
            show_menu_playback (oxine);
            ev->source.key = OXINE_KEY_NULL;
        }
        else if (is_current_menu (show_menu_playback)) {
            hide_user_interface (oxine);
            set_backto_menu (hide_user_interface, NULL);
            ev->source.key = OXINE_KEY_NULL;
        }
        break;
    case OXINE_KEY_ACTIVATE:
        if (!oxine->stream_parameter_is_visible) {
            show_stream_parameter (ODK_PARAM_SPEED);
            ev->source.key = OXINE_KEY_NULL;
        }
        else {
            hide_user_interface (oxine);
            set_backto_menu (hide_user_interface, NULL);
            ev->source.key = OXINE_KEY_NULL;
        }
        break;
    default:
        break;
    }

    /* If we're currently playing an image we quit at this point as none of
     * the following events are relevant for image viewing. */
    if (odk_current_is_image (oxine->odk)) {
        return (ev->source.key == OXINE_KEY_NULL);
    }

    switch (ev->source.key) {
    case OXINE_KEY_ZOOM:
        odk_change_stream_param (oxine->odk, ODK_PARAM_VO_ZOOM_X,
                                 ev->how, 0, 400);
        odk_change_stream_param (oxine->odk, ODK_PARAM_VO_ZOOM_Y,
                                 ev->how, 0, 400);
        show_stream_parameter (ODK_PARAM_VO_ZOOM_X);
        break;
    case OXINE_KEY_ZOOM_X:
        odk_change_stream_param (oxine->odk, ODK_PARAM_VO_ZOOM_X,
                                 ev->how, 0, 400);
        show_stream_parameter (ODK_PARAM_VO_ZOOM_X);
        break;
    case OXINE_KEY_ZOOM_Y:
        odk_change_stream_param (oxine->odk, ODK_PARAM_VO_ZOOM_Y,
                                 ev->how, 0, 400);
        show_stream_parameter (ODK_PARAM_VO_ZOOM_X);
        break;
    case OXINE_KEY_ZOOM_RESET:
        odk_set_stream_param (oxine->odk, ODK_PARAM_VO_ZOOM_X, 100);
        odk_set_stream_param (oxine->odk, ODK_PARAM_VO_ZOOM_Y, 100);
        show_stream_parameter (ODK_PARAM_VO_ZOOM_X);
        break;
    case OXINE_KEY_SPU_CHANNEL:
        {
            int max_channels = odk_get_stream_info (oxine->odk,
                                                    ODK_STREAM_INFO_MAX_SPU_CHANNEL);
            if (max_channels > 1) {
                odk_change_stream_param (oxine->odk,
                                         ODK_PARAM_SPU_CHANNEL,
                                         ev->how, -2, max_channels - 1);
                show_stream_parameter (ODK_PARAM_SPU_CHANNEL);
            }
        }
        break;
    case OXINE_KEY_SPU_OFFSET:
        odk_change_stream_param (oxine->odk, ODK_PARAM_SPU_OFFSET,
                                 ev->how, -90000, 90000);
        show_stream_parameter (ODK_PARAM_SPU_OFFSET);
        break;
    case OXINE_KEY_ASPECT_RATIO:
        odk_change_stream_param (oxine->odk,
                                 ODK_PARAM_VO_ASPECT_RATIO, 1,
                                 ODK_VO_ASPECT_RATIO_AUTO,
                                 ODK_VO_ASPECT_RATIO_DVB);
        show_stream_parameter (ODK_PARAM_VO_ASPECT_RATIO);
        break;
    case OXINE_KEY_DEINTERLACE:
        odk_change_stream_param (oxine->odk,
                                 ODK_PARAM_VO_DEINTERLACE, 1,
                                 ODK_VO_DEINTERLACE_OFF,
                                 ODK_VO_DEINTERLACE_ON);
        show_stream_parameter (ODK_PARAM_VO_DEINTERLACE);
        break;
    default:
        break;
    }

    return (ev->source.key == OXINE_KEY_NULL);
}


/**
 * This event handler is responsible for handling keyboard events.
 */
static void
oxine_key_handler (oxine_t * oxine, oxine_event_t * ev)
{
    /* Evaluate all playback related keypress events. */
    if (oxine_playback_key_handler (oxine, ev)) {
        return;
    }

    switch (ev->source.key) {
    case OXINE_KEY_MENU_MAIN:
        if (is_current_menu (show_menu_main)
            && odk_current_is_playback_mode (oxine->odk)) {
            show_menu_playback (oxine);
        }
        else {
            show_menu_main (oxine);
        }
        break;
    case OXINE_KEY_MENU_MUSIC:
        if (is_current_menu (show_menu_music)
            && odk_current_is_playback_mode (oxine->odk)) {
            show_menu_playback (oxine);
        }
        else {
            show_menu_music (oxine);
        }
        break;
    case OXINE_KEY_MENU_VIDEO:
        if (is_current_menu (show_menu_video)
            && odk_current_is_playback_mode (oxine->odk)) {
            show_menu_playback (oxine);
        }
        else {
            show_menu_video (oxine);
        }
        break;
#ifdef HAVE_IMAGE_PLAYBACK
    case OXINE_KEY_MENU_IMAGE:
        if (is_current_menu (show_menu_image)
            && odk_current_is_playback_mode (oxine->odk)) {
            show_menu_playback (oxine);
        }
        else {
            show_menu_image (oxine);
        }
        break;
#endif
#ifdef HAVE_WEATHER
    case OXINE_KEY_MENU_WEATHER:
        if (is_current_menu (show_menu_weather)
            && odk_current_is_playback_mode (oxine->odk)) {
            show_menu_playback (oxine);
        }
        else {
            show_menu_weather (oxine);
        }
        break;
#endif
    case OXINE_KEY_MENU_SETTINGS:
        if (is_current_menu (show_menu_settings)
            && odk_current_is_playback_mode (oxine->odk)) {
            show_menu_playback (oxine);
        }
        else {
            show_menu_settings (oxine);
        }
        break;
    case OXINE_KEY_MENU_PLAYLIST:
        if (is_current_menu (show_menu_playlist)
            && odk_current_is_playback_mode (oxine->odk)) {
            show_menu_playback (oxine);
        }
        else {
            show_menu_playlist (oxine);
        }
        break;
    case OXINE_KEY_MENU_HELP:
        if (is_current_menu (show_menu_help)
            && odk_current_is_playback_mode (oxine->odk)) {
            show_menu_playback (oxine);
        }
        else {
            show_menu_help (oxine);
        }
        break;
    case OXINE_KEY_TELEVISION:
        odk_stop_stream (oxine->odk);
        do_not_restart_playlist ();
        play_tv_cb (oxine);
        break;
    case OXINE_KEY_FULLSCREEN:
        odk_window_set_fullscreen (oxine->odk,
                                   !odk_window_is_fullscreen (oxine->odk));
        break;
    case OXINE_KEY_BACK:
        backto_menu_cb (oxine);
        break;
    case OXINE_KEY_EJECT:
        eject_cb (oxine);
        break;
    case OXINE_KEY_QUIT:
        playlist_rw_change_cb (oxine->rw_playlist);
        odk_exit (oxine->odk);
        break;
    case OXINE_KEY_SHUTDOWN:
        playlist_rw_change_cb (oxine->rw_playlist);
        shutdown_cb (oxine);
        break;
    default:
        break;
    }
}


/**
 * This event handler is responsible for reacting to mouse button events.
 */
static void
oxine_button_handler (oxine_t * oxine, oxine_event_t * event)
{
    switch (event->source.button) {
    case OXINE_MOUSE_BUTTON_RIGHT:
    case OXINE_MOUSE_BUTTON_MIDDLE:
        {
            oxine_event_t ev;
            ev.type = OXINE_EVENT_KEY;
            ev.source.key = OXINE_KEY_SHOW_OSD;
            odk_oxine_event_send (oxine->odk, &ev);
        }
        break;
    case OXINE_MOUSE_SCROLLWHEEL_UP:
        {
            oxine_event_t ev;
            ev.type = OXINE_EVENT_KEY;
            ev.source.key = OXINE_KEY_VOLUME;
            ev.how = +5;
            odk_oxine_event_send (oxine->odk, &ev);
        }
        break;
    case OXINE_MOUSE_SCROLLWHEEL_DOWN:
        {
            oxine_event_t ev;
            ev.type = OXINE_EVENT_KEY;
            ev.source.key = OXINE_KEY_VOLUME;
            ev.how = -5;
            odk_oxine_event_send (oxine->odk, &ev);
        }
        break;
    case OXINE_MOUSE_BUTTON_6:
        {
            oxine_event_t ev;
            ev.type = OXINE_EVENT_KEY;
            ev.source.key = OXINE_KEY_PREV;
            odk_oxine_event_send (oxine->odk, &ev);
        }
        break;
    case OXINE_MOUSE_BUTTON_7:
        {
            oxine_event_t ev;
            ev.type = OXINE_EVENT_KEY;
            ev.source.key = OXINE_KEY_NEXT;
            odk_oxine_event_send (oxine->odk, &ev);
        }
        break;
    default:
        break;
    }
}


/**
 * This event handler is responsible for reacting to mouse movements.
 */
static void
oxine_motion_handler (oxine_t * oxine, oxine_event_t * event)
{
    if (odk_current_is_playback_mode (oxine->odk)
#ifdef HAVE_VDR
        && !odk_current_is_vdr (oxine->odk)
#endif
        && (is_current_menu (hide_user_interface)
            || is_current_menu (show_playback_info)
            || is_current_menu (show_playback_controls))) {

        if (!oxine->playback_controls_are_visible
            && (event->data.mouse.pos.y > 550)) {
            show_playback_controls (oxine);
        }

        if (oxine->playback_controls_are_visible
            && (event->data.mouse.pos.y < 500)) {
            hide_user_interface (oxine);
        }

        if (oxine->playback_controls_are_visible
            && (event->data.mouse.pos.y > 500)) {
            const int timeout = 60 * 1000;
            schedule_hide_gui_job (timeout);
        }
    }
}


/**
 * This is oxine's main event handler. All events generated by X, xine-lib
 * or oxine itself pass through here. 
 *
 * If an underlying level has already reacted to an event (consumed the
 * event) the event source may be OXINE_KEY_NULL or OXINE_MOUSE_BUTTON_NULL.
 * In that case the event is purely informational, no further action can be
 * taken on it.
 */
void
oxine_event_handler (void *oxine_p, oxine_event_t * event)
{
    oxine_t *oxine = (oxine_t *) oxine_p;

#ifdef DEBUG_EVENTS
    switch (event->type) {
    case OXINE_EVENT_KEY:
        debug ("got OXINE_EVENT_KEY %d", event->source.key);
        break;
    case OXINE_EVENT_BUTTON:
        debug ("got OXINE_EVENT_BUTTON %d +%d+%d", event->source.button,
               event->data.pos.x, event->data.pos.y);
        break;
    case OXINE_EVENT_MOTION:
        debug ("got OXINE_EVENT_MOTION +%d+%d", event->data.pos.x,
               event->data.pos.y);
        break;
    case OXINE_EVENT_PLAYBACK_STARTED:
        debug ("got OXINE_EVENT_PLAYBACK_STARTED");
        break;
    case OXINE_EVENT_PLAYBACK_STOPPED:
        debug ("got OXINE_EVENT_PLAYBACK_STOPPED");
        break;
    case OXINE_EVENT_PLAYBACK_FINISHED:
        debug ("got OXINE_EVENT_PLAYBACK_FINISHED");
        break;
    default:
        break;
    }
#endif

    switch (event->type) {
    case OXINE_EVENT_KEY:
        oxine_key_handler (oxine, event);
        break;
    case OXINE_EVENT_MOTION:
        oxine_motion_handler (oxine, event);
        break;
    case OXINE_EVENT_BUTTON:
        oxine_button_handler (oxine, event);
        break;
    case OXINE_EVENT_PLAYBACK_STARTED:
        /* We ignore this event if we're in logo mode. */
        if (!odk_current_is_playback_mode (oxine->odk)) {
            /* Do nothing. */
        }
        /* If the stream is a TV, DVD, or VCD stream we hide the GUI. */
        else if (odk_current_is_television (oxine->odk)
                 || odk_current_is_dvd (oxine->odk)
                 || odk_current_is_vcd (oxine->odk)) {
            hide_user_interface (oxine);
        }
        /* If it's an audio-only stream and the current menu is
         * hide_user_interface we show the playback menu. */
        else if (!odk_current_has_video (oxine->odk)
                 && is_current_menu (hide_user_interface)) {
            show_menu_playback (oxine);
        }
        /* If it's an stream with video and the current menu is
         * hide_user_interface we show the playback info. */
        else if (odk_current_has_video (oxine->odk)
                 && !odk_current_is_image (oxine->odk)
                 && is_current_menu (hide_user_interface)) {
            show_playback_info (oxine);
        }
        /* Else we repaint the current menu. */
        else {
            show_menu_current ();
        }
        break;
    case OXINE_EVENT_PLAYBACK_STOPPED:
        do_not_restart_playlist ();
        playback_ended_menu_cb (oxine);
        break;
    case OXINE_EVENT_PLAYBACK_FINISHED:
#ifdef XINE_PARAM_GAPLESS_SWITCH
        if (xine_check_version (1, 1, 1)) {
            odk_set_stream_param (oxine->odk, ODK_PARAM_GAPLESS, 1);
        }
#endif
        playlist_play_next ();
        break;
    case OXINE_EVENT_PLAYBACK_ERROR:
        show_message_dialog ((otk_cb_t) playlist_play_next, oxine,
                             (otk_cb_t) playback_ended_menu_cb, oxine,
                             DIALOG_OK_CANCEL, NULL,
                             _("Could not play title!"));
        break;
    case OXINE_EVENT_WAITING_FOR_ALTERNATIVE:
        show_message_dialog (NULL, NULL, NULL, NULL, DIALOG_PLAIN, NULL,
                             _("Waiting for alternative stream..."));
        break;
    case OXINE_EVENT_GUI_REPAINT:
        show_menu_current ();
        break;
    case OXINE_EVENT_GUI_HIDE:
        if (odk_current_is_playback_mode (oxine->odk)) {
            hide_user_interface (oxine);
        }
        break;
    default:
        break;
    }

    /* When we're in playback mode and are showing a menu that is not the
     * playback menu we start a timeout of 60 secs. after which we hide the
     * user interface if we're playing a stream with video or show the
     * playback menu if it's an audio only stream. */
    switch (event->type) {
    case OXINE_EVENT_KEY:
    case OXINE_EVENT_MOTION:
    case OXINE_EVENT_BUTTON:
        cancel_job (oxine->playback_menu_job);
        if (odk_current_is_playback_mode (oxine->odk)
            && oxine->user_interface_is_visible
            && !is_current_menu (show_menu_playback)) {
            const int timeout = 60 * 1000;
            if (odk_current_has_video (oxine->odk)) {
                schedule_hide_gui_job (timeout);
            }
            else if (odk_current_is_image (oxine->odk)) {
                schedule_hide_gui_job (timeout);
            }
            else {
                oxine->playback_menu_job =
                    schedule_job (timeout, show_menu_playback_job, NULL);
            }
        }
        break;
    default:
        break;
    }
}
