#include <qmessagebox.h>
#include <qaction.h>
#include <q3popupmenu.h>
#include <QMainWindow>
#include <qmutex.h>

#include <unistd.h>		// for getuid and geteuid
#include <sys/types.h>	//

// Tagcoll
#include <tagcoll/Expression.h>
#include <tagcoll/Filter.h>
#include <tagcoll/StdioParserInput.h>
#include <tagcoll/Exception.h>

// AptFront
#include <apt-front/init.h>
#include <apt-front/cache/cache.h>
#include <apt-front/cache/component/packagetags.h>
#include <apt-front/cache/component/tags.h>

#include "debtagsplugincontainer.h"

// NUtil
#include "helpers.h"

// NApplication
#include "applicationfactory.h"
#include "runcommand.h"

// NPlugin
#include <iprovider.h>
#include <plugincontainer.h>
#include <iprogressobserver.h>
#include "debtagsplugin.h"
#include "relatedplugin.h"
#include "debtagsactionplugin.h"
#include "debtagspluginfactory.h"

#include "debtagssettingswidget.h"

using namespace aptFront::cache::component;
using namespace aptFront::cache::entity;

extern "C" 
{ 
	NPlugin::PluginContainer* new_debtagsplugin() 
	{
		return new NPlugin::DebtagsPluginContainer;
	} 

	NPlugin::PluginInformation get_pluginInformation()
	{
		return NPlugin::PluginInformation("debtagsplugin", "2.0.4", "Benjamin Mesing");
	} 
}

/** Initialize the plugin. */
__attribute__ ((constructor)) void init() 
{
}
      
      
// __attribute__ ((destructor)) void fini() 
// {
//   /* code here is executed just before dlclose() unloads the module */
// } 


namespace NPlugin
{

DebtagsPluginContainer::DebtagsPluginContainer() :
	_pCollection(0)
{
	// this assumes, that only one instance of the DebtagsPluginContainer is created
	// (no two versions of libapt-front may be opened at any time)
	// This is ok for our purpose - though usually there should be a singleton ensuring
	// this constraint...
	aptFront::init();
	DebtagsPluginFactory::getInstance()->setContainer(this);
	_pCommand = 0;
	_pRelatedPlugin = 0;
	_pDebtagsPlugin = 0;
	_pDebtagsActionPlugin = 0;
	_pSettingsWidget = 0;
	addPlugin("DebtagsPlugin");
	addPlugin("RelatedPlugin");
	addPlugin("DebtagsActionPlugin");
	
	_debtagsEnabled=false;
	
}
 
DebtagsPluginContainer::~DebtagsPluginContainer()
{
	delete _pCommand;
	qDebug("Closing global aptFront Cache");
	aptFront::cache::Global::get().close();
}

/////////////////////////////////////////////////////
// PluginContainer Interface
/////////////////////////////////////////////////////

bool DebtagsPluginContainer::init(IProvider* pProvider)
{
	BasePluginContainer::init(pProvider, DebtagsPluginFactory::getInstance());
	// use dynamic cast here because of the virtual base class 
	// (static_cast is not allowed there)
	updateDebtags();
	_pRelatedPlugin = dynamic_cast<RelatedPlugin*>(requestPlugin("RelatedPlugin"));
	_pDebtagsPlugin = dynamic_cast<DebtagsPlugin*>(requestPlugin("DebtagsPlugin"));
	_pDebtagsActionPlugin = dynamic_cast<DebtagsActionPlugin*>(requestPlugin("DebtagsActionPlugin"));
	
	connect( _pDebtagsActionPlugin->qDebtagsUpdateAction(), SIGNAL(triggered(bool)), SLOT(onDebtagsUpdate()) );

	return debtagsEnabled();
}

vector< pair<QString, QAction*> > DebtagsPluginContainer::actions()
{
	vector< pair<QString, QAction*> > result;
	return result;
}

QWidget* DebtagsPluginContainer::getSettingsWidget(QWidget* pParent)
{
	_pSettingsWidget = new DebtagsSettingsWidget(_hiddenFacets, pParent, "DebtagsSettingsWidget");
 	return _pSettingsWidget;
}

void DebtagsPluginContainer::applySettings()
{
	assert(_pSettingsWidget);
	_hiddenFacets = _pSettingsWidget->hiddenFacets();
	updateVocabulary();
}



/////////////////////////////////////////////////////
// BasePluginContainer Interface
/////////////////////////////////////////////////////


QDomElement DebtagsPluginContainer::loadContainerSettings(const QDomElement source)
{
	if (source.tagName() != "ContainerSettings")
		return source;
	float settingsVersion;
	NXml::getAttribute(source, settingsVersion, "settingsVersion", 0.0f);
	
	QDomNodeList hiddenFacets = source.elementsByTagName("HiddenFacet");
	for (int i=0; i < hiddenFacets.count(); ++i)
	{
		string hiddenFacet = toString( hiddenFacets.item(i).toElement().text() );
		_hiddenFacets.insert(hiddenFacet);
	}
	if (debtagsEnabled())
		updateVocabulary();
	return NXml::getNextElement(source);
}

void DebtagsPluginContainer::saveContainerSettings(NXml::XmlData& outData, QDomElement parent) const
{
	QDomElement containerElement = outData.addElement(parent, "ContainerSettings");
	outData.addAttribute(containerElement, 0.1f, "settingsVersion");
	for (set<string>::const_iterator it = _hiddenFacets.begin(); it != _hiddenFacets.end(); ++it)
	{
		QDomElement hiddenFacetElement = outData.addElement(containerElement, "HiddenFacet");
		outData.addText(hiddenFacetElement, *it);
	}
}





/////////////////////////////////////////////////////
// Helper Methods
/////////////////////////////////////////////////////

void DebtagsPluginContainer::onDebtagsUpdate()
{
	_pDebtagsActionPlugin->qDebtagsUpdateAction()->setEnabled(false);
	// this will fetch us the update of the db
	NApplication::ApplicationFactory fac;
	_pCommand = fac.getRunCommand("DebtagsUpdateProcess");
	connect(_pCommand, SIGNAL(quit()), SLOT(onDebtagsUpdateFinished()) );
	_pCommand->addArgument("/usr/bin/debtags");
	_pCommand->addArgument("update");
	
	try 
	{
		if ( !_pCommand->startAsRoot() )
		{
			provider()->reportError( tr("Command not executed"), tr("For an unknwon reason, the command could "
				"not be executed.") );
			delete _pCommand;
			_pCommand = 0;
			_pDebtagsActionPlugin->qDebtagsUpdateAction()->setEnabled(true);
		}
	}
	catch (const NException::RuntimeException& e)
	{
		provider()->reportError(tr("Command not executed"), toQString(e.description()));
		delete _pCommand;
		_pCommand = 0;
		_pDebtagsActionPlugin->qDebtagsUpdateAction()->setEnabled(true);
	}
}

void DebtagsPluginContainer::onDebtagsUpdateFinished()
{
	if (_pCommand->processExitedSuccessful())
	{
		updateDebtags();
	}
	delete _pCommand;
	_pCommand = 0;
	_pDebtagsActionPlugin->qDebtagsUpdateAction()->setEnabled(true);
}


void DebtagsPluginContainer::updateVocabulary(bool informPlugins)
{
	Tags &tags = aptFront::cache::Global::get().tags();

	Tagcoll::OpSet<Facet> hidden;
	for (set<string>::const_iterator i = _hiddenFacets.begin();
		i != _hiddenFacets.end(); i++)
	{
		Facet f = tags.facetByName(*i);
		if (f.valid())
			hidden += f;
	}
	
	_facets = tags.facets() - hidden;

	if (_pDebtagsPlugin)
		_pDebtagsPlugin->debtagsDataChanged();
	if (_pRelatedPlugin)
		_pRelatedPlugin->debtagsDataChanged();
}

void DebtagsPluginContainer::updateDebtags()
{
	try 
	{
		aptFront::cache::Global::get().close();
		aptFront::cache::Global::get().open(aptFront::cache::Cache::OpenDefault
				| aptFront::cache::Cache::OpenReadOnly
				| aptFront::cache::Cache::OpenTags
				| aptFront::cache::Cache::OpenDebtags);
	}
	catch (Tagcoll::FileException e)
	{
		setDebtagsEnabled(false);	// disable the debtags system
		provider()->reportError(
			tr("Tag Database Not Available" ),
			tr(
				"<p>The tag database is not available and the debtags plugin was disabled!</p>"
				"<p>"
				"You must execute <tt><b>debtags update</b></tt> as root on the commandline to download "
				"the database. If debtags is not on your system you can install it via "
				"<tt><b>apt-get install debtags</b></tt><br>"
				"After this you can enable the debtags plugin via the plugin menu -> Control Plugins."
				"</p>"
				"<p>"
				"The original error message reported by debtags was:<br>"
				"<tt>"
			) +
			toQString(e.desc()) +
			"</tt>"
			"</p>"
		);
		return;
	}
	catch (aptFront::exception::Error e)
	{
		provider()->reportError(
			tr("Unknown Error when updateing the Debtags database" ),
			tr("An unknown error occured when updating the debtags database.<br>"
				"The error reported by debtags is:.<br><tt>") +
			toQString(e.what()) +
			"</tt>"
		);
		return;
	}

	PackageTags &packagetags = aptFront::cache::Global::get().debtags();
	if ( !PackageTags::hasTagDatabase() )	// not all debtags files are existent
	{
		provider()->reportError(
			tr("Tag Database Not Available" ),
			tr(
				"<p>The tag database is not available and the  debtags plugin was disabled!</p>"
				"<p>"
				"You must execute <tt><b>debtags update</b></tt> as root on the commandline to download "
				"the database. If debtags is not on your system you can install it via "
				"<tt><b>apt-get install debtags</b></tt><br>"
				"After this you can enable the debtags plugin via the plugin menu -> Control Plugins."
				"</p>"
			) 
		);
		/* New text
						tr(
						"<p>The tag database is not available!</p>"
						"The database is required "
						"by this program. Do you want to try downloading it now?<br>"
						"Note: Super user priviledges are needed for this. You can also update by hand "
						"executing <tt>debtags update</tt> on the commandline.<br>"
						"If debtags is not on your system you can get it via <tt>apt-get install debtags</tt>."
						"</p>"
					),
					QMessageBox::Yes, QMessageBox::No )
				== 0
		*/
		setDebtagsEnabled(false);	// disable the debtags system
	}
	else 
		setDebtagsEnabled(true);
	delete _pCollection;
	_pCollection = 0;
	_facets.clear();
	if (debtagsEnabled())
	{
		try
		{
		
			/// @todo TODO implement reloading of the database here, something like
			/// aptFront::cache::Global::get().reopen might help.
			// this is set to 0 if no listener is available
			NUtil::IProgressObserver* pProgressObserver = provider()->progressObserver();
			if (pProgressObserver)
			{
				pProgressObserver->setText("reading tag database");
			}

			if (pProgressObserver)
			{
				pProgressObserver->setProgress(25);
			}
			_pCollection = & (aptFront::cache::Global::get().debtags().tagdb());
			if (pProgressObserver)
			{
				pProgressObserver->setProgress(75);
				pProgressObserver->setText("reading vocabulary");
			}
			// load the new vocabulary
			updateVocabulary(false);
			if (pProgressObserver)
			{
				pProgressObserver->setProgress(100);
			}
		}	
		catch (Tagcoll::ContextException& e)	// PackageDBException e
			// this might be a FileException, a ParserException or a PackageDBException
		{
			provider()->reportError(
				tr("Loading failed"),
				QObject::tr(
				"<p>Loading the debtags database failed!</p>"
				"<p>Most likely it is malformed.<br>"
				"Try selecting \"System\"->\"Debtags Update\" from the menu or running "
				"<tt>debtags update</tt> on the commandline.</p>"
				"This was the original error message:<br>") 
				+ toQString(e.desc())
			);
		}
	}
	// inform the plugins about the new collections
	if (_pDebtagsPlugin)
		_pDebtagsPlugin->debtagsDataChanged();
	if (_pRelatedPlugin)
		_pRelatedPlugin->debtagsDataChanged();
}

void DebtagsPluginContainer::setDebtagsEnabled(bool enabled)
{
	_debtagsEnabled = enabled;
}


}	// namespace NPlugin

