#include "aptpackagesearch.h"

#include <QRegularExpression>

#include "ipackagedb.h"
#include "iprogressobserver.h"

namespace NApt {

AptPackageSearch::AptPackageSearch(IPackageDB* pPackageDB) :
    _pPackageDB(pPackageDB)
{
    /*
    cout << " ====================== " << endl;
     // testing regexp
    cout << "matches1.1:" << NApt::AptPackageSearch::matches("abcd", "abc", Qt::CaseSensitive, true) << endl;
    cout << "matches1.2:" << NApt::AptPackageSearch::matches("abc12", "abc", Qt::CaseSensitive, true) << endl;
    cout << "matches1.3:" << NApt::AptPackageSearch::matches("abc12", "Abc", Qt::CaseSensitive, true) << endl;
    cout << "matches1.4:" << NApt::AptPackageSearch::matches("abc 123", "abc", Qt::CaseSensitive, true) << endl;
    cout << "matches1.5:" << NApt::AptPackageSearch::matches("abc12", "12", Qt::CaseSensitive, true) << endl;
    cout << "matches2.1:" << NApt::AptPackageSearch::matches("ABC12", "abc", Qt::CaseInsensitive, true) << endl;
    cout << "matches2.2:" << NApt::AptPackageSearch::matches("Abc12", "abc", Qt::CaseInsensitive, true) << endl;
    cout << "matches3.1:" << NApt::AptPackageSearch::matches("abc12", "abc", Qt::CaseInsensitive, false) << endl;
     */
}

bool AptPackageSearch::matches(
    QString findIn,
    QString searchString,
    Qt::CaseSensitivity caseSensitivity,
    bool wholeWords
) {
    if (!wholeWords) {
        return findIn.contains(searchString, caseSensitivity);
    } else {
        if (!findIn.contains(searchString, caseSensitivity)) {
            // fast return, if we can't even smartfind the string without obeying whole words condition,
            // this boosts search speed significantly
            return false;
        }
        if (caseSensitivity == Qt::CaseInsensitive) {
            return findIn.toLower().contains(QRegularExpression("^([^a-zA-Z].*|lib|)" + searchString.toLower() + "([^a-zA-Z].*|)$"));
        } else {
            return findIn.contains(QRegularExpression("^([^a-zA-Z].*|lib|)" + searchString + "([^a-zA-Z].*|)$"));
        }

    }
}

bool AptPackageSearch::search(std::set<string>& result, const QStringList& includePatterns,
                              const QStringList& excludePatterns, bool searchDescr,
                              bool caseSensitive, bool wholeWords) const {
    bool foundSomething = false;
    Qt::CaseSensitivity caseSensitivity = caseSensitive ? Qt::CaseSensitive : Qt::CaseInsensitive;
    for ( auto it = _pPackageDB->packagesBegin(); it != _pPackageDB->packagesEnd(); ++it )
    {
        IPackage* package = *it;
        QString name = package->name();
        QString shortDesc = (searchDescr) ? package->shortDescription() : "";
        QString desc = (searchDescr) ? package->description() : "";
        // holds if the search matches the package
        bool included = true;
        // check if each included pattern occurs in the package
        for (QStringList::const_iterator jt = includePatterns.begin(); jt != includePatterns.end(); ++jt)
        {
            if (matches(name, *jt, Qt::CaseInsensitive, wholeWords) ||
                matches(shortDesc, *jt, caseSensitivity, wholeWords) ||
                matches(desc, *jt, caseSensitivity, wholeWords)
            )
            {
                // continue evaluating next include pattern
                continue;
            } else {
                included = false;
                break;
            }
        }
        if (!included)
            continue;
        bool excluded = false;
        // check if each excluded pattern does not occur in the package
        for (QStringList::const_iterator jt = excludePatterns.begin(); jt != excludePatterns.end(); ++jt)
        {
            if (
                matches(name, *jt, Qt::CaseInsensitive, wholeWords) ||
                matches(shortDesc, *jt, caseSensitivity, wholeWords) ||
                matches(desc, *jt, caseSensitivity, wholeWords)
                )
            {
                excluded = true;
                break;
            }
        }
        if (included && !excluded) {
            foundSomething = true;
            result.insert(package->name().toStdString());
        }
    }
    return foundSomething;
};

void AptPackageSearch::reloadPackageInformation(NUtil::IProgressObserver* pObserver) {
    if (pObserver)
        pObserver->setProgress(100);
}

} // namespace NApt
