(*---------------------------------------------------------------------------*
  INTERFACE  cf_machine.mli

  Copyright (c) 2005, James H. Woodyatt
  All rights reserved.

  Redistribution and use in source and binary forms, with or without
  modification, are permitted provided that the following conditions
  are met:

    Redistributions of source code must retain the above copyright
    notice, this list of conditions and the following disclaimer.

    Redistributions in binary form must reproduce the above copyright
    notice, this list of conditions and the following disclaimer in
    the documentation and/or other materials provided with the
    distribution

  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
  ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
  FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE
  COPYRIGHT HOLDERS OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
  INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
  (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
  SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
  HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
  STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
  ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
  OF THE POSSIBILITY OF SUCH DAMAGE. 
 *---------------------------------------------------------------------------*)

(** Object-oriented framework for monadic composition of complex stream
    processors.
*)

(** {6 Overview}
    
    This module extends {!Cf_gadget} with an object-oriented framework.  The
    idea is to facilitate reuse of gadget monads among related processes by
    class inheritance.
*)

(** A virtual class to represent the public interface of object-oriented gadget
    states.  Use [inherit \['i, 'o\] substate] or [constrain 'self = ('i, 'o)
    #substate] to derive a class that implements part of a state in a machine.
*)
class virtual ['i, 'o] substate:
    object('self)
        (** Use [obj#next] to transition the state of the machine to [obj]. *)
        method virtual next: ('self, 'i, 'o, unit) Cf_gadget.t
    end

(** A virtual class to represent the public and private interface of object-
    oriented gadget states.  Use [inherit \['i, 'o\] state] to derive a class
    that implements a complete state in a state machine.
*)
class virtual ['i, 'o] state:
    object('self)
        inherit ['i, 'o] substate

        (** Use [obj#next] to transition the state of the machine by storing
            [obj] in the state of the gadget and applying {!Cf_gadget.guard}
            [self#guard].
        *)
        method next: ('self, 'i, 'o, unit) Cf_gadget.t
        
        (** Implement to produce a guard for receiving events on one or more
            wires.
        *)
        method private virtual guard: ('self, 'i, 'o, unit) Cf_gadget.guard_t
    end

(** Use [inherit \['i, 'o\] initial] to derive a class to represent the
    complete initial state of a machine.  While this class is not virtual, its
    [guard] method is only defined as a stub that references no gates.  It is
    provided in the base class to make the class type more useful.
*)
class ['i, 'o] initial:
    object('self)
        inherit ['i, 'o] state

        (** A stub method that produces a guard on no gates at all.  The intent
            is for subclasses to override this method as necessary.
        *)
        method private guard: ('self, 'i, 'o, unit) Cf_gadget.guard_t
        
        (** Starts a new gadget process.  Defined as {!Cf_gadget.start}
            [(Cf_gadget.guard self#guard) self].
        *)
        method start: 's. ('s, 'i, 'o, unit) Cf_gadget.t
    end

(*--- End of File [ cf_machine.mli ] ---*)
