/*
**  CWParser.m
**
**  Copyright (c) 2001-2005
**
**  Author: Ludovic Marcotte <ludovic@Sophos.ca>
**
**  This library is free software; you can redistribute it and/or
**  modify it under the terms of the GNU Lesser General Public
**  License as published by the Free Software Foundation; either
**  version 2.1 of the License, or (at your option) any later version.
**  
**  This library is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
**  Lesser General Public License for more details.
**  
**  You should have received a copy of the GNU Lesser General Public
**  License along with this library; if not, write to the Free Software
**  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
*/
#include <Pantomime/CWParser.h>

#include <Pantomime/CWConstants.h>
#include <Pantomime/CWFlags.h>
#include <Pantomime/CWInternetAddress.h>
#include <Pantomime/CWMessage.h>
#include <Pantomime/CWMIMEUtility.h>
#include <Pantomime/NSData+Extensions.h>
#include <Pantomime/NSString+Extensions.h>

#include <Pantomime/elm_defs.h>

#include <Foundation/NSTimeZone.h>


//
// private methods
//
@interface CWParser (Private)
+ (NSData *) _parameterValueUsingLine: (NSData *) theLine
                                range: (NSRange) theRange;
@end


//
//
//
@implementation CWParser

+ (void) parseContentDescription: (NSData *) theLine
                          inPart: (CWPart *) thePart
{
  NSData *aData;

  aData = [[theLine subdataFromIndex: 20] dataByTrimmingWhiteSpaces];

  if (aData && [aData length])
    {
      [thePart setContentDescription: [[aData dataFromQuotedData] asciiString] ];
    }
}


//
//
//
+ (void) parseContentDisposition: (NSData *) theLine
                          inPart: (CWPart *) thePart
{  
  if ([theLine length] > 21)
    {
      NSData *aData;
      NSRange aRange;

      aData = [theLine subdataFromIndex: 21];
      aRange = [aData rangeOfCString: ";"];
      
      if (aRange.length > 0)
	{
	  NSRange filenameRange;
	  
	  // We set the content disposition to this part
	  [thePart setContentDisposition: [[aData subdataWithRange: NSMakeRange(0, aRange.location)] asciiString]];
	  
	  // We now decode our filename
	  filenameRange = [aData rangeOfCString: "filename"];

	  if (filenameRange.length > 0)
	    {
	      [thePart setFilename: [CWMIMEUtility decodeHeader: [[CWParser _parameterValueUsingLine: aData range: filenameRange] dataFromQuotedData]
						   charset: [thePart defaultCharset]] ];
	    }
	}
      else
	{
	  [thePart setContentDisposition: [[aData dataByTrimmingWhiteSpaces] asciiString] ];
	}
    }
  else
    {
      [thePart setContentDisposition: @""];
    }
}


//
//
//
+ (void) parseContentID: (NSData *) theLine
		 inPart: (CWPart *) thePart
{
  if ([theLine length] > 12)
    {
      NSData *aData;
      
      aData = [theLine subdataFromIndex: 12];
      
      if ([aData hasCPrefix: "<"] && [aData hasCSuffix: ">"])
	{
	  [thePart setContentID: [[aData subdataWithRange: NSMakeRange(1, [aData length]-2)] asciiString]];
	}
      else
	{
	  [thePart setContentID: [aData asciiString]];
	}
    }
  else
    {
      [thePart setContentID: @""];
    }
}


//
//
//
+ (void) parseContentTransferEncoding: (NSData *) theLine
                               inPart: (CWPart *) thePart
{
  if ([theLine length] > 26)
    {
      NSData *aData;
      
      aData = [[theLine subdataFromIndex: 26] dataByTrimmingWhiteSpaces];
      
      if ([aData caseInsensitiveCCompare: "quoted-printable"] == NSOrderedSame)
	{
	  [thePart setContentTransferEncoding: PantomimeEncodingQuotedPrintable];
	}
      else if ([aData caseInsensitiveCCompare: "base64"] == NSOrderedSame)
	{
	  [thePart setContentTransferEncoding: PantomimeEncodingBase64];
	}
      else if ([aData caseInsensitiveCCompare: "8bit"] == NSOrderedSame)
	{
	  [thePart setContentTransferEncoding: PantomimeEncoding8bit];
	}
      else if ([aData caseInsensitiveCCompare: "binary"] == NSOrderedSame)
	{
	  [thePart setContentTransferEncoding: PantomimeEncodingBinary];
	}
      else
	{
	  [thePart setContentTransferEncoding: PantomimeEncodingNone];
	}
    }
  else
    {
      [thePart setContentTransferEncoding: PantomimeEncodingNone];
    }
}


//
//
//
+ (void) parseContentType: (NSData *) theLine
		   inPart: (CWPart *) thePart
{
  NSRange aRange;
  NSData *aData;
  int x;

  if ([theLine length] <= 14)
    {
      [thePart setContentType: @"text/plain"];
      return;
    }

  aData = [[theLine subdataFromIndex: 13] dataByTrimmingWhiteSpaces];

  // We first skip the parameters, if we need to
  x = [aData indexOfCharacter: ';'];
  if (x > 0)
    {
      aData = [aData subdataToIndex: x];
    } 
  
  // We see if there's a subtype specified for text, if none was specified, we append "/plain"
  x = [aData indexOfCharacter: '/'];

  if (x < 0 && [aData hasCaseInsensitiveCPrefix: "text"])
    {
      [thePart setContentType: [[[aData asciiString] stringByAppendingString: @"/plain"] lowercaseString]];
    }
  else
    {
      [thePart setContentType: [[aData asciiString] lowercaseString]];
    }

  //
  // We decode our boundary (if we need to)
  //
  aRange = [theLine rangeOfCString: "boundary"  options: NSCaseInsensitiveSearch];
  
  if (aRange.length > 0)
    {
      [thePart setBoundary: [CWParser _parameterValueUsingLine: theLine  range: aRange]];
    }

  //
  // We decode our charset (if we need to)
  //
  aRange = [theLine rangeOfCString: "charset"  options: NSCaseInsensitiveSearch];
  
  if (aRange.length > 0)
    {
      [thePart setCharset: [[CWParser _parameterValueUsingLine: theLine  range: aRange] asciiString]];
    }
  
  
  //
  // We decode our format (if we need to). See RFC2646.
  //
  aRange = [theLine rangeOfCString: "format"  options: NSCaseInsensitiveSearch];
  
  if (aRange.length > 0)
    {
      NSData *aFormat;
      
      aFormat = [CWParser _parameterValueUsingLine: theLine  range: aRange];
      
      if ([aFormat caseInsensitiveCCompare: "flowed"] == NSOrderedSame)
	{
	  [thePart setFormat: PantomimeFormatFlowed];
	}
      else
	{
	  [thePart setFormat: PantomimeFormatUnknown];
	}
    }
  else
    {
      [thePart setFormat: PantomimeFormatUnknown];
    }

  //
  // We decode the parameter "name" iif the thePart is an instance of Part
  //
  if ([thePart isKindOfClass: [CWPart class]])
  {
    aRange = [theLine rangeOfCString: "name"  options: NSCaseInsensitiveSearch];

    if (aRange.length > 0)
      {
	[thePart setFilename: [CWMIMEUtility decodeHeader: [CWParser _parameterValueUsingLine: theLine  range: aRange]
					     charset: [thePart defaultCharset]]];
      }
  }
}


//
//
//
+ (void) parseDate: (NSData *) theLine
	 inMessage: (CWMessage *) theMessage
{
  if ([theLine length] > 6)
    {
      struct header_rec hdr;
      NSData *aData;
      
      aData = [theLine subdataFromIndex: 6];
      
      if (parse_arpa_date([aData cString], &hdr))
	{
	  NSCalendarDate *aDate;
		
	  aDate = [NSCalendarDate dateWithTimeIntervalSince1970: hdr.time_sent];
	  [aDate setTimeZone: [NSTimeZone timeZoneForSecondsFromGMT: hdr.tz_offset]];
	  [theMessage setReceivedDate: aDate];
	}
    }
}


//
//
//
+ (void) parseDestination: (NSData *) theLine
		  forType: (PantomimeRecipientType) theType
		inMessage: (CWMessage *) theMessage
{  
  CWInternetAddress *anInternetAddress;
  char abuf[128], nbuf[128], *cf = "", *nf;
  int rc;

  if (theType == PantomimeBccRecipient && ([theLine length] > 5))
    {
      [theMessage addHeader: @"Bcc"  withValue: @""];
      cf = (char*)[[theLine subdataFromIndex: 5] cString];
    }
  else if (theType == PantomimeCcRecipient && ([theLine length] > 4))
    {
      [theMessage addHeader: @"Cc"  withValue: @""];
      cf = (char*)[[theLine subdataFromIndex: 4] cString];
    }
  else if (theType == PantomimeToRecipient && ([theLine length] > 4))
    {
      [theMessage addHeader: @"To"  withValue: @""];
      cf = (char*)[[theLine subdataFromIndex: 4] cString];
    }
  else if (theType == PantomimeResentBccRecipient && ([theLine length] > 12))
    {
      [theMessage addHeader: @"Resent-Bcc"  withValue: @""];
      cf = (char*)[[theLine subdataFromIndex: 12] cString];
    }
  else if (theType == PantomimeResentCcRecipient && ([theLine length] > 11))
    {
      [theMessage addHeader: @"Resent-Cc"  withValue: @""];
      cf = (char*)[[theLine subdataFromIndex: 11] cString];
    }
  else if (theType == PantomimeResentToRecipient && ([theLine length] > 11))
    {
      [theMessage addHeader: @"Resent-To"  withValue: @""];
      cf = (char*)[[theLine subdataFromIndex: 11] cString];
    }

  while (*cf != '\0')
    {
      rc = parse_arpa_mailbox(cf, abuf, sizeof(abuf), nbuf, sizeof(nbuf), &nf);

      anInternetAddress = [[CWInternetAddress alloc] init];
      
      if (rc < 0)
	{
	  [anInternetAddress setPersonal: [CWMIMEUtility decodeHeader: [NSData dataWithCString: cf]
							 charset: [theMessage defaultCharset]]];
	}
      else
	{
	  [anInternetAddress setPersonal: [CWMIMEUtility decodeHeader: [NSData dataWithCString: nbuf]
							 charset: [theMessage defaultCharset]]];
	  [anInternetAddress setAddress: [NSString stringWithCString: abuf]];
	}
      
      [anInternetAddress setType: theType];
      [theMessage addRecipient: anInternetAddress];
      RELEASE(anInternetAddress);
      cf = nf;
    }
}


//
//
//
+ (void) parseFrom: (NSData *) theLine
	 inMessage: (CWMessage *) theMessage
{
  CWInternetAddress *anInternetAddress;
  char abuf[128], nbuf[128], *cf, *nf;
  int rc;
  
  if (!([theLine length] > 6))
    {
      return;
    }
 
  cf = (char*)[[theLine subdataFromIndex: 6] cString];
  rc = parse_arpa_mailbox(cf, abuf, sizeof(abuf), nbuf, sizeof(nbuf), &nf);

  anInternetAddress = [[CWInternetAddress alloc] init];

  if (rc < 0)
    {
      [anInternetAddress setPersonal: [CWMIMEUtility decodeHeader: [NSData dataWithCString: cf]
						     charset: [theMessage defaultCharset]]];
    }
  else
    {
      [anInternetAddress setPersonal: [CWMIMEUtility decodeHeader: [NSData dataWithCString: nbuf]
						     charset: [theMessage defaultCharset]]];
      [anInternetAddress setAddress: [NSString stringWithCString: abuf] ];
    }
  
  [theMessage setFrom: anInternetAddress];
  RELEASE(anInternetAddress);
}


//
// This method is used to parse the In-Reply-To: header value.
//
+ (void) parseInReplyTo: (NSData *) theLine
              inMessage: (CWMessage *) theMessage
{
  if (!([theLine length] > 13))
    {
      return;
    }

  [theMessage setInReplyTo: [[theLine subdataFromIndex: 13] asciiString]];    
}


//
//
//
+ (void) parseMessageID: (NSData *) theLine
	      inMessage: (CWMessage *) theMessage
{
  if (!([theLine length] > 12))
    {
      return;
    }

  [theMessage setMessageID: [[theLine subdataFromIndex: 12] asciiString]];    
}


//
//
//
+ (void) parseMIMEVersion: (NSData *) theLine
		inMessage: (CWMessage *) theMessage
{
  if ([theLine length] > 14)
    {
      [theMessage setMIMEVersion: [[theLine subdataFromIndex: 14] asciiString]];
    }
}


//
//
//
+ (void) parseReferences: (NSData *) theLine
               inMessage: (CWMessage *) theMessage
{
  if ([theLine length] > 12)
    {
      NSMutableArray *aMutableArray;
      NSArray *allReferences;
      int i, count;

      allReferences = [[theLine subdataFromIndex: 12] componentsSeparatedByCString: " "];

      aMutableArray = [[NSMutableArray alloc] initWithCapacity: [allReferences count]];
      
      count = [allReferences count];
      for (i = 0; i < count; i++)
	{
	  [aMutableArray addObject: [[allReferences objectAtIndex: i] asciiString]];
	}

      [theMessage setReferences: aMutableArray];
      RELEASE(aMutableArray);
    }
}


//
//
//
+ (void) parseReplyTo: (NSData *) theLine
	    inMessage: (CWMessage *) theMessage
{
  CWInternetAddress *anInternetAddress;
  char abuf[128], nbuf[128], *cf, *nf;
  int rc;
  
  if (!([theLine length] > 10))
    {
      return;
    }

  cf = (char*)[[theLine subdataFromIndex: 10] cString];
  rc = parse_arpa_mailbox(cf, abuf, sizeof(abuf), nbuf, sizeof(nbuf), &nf);

  anInternetAddress = [[CWInternetAddress alloc] init];
  
  if (rc < 0)
    {
      [anInternetAddress setPersonal: [CWMIMEUtility decodeHeader: [NSData dataWithCString: cf]
						     charset: [theMessage defaultCharset]]];
    }
  else
    {
      [anInternetAddress setPersonal: [CWMIMEUtility decodeHeader: [NSData dataWithCString: nbuf]
						     charset: [theMessage defaultCharset]]];
      [anInternetAddress setAddress: [NSString stringWithCString: abuf]];
    }

  [theMessage setReplyTo: anInternetAddress];
  RELEASE(anInternetAddress);
}


//
//
//
+ (void) parseResentFrom: (NSData *) theLine
	 inMessage: (CWMessage *) theMessage
{
  CWInternetAddress *anInternetAddress;
  char abuf[128], nbuf[128], *cf, *nf;
  int rc;
  
  if (!([theLine length] > 13))
    {
      return;
    }
  
  cf = (char*)[[theLine subdataFromIndex: 13] cString];
  rc = parse_arpa_mailbox(cf, abuf, sizeof(abuf), nbuf, sizeof(nbuf), &nf);

  anInternetAddress = [[CWInternetAddress alloc] init];
  
  if (rc < 0)
    {
      [anInternetAddress setPersonal: [CWMIMEUtility decodeHeader: [NSData dataWithCString: cf]
						     charset: [theMessage defaultCharset]]];
    }
  else
    {
      [anInternetAddress setPersonal: [CWMIMEUtility decodeHeader: [NSData dataWithCString: nbuf]
						     charset: [theMessage defaultCharset]]];
      [anInternetAddress setAddress: [NSString stringWithCString: abuf]];
    }
  
  [theMessage setResentFrom: anInternetAddress];

  RELEASE(anInternetAddress);
}


//
//
//
+ (void) parseStatus: (NSData *) theLine
	   inMessage: (CWMessage *) theMessage
{
  if ([theLine length] > 8)
    {
      NSData *aData;
      
      aData = [theLine subdataFromIndex: 8];
      [[theMessage flags] addFlagsFromData: aData  format: PantomimeFormatMbox];
      [theMessage addHeader: @"Status"  withValue: [aData asciiString]];
    }
}


//
//
//
+ (void) parseXStatus: (NSData *) theLine
	    inMessage: (CWMessage *) theMessage
{
  if ([theLine length] > 10)
    {
      NSData *aData;

      aData = [theLine subdataFromIndex: 10];
      [[theMessage flags] addFlagsFromData: aData  format: PantomimeFormatMbox];
      [theMessage addHeader: @"X-Status"  withValue: [aData asciiString]];
    }
}


//
//
//
+ (void) parseSubject: (NSData *) theLine
	    inMessage: (CWMessage *) theMessage
{
  NSString *subject;

  if ([theLine length] > 9)
    {
      subject = [CWMIMEUtility decodeHeader: [[theLine subdataFromIndex: 8] dataByTrimmingWhiteSpaces]
			       charset: [theMessage defaultCharset]];
    }
  else
    {
      subject = @"";
    }
  
  [theMessage setSubject: subject];
}


//
//
//
+ (void) parseUnknownHeader: (NSData *) theLine
		  inMessage: (CWMessage *) theMessage
{
  NSData *aName, *aValue;
  NSRange range;

  range = [theLine rangeOfCString: ":"];
  
  if (range.location != NSNotFound)
    {
      aName = [theLine subdataWithRange: NSMakeRange(0, range.location) ];
      
      // we keep only the headers that have a value
      if (([theLine length] - range.location - 1) > 0)
	{
	  aValue = [theLine subdataWithRange: NSMakeRange(range.location + 2, [theLine length]-range.location-2) ];
	  
	  [theMessage addHeader: [aName asciiString]  withValue: [aValue asciiString]];
	}
    }
}


//
//
//
+ (void) parseOrganization: (NSData *) theLine
		 inMessage: (CWMessage *) theMessage
{
  NSString *organization;

  if ([theLine length] > 14)
    {
      organization = [CWMIMEUtility decodeHeader: [[theLine subdataFromIndex: 13] dataByTrimmingWhiteSpaces]
				    charset: [theMessage defaultCharset]];
    }
  else
    {
      organization = @"";
    }
  
  [theMessage setOrganization: organization];    
}

@end


//
// private methods
//
@implementation CWParser (Private)

+ (NSData *) _parameterValueUsingLine: (NSData *) theLine
                                range: (NSRange) theRange
{
  NSData *aData;
  NSRange r;
  
  int valueStart, valueEnd;
  
  // The parameter can be quoted or not like this (for example, with a charset):
  // charset="us-ascii"
  // charset = "us-ascii"
  // charset=us-ascii
  // charset = us-ascii
  // It can be terminated by ';' or end of line.
  
  // Look the the first occurrence of ';'.
  // That marks the end of this key value pair.
  // If we don't find one, we set it to the end of the line.
  r = [theLine rangeOfCString: ";"
	       options: 0
	       range: NSMakeRange(theRange.location + theRange.length,
				  [theLine length] - theRange.location - theRange.length)];

  if (r.length > 0)
    {
      valueEnd = r.location - 1;
    }
  else
    {
      valueEnd = [theLine length] - 1;
    }
  
  // Look for the first occurance of '=' before the valueEnd
  // That markes the beginning of the value.
  // If we don't find one, we set the beggining right after the end of the key tag
  r = [theLine rangeOfCString: "=" options: 0
	       range: NSMakeRange(theRange.location + theRange.length,
				  [theLine length] - theRange.location - theRange.length)];

  if (r.length > 0)
    {
      // If "=" was found, but after ";", something is very broken
      // and we just return nil. That can happen if we have a Content-Type like:
      //
      // Content-Type: text/x-patch; name=mpg321-format-string.diff; charset=ISO-8859-1
      //
      // "format" is part of the _name_ parameter. It has nothing to do with format=flowed.
      //
      if (r.location > valueEnd)
	{
	  return nil;
	}
      valueStart = r.location + r.length;
    }
  else
    {
      valueStart = theRange.location + theRange.length;
    }  

  // We now have a range that should contain our value.
  // Build a NSRange out of it.
  r = NSMakeRange(valueStart, valueEnd - valueStart + 1);
  
  // Grab our substring
  aData = [theLine subdataWithRange: r];
  
  // Trim white spaces
  aData = [aData dataByTrimmingWhiteSpaces];
  
  // If it quoted, grab the stuff in-between.
  aData = [aData dataFromQuotedData];

  return aData;
}

@end
