#include <string.h>
#include <ctype.h>
#include <time.h>

extern "C" {
#include <libgraph.h>
};

#include "mudclient.h"
#include "PapayaList.h"
#include "PluginHandler.h"
#include "TurfProtocol.h"
#include "TriggerHandler.h"
#include "Graph.h"

extern PluginHandler * phandler;
extern TriggerHandler * thandler;
static List * graph_list = NULL;

#define MAX_INDEX 1024

struct level_data {
  int exp[MAX_INDEX][2];
};

static Graph * graph = NULL;

extern "C" void plugin_init(void) {
  graph = new Graph();
}

extern "C" void plugin_cleanup(void) {
  delete graph;
}

#define MAJOR "1"
#define MINOR "1"

extern "C" char * plugin_query_name() {
  return "Graph";
}

extern "C" char * plugin_query_description() {
  return "Creates a pretty graph of experience over time.";
}

extern "C" char * plugin_query_major() {
  return MAJOR;
}

extern "C" char * plugin_query_minor() {
  return MINOR;
}

/**
 * Saves the graph to ~/.papaya/graph.png
 */

void saveGraphImage(Connection * c) {
  ListElement * tmp = graph_list->findEntry(c);
  if (!tmp)
    return;

  char filename[1024];
  char * home = getenv("HOME");
  if (home)
    sprintf(filename, "%s/.papaya/graph.png", home);
  else
    sprintf(filename, ".papaya/graph.png");

  struct level_data * data = (struct level_data *)tmp->getData();

  graphPtr gp = lg_graph_line_new("Experience Change", 640, 480);
  
  for (int index = 0; index < MAX_INDEX; index++) {
    if (data->exp[index][0] == -1)
      break;

    char buf[1024];
    sprintf(buf, "%d", data->exp[index][0]);

    lg_graph_line_add_point(LG_GRAPH_LINE(gp), buf, data->exp[index][1]);
  }

  lg_graph_line_render(LG_GRAPH_LINE(gp));
  lg_graph_save(gp, filename);
  lg_graph_line_destroy(LG_GRAPH_LINE(gp));
}

/**
 * This is called when the output of score is returned to us via the Turf
 * Protocol module.
 */

void Graph_Turf_Callback(Connection * conn, char * buf, void * d) {
  time_t curr_time;
  
  if (!buf)
    return;

  if (!strncasecmp(buf, "Exp:", 4)) {
    // Make sure we have an entry for this connection.  Create if not.
    ListElement * e = graph_list->findEntry(conn);
    if (!e)
      e = graph_list->newEntry(conn, NULL);

    // Ensure that we have a data set for this connection.
    struct level_data * data = (struct level_data *)e->getData();
    if (!data) {
      data = (struct level_data *)malloc(sizeof(struct level_data));
      memset(data, 0, sizeof(struct level_data));
      data->exp[0][0] = -1;
      data->exp[0][1] = -1;
      e->setData(data);
    }

    // Figure out where the number starts.
    char * ptr = buf + 4;
    while (isspace(*ptr))
      ptr++;

    // Figure out where the number ends
    char * end = ptr;
    while (isdigit(*end))
      end++;
    
    *end = '\0';

    // Get our two data fields.
    int exp = atoi(ptr);
    time(&curr_time);
    
    for (int index = 0; index < MAX_INDEX; index++) {
      if (data->exp[index][0] == -1) {
	// This is the one.
	data->exp[index][0] = curr_time;
	data->exp[index][1] = exp;

	data->exp[index+1][0] = -1;
	data->exp[index+1][1] = -1;
	saveGraphImage(conn);
	return;
      }
    }
  }

}

/**
 * This is called when a trigger returns.
 *
 * Once this has been called, we need to find out our score.
 */

int Graph_Callback(regex_t * regexp, char * in, void * c, Connection * conn) {

  TurfProtocol * tp = NULL;
  
  // Try and find the TurfProtocol object.
  void * handle = phandler->findPlugin("TurfProtocol");
  if (!handle) {
    new Message("Graph: Error", "TurfProtocol is not loaded.  This plugin does not work without it.", NULL);
    return;
  }
  
  if (!findTurf(handle, (void **)&tp)) {
    new Message("Graph: Error", "TurfProtocol is not loaded.  This plugin does not work without it.", NULL);
    return;    printf("papaya.turf_protocol_add: TurfProtocol not loaded.\n");
  }
  
  if (!tp) {
    new Message("Graph: Error", "TurfProtocol is not loaded.  This plugin does not work without it.", NULL);
    return;    printf("papaya.turf_protocol_add: TurfProtocol not loaded.\n");
  }
  

  tp->addCommand(conn, "score", Graph_Turf_Callback, NULL);
  return 1;
}

Graph::Graph() {
  version = 0.1;
  name = strdup("Grapher");
  graph_list = new List();

  thandler->addTrigger(new Trigger("You have been KILLED!!", Graph_Callback, NULL, false, NULL));
  thandler->addTrigger(new Trigger("You receive [0-9]+ experience points.", Graph_Callback, NULL, false, NULL));
  thandler->addTrigger(new Trigger("You failed to flee!  You lose 10 exps.", Graph_Callback, NULL, false, NULL));
  thandler->addTrigger(new Trigger("You flee from combat!  You lose 25 exps.", Graph_Callback, NULL, false, NULL));
  thandler->addTrigger(new Trigger("You failed!  You lose 50 exps.", Graph_Callback, NULL, false, NULL));
  thandler->addTrigger(new Trigger("You recall from combat!  You lose 100 exps.", Graph_Callback, NULL, false, NULL));

  

  phandler->registerPlugin(this, VERSION);
}

Graph::~Graph() {
  delete graph_list;

  phandler->unregisterPlugin(this);
}
