/*****************************************************************************
 *                                                                           *
 * Program:   paul                                                           *
 *            (P)rogramm zur (A)uswertung und (U)mformung von                *
 *            (L)aserbildern                                                 *
 * Uses:      libPNG (works best with 1.0.0, may be fail with 1.0.2)         *
 * Modul:     pngload.c                                                      *
 *            Load PNG files and take care for spezific Chunks               *
 * Author:    Andreas Tille                                                  *
 * Date:      18.06.1998                                                     *
 * Copyright: Andreas Tille, 1999; GNU Public License                        *
 *                                                                           *
 *****************************************************************************/

#include <png.h>
#include <stdio.h>
#include <setjmp.h>

#include "paul.h"

static void png_paul_error(png_struct *png_ptr, const char *message);
static void png_paul_warning(png_struct *png_ptr, const char *message);


int PaulLoadPNG(PICTURE *bild, FILE *fp, long flag, int *read_anything)
/* read PNG file
 * --- Parameter: ---
 * PICTURE       *bild         : image structure
 * FILE          *fp           : file pointer which points to open PNG file
 * long           flag         : may be only image info is whished -> ask this flag
 * --- Return: ---
 * PICTURE       *bild         : read image
 * int           *PaulLoadPNG(): RET_ERR or RET_OK
 */
{
  png_struct               *png_ptr;
  png_info                 *info_ptr;
  int                       i, color_type, interlace_type, bit_depth;
  register   unsigned char *ap = NULL, *fip = NULL;
  unsigned char            *ptr = NULL;
  png_text                 *chunks, *cp, *fcp;
  png_time                 *zeit;
  png_color                *gamma;
  png_uint_32               ww, hh;
  png_color_16              my_background;

  *read_anything = 0;
  g_return_val_if_fail ( IS_PICTURE(bild), RET_ERR);
  g_return_val_if_fail ( fp, RET_ERR);
  /* Init PNG Reader */
  g_return_val_if_fail ( 
     (png_ptr = png_create_read_struct(PNG_LIBPNG_VER_STRING, &ptr, png_paul_error, png_paul_warning)),
     RET_ERR ) ;
  if ( !(info_ptr = png_create_info_struct(png_ptr)) ) {
    png_destroy_read_struct(&png_ptr, NULL, NULL);
    return RET_ERR;
  }
  /* important to handle read errors */
  if ( setjmp(png_ptr->jmpbuf) ) {
    png_destroy_read_struct(&png_ptr, &info_ptr, NULL);
    return RET_ERR;
  }

  png_init_io(png_ptr, fp);
  png_read_info(png_ptr, info_ptr);

  png_get_IHDR(png_ptr, info_ptr, &ww, &hh, &bit_depth, &color_type, &interlace_type,
	       NULL, NULL);
  bild->size  = ( bild->W = ww ) * ( bild->H = hh );
  if ( interlace_type != PNG_INTERLACE_NONE ) {
     /* The reason for this is that png_read_rows() which automatically handles interlaced *
      * images, failed since libpng 1.0.2.  So now single rows are read and handling of    *
      * interlaced PNG images could be added on demand.                                    */
    g_warning(_("Sorry, %s does not yet support interlaced PNG images.\nLoading of %s failed."),
               exename, bild->file);
    png_destroy_read_struct(&png_ptr, &info_ptr, NULL);
    return RET_ERR;
  }

  if ( bit_depth < 8 ) png_set_packing(png_ptr);
  
  /* first attempt to detect gray image ... this could be more sophisticated! */
  if ( color_type == PNG_COLOR_TYPE_GRAY ) bild->spp   = 1;
  else                                     bild->spp   = 3;

  if ( bit_depth == 16 ) png_set_strip_16(png_ptr);

  /* Setup Translators */
  if ( color_type == PNG_COLOR_TYPE_GRAY || color_type == PNG_COLOR_TYPE_GRAY_ALPHA)
    png_set_expand(png_ptr);
  if ( color_type == PNG_COLOR_TYPE_PALETTE || color_type == PNG_COLOR_TYPE_GRAY ) 
    png_set_expand(png_ptr);
  if ( bit_depth  == 16 )
    png_set_strip_16(png_ptr);
  if ( png_get_valid(png_ptr, info_ptr, PNG_INFO_tRNS) ) 
    png_set_expand(png_ptr);
    
  /*  png_set_filler(png_ptr, 0x0000ff, PNG_FILLER_AFTER); 
   *  This could help in reading alpha channel data but breaks loading
   *  of certain PNG files. */

  if ( png_get_valid(png_ptr, info_ptr, PNG_INFO_bKGD) ) 
    png_set_background(png_ptr, &info_ptr->background, PNG_BACKGROUND_GAMMA_FILE, 1, 1.0);
  else {
    my_background.red = my_background.green = my_background.blue = my_background.gray = 0;
    png_set_background(png_ptr, &my_background, PNG_BACKGROUND_GAMMA_SCREEN, 0, 1.0);
  }

  bild->trans = 0;

  if ( png_get_text(png_ptr, info_ptr, &chunks, &i) ) {
    for ( fcp = (cp = chunks) + i; cp < fcp; cp++ ) {
      if ( SetSpec(bild->spec, cp->key, cp->text) ) 
        g_warning(_("Unknown key \"%s\": %s"), cp->key, cp->text);
    }
  }

  if ( png_get_tIME(png_ptr, info_ptr, &zeit) ) 
    bild->zeit = convert_png_time_to_time_t(zeit);
  GetPictureSpecs(bild, bild->file);

  if ( png_get_valid(png_ptr, info_ptr, PNG_INFO_pHYs) && 
    info_ptr->x_pixels_per_unit == info_ptr->y_pixels_per_unit ) /* sonst kann ja jeder kommen */
    bild->res = info_ptr->x_pixels_per_unit;
  if ( png_get_valid(png_ptr, info_ptr, PNG_INFO_oFFs) ) {
    png_get_oFFs(png_ptr, info_ptr, &(bild->x_offset), &(bild->y_offset), &i);
    if ( i != PNG_OFFSET_PIXEL ) g_warning(_("Suspicious unit in scanner offset!"));
  }
  if ( PNG_INFO_PLTE != png_get_PLTE(png_ptr, info_ptr, &gamma, &(bild->n_gamma)) &&
       png_get_valid(png_ptr, info_ptr, PNG_INFO_PLTE) )
     g_warning(_("Error while reading Gamma table!"));
  else if ( bild->n_gamma ) {
    png_color *gap, *gip;
    bild->gamma = g_new(png_color, bild->n_gamma);
    for ( gip = gamma + bild->n_gamma, gap = bild->gamma; gamma < gip; gamma++, gap++ ) {
      gap->red   = gamma->red;
      gap->green = gamma->green;
      gap->blue  = gamma->blue;
    }
  }
  if ( OnlyInfo(flag) ) {
    bild->DATA = NULL;
    png_destroy_read_struct(&png_ptr,&info_ptr,NULL);
    return RET_OK;
  }
  bild->data = g_new0(unsigned char, 3*bild->size);

  png_read_update_info(png_ptr, info_ptr);
  png_start_read_image(png_ptr);

  if(info_ptr->color_type == PNG_COLOR_TYPE_RGB || info_ptr->color_type == PNG_COLOR_TYPE_RGB_ALPHA) {
    bild->spp = 3;
    for ( fip = (ap = bild->data) + 3*bild->size; ap < fip; ap += 3*bild->W ) {
      png_read_row(png_ptr, ap, NULL);
      ++*read_anything;
    }
  } else {
    register unsigned char *pp, *fpp;
     
    bild->spp = 1;
    ptr       = g_new(unsigned char, bild->W);
    
    for ( fip = (ap = bild->data) + 3*bild->size, ap++; ap < fip; ) {
      png_read_row(png_ptr, ptr, NULL);
      if ( DisplayGray(flag) )
        for ( fpp = (pp = ptr) + bild->W; pp < fpp; pp++, ap += 3 ) *ap = *(ap-1) = *(ap+1) = *pp;
      else
        for ( fpp = (pp = ptr) + bild->W; pp < fpp; pp++, ap += 3 ) *ap = *pp;
      ++*read_anything;
    }
    FREE(ptr);
  }

  png_read_end(png_ptr, info_ptr);
  png_destroy_read_struct(&png_ptr, &info_ptr, (png_infopp)NULL);

  return RET_OK;
}


static void png_paul_error(png_struct *png_ptr, const char *message)
{
  g_warning("libpng error: %s", message);

  if ( png_ptr->error_ptr ) FREE(*(unsigned char **)png_ptr->error_ptr);

  longjmp(png_ptr->jmpbuf, 1);
}


static void png_paul_warning(png_struct *png_ptr, const char *message)
{
  if ( !png_ptr ) return;

  g_warning("libpng warning: %s", message);
}

