/*****************************************************************************
 *                                                                           *
 * Program:   paul                                                           *
 *            (P)rogramm zur (A)uswertung und (U)mformung von                *
 *            (L)aserbildern                                                 *
 * Modul:     gray.c                                                         *
 *            make gray images using several methods                         *
 * Author:    Andreas Tille                                                  *
 * Date:      25.05.1998                                                     *
 * Copyright: Andreas Tille, 1999; GNU Public License                        *
 *                                                                           *
 *****************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "paul.h" 

static int MakeSingleGreenFromAll(PICTURE *bild, unsigned char eps)
/* create monochrome image from RGB via Gray = (r+g+b)/3 
 * if |r-g| < eps, |r-b| < eps and |g-b| < eps
 * --- Parameter: ---
 * PICTURE  *bild       : list of images
 * unsigned char greps  : threshold up to which RGB differences are accepted
 *                        greps == 255  =>  accept all
 * --- Return: ---
 * int MakeSingleGreen(): RET_ERR or RET_OK
 */
{
  register int   i, j, k, storepix;
  unsigned char *ap, *fip;
  char          *desc;
   
  g_return_val_if_fail ( IS_PICTURE(bild), RET_ERR ) ;

  storepix = bild->storepix;
  if ( eps < 255 ) {
    for ( fip = (ap = bild->DATA + 1) + storepix*bild->size; ap < fip; ap += storepix ) {
      i = *(ap-1) - *ap;
      j = *ap - *(ap+1);
      k = *(ap+1) - *(ap-1);
      if ( abs(i) > eps || abs(j) > eps || abs(k) > eps ) {
        /* printf("Border %i overrun: R-G=%i, G-B=%i, B-R=%i\n", eps, i, j, k); */
        printf("%i: %i, %i, %i\n", (int)(ap - (bild->DATA))/storepix, *(ap-1), *ap, *(ap+1));
        /*  return 1; */
      }
      i   = *(ap-1) + *ap + *(ap+1);
      *ap = (int)i / storepix;
      *(ap-1) = *(ap+1) = 0;
    }
  } else {
    for ( fip = (ap = bild->DATA + 1) + storepix*bild->size; ap < fip; ap += storepix ) {
      i   = *(ap-1) + *ap + *(ap+1);
      *ap = (int)i / storepix;
      *(ap-1) = *(ap+1) = 0;
    }
  }
  desc = g_strdup_printf("Monochrom %s via middle of all channels", ImgFileName(bild));
  ImgChunksUpdate(bild, TypGray, desc, APPGRAY, MAKEGRAY);
  FREE(desc);

  return RET_OK;
}

static int MakeSingleGreenFromChannel(PICTURE *bild, int channel)
/* create monochrome image from one channel
 * --- Parameter: ---
 * PICTURE  *bild                  : list of images
 * int       channel               : 0 = red, 1 = green, 2 = blue
 * --- Return: ---
 * int MakeSingleGreenFromChannel(): RET_ERR or RET_OK
 */
{
  register int            storepix;
  register unsigned char *ap, *bp, *fip;
  char                   *desc, *chan[] = {"red", "green", "blue"};

  g_return_val_if_fail ( IS_PICTURE(bild), RET_ERR ) ;

  storepix = bild->storepix;
  if ( channel != 1 ) {
    bp = bild->DATA + channel;
    for ( fip = (ap = bild->DATA + 1) + storepix*bild->size; ap < fip; ap += storepix, bp += storepix )
      *ap = *bp;
  }
  for ( fip = (ap = bild->DATA) + storepix*bild->size; ap < fip; ap += storepix ) 
    *ap = *(ap+2) = 0;

  desc = g_strdup_printf("Monochrom %s from %s channel", ImgFileName(bild), chan[channel]);
  ImgChunksUpdate(bild, TypGray, desc, APPGRAY, MAKEGRAY);
  FREE(desc);

  return RET_OK;
}

void MakeSingleGreen(PICTURE *bild, int ch)
/* Decides, whether to use single channel or mix from all
 * --- Parameter: ---
 * PICTURE *bild: image to make monochrome
 * int      ch  : flag
 */
{
  g_return_if_fail ( IS_PICTURE(bild) );
   
  if ( ch < 3 ) MakeSingleGreenFromChannel(bild, ch);
  else          MakeSingleGreenFromAll(bild, ch);
  bild->spp   = 1;
}


int MakeGray(PAUL *p)
/* converts RGB-images into gray images
 * --- Parameter: ---
 * PAUL *p         : list of images, options
 *                 : used options:
 *                   greps : allowed noise level, 
 *                   greps == 0  =>  accept all
 * --- Return: ---
 * int   MakeGray(): RET_ERR or RET_OK
 */
{
  PICTURE *bild;
  GList   *pl;

  g_return_val_if_fail ( IS_PAUL(p), RET_ERR);
  g_return_val_if_fail ( BILD(p->piclist), RET_ERR );

  for ( bild = BILD(pl = p->piclist); pl; bild = BILD(pl = pl->next) ) {
    if ( IsMonochrom(bild) ) g_warning(_("%s is monochrom yet."), bild->file);
    else                     MakeSingleGreen(bild, p->opt->greps);
  }
  return RET_OK;
}


