/*
 * pftp -- sends files from host to host through free choosable ports
 *
 * Copyright (C) 1996-2000 Ben Schluricke
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the emplied warranty of MERCHANT-
 * ABILITY OF FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
 * Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 * 
 *    Written by Ben Schluricke
 *    E-Mail:    support@pftp.de
 *
 * This program is dedicated to my girl-friend, Heather O'Rourke.
 *
 *
 */
#if defined USE_POSIX_THREAD
#define _REENTRANT
#include <pthread.h>
#endif
#ifdef FreeBSD
#include <sys/errno.h>
#include <sys/types.h>
#else
#include <sys/sysmacros.h>
#endif
#include <dirent.h>
#if defined Linux || defined NEXTSTEP
#include <sys/dir.h>
#endif
#if defined unicos || defined SunOS
#include <fcntl.h>
#endif
#include <sys/time.h>
#include "main.h"

#ifdef __USE_FILE_OFFSET64
#define PFTP_PRINTF_LLD "%9lld"
#else
#define PFTP_PRINTF_LLD "%9ld"
#endif

extern void set_tty(int);
extern int is_pftp_mesg(char *);
extern short mycmp(char *, char *);
extern void bubble_menu(char **, int);
extern void quick_sort(char **, char *, long);
extern void print_error_mesg(char *, char *);

extern long tlastentry;
extern long dlastentry;
extern char *tfiles[PFTP_MAX_PFM_ENTRY]; /* holds the selected files */
extern char *dfiles[PFTP_MAX_PFM_ENTRY]; /* holds the selected files */
extern char *blankline;

short empty_dir(char *s)
{
   DIR *dp;
#if defined Linux || defined NEXSTEP
   struct direct *dir;
#else
   struct dirent *dir;
#endif

   if (!(dp = opendir(s))) {
      return 0;
   }
   while ((dir = readdir(dp)))
   {
      if (dir->d_ino) {
         if (strcmp(dir->d_name, "..") && strcmp(dir->d_name, ".")) {
            closedir(dp);
            return 0;
         }
      }
   }
   closedir(dp);

   return 1;
}


void rm_empty_dirs(void)
{
   struct stat buf;
   DIR *dp;
#if defined Linux || defined NEXSTEP
   struct direct *dir;
#else
   struct dirent *dir;
#endif

   if (!(dp = opendir("."))) {
      return;
   }
   while ((dir = readdir(dp)))
   {
      if (dir->d_ino && strcmp(dir->d_name, "..") && strcmp(dir->d_name, ".")) {
         buf.st_mode = 0;
         if (!stat(dir->d_name, &buf) && ((buf.st_mode & S_IFMT) == S_IFDIR)) {
            if (empty_dir(dir->d_name)) rmdir(dir->d_name);
         }
      }
   }
   closedir(dp);
}


long read_dir_entries(long i, char **dirs, char *dirname, long *incoming)
{
   struct stat buf;
   char ilock[SONAME];
   char msg_file[SONAME];
   long j=0;
   int len=0;
   DIR *dp;
#if defined Linux || defined NEXSTEP
   struct direct *dir;
#else
   struct dirent *dir;
#endif

   if (chdir(dirname) < 0) return i;

   /*
    * Reading directory entries.
    */
   if (!(dp = opendir("."))) {
      if (dirs) *dirs = NULL;
      chdir("..");
      return i;
   }
   while ((dir = readdir(dp)))
   {
      if (dir->d_ino && strcmp(dir->d_name, "..") && strcmp(dir->d_name, ".")) {
         buf.st_mode = 0;
         if (!lstat(dir->d_name, &buf) && ((buf.st_mode & S_IFMT) == S_IFDIR)) {
            sprintf(ilock, "incoming:%s", dir->d_name);
            if (dirs) {
               if (!stat(ilock, &buf)) {
                  MEM_CHECK((*(dirs+i) = (char *)calloc(SONAME, sizeof(char))));
                  **(dirs+i) = 'I';
                  (*incoming)++;
               }
               else {
                  if (empty_dir(dir->d_name)) {
                     rmdir(dir->d_name);
                     continue;
                  }
                  MEM_CHECK((*(dirs+i) = (char *)calloc(SONAME, sizeof(char))));
                  **(dirs+i) = 'N';
               }
               for (j=1; j < DATECOLUMN; j++) *(*(dirs+i)+j) = ' ';
               sprintf(*(dirs+i)+j, "%s %s", dir->d_name, dirname);
               sprintf(msg_file, "%s/%s", dir->d_name, PFTP_INFO_FILE);
               if (!access(msg_file, R_OK) && is_pftp_mesg(msg_file)) *(*(dirs+i)+1) = 'M';
            }
            else if (!stat(ilock, &buf)) (*incoming)++;
            else if (empty_dir(dir->d_name)) {
               rmdir(dir->d_name);
               continue;
            }
            i++;
         }
         else if ((len = strlen(dir->d_name)) > 9 \
           && lstat((dir->d_name)+9, &buf) \
           && !((buf.st_mode & S_IFMT) == S_IFDIR)) {
            unlink(dir->d_name);
         }
         else if (len < 10) unlink(dir->d_name);
      }
   }
   closedir(dp);
   if (dirs) *(dirs+i) = NULL;
   chdir("..");
   return i;
}


long read_cur_dir(char **dirs, long *incoming)
{
   DIR *dp;
   struct stat buf;
   int i=0;
#if defined Linux || defined NEXSTEP
   struct direct *dir;
#else
   struct dirent *dir;
#endif
   *incoming = 0;
   if (!(dp = opendir("."))) {
      return -1;
   }
   while ((dir = readdir(dp)))
   {
      if (dir->d_ino && strcmp(dir->d_name, "..") && strcmp(dir->d_name, ".")) {
         buf.st_mode = 0;
         if (!stat(dir->d_name, &buf) && ((buf.st_mode & S_IFMT) == S_IFDIR)) {
            i = read_dir_entries(i, dirs, dir->d_name, incoming);
         }
      }
   }
   closedir(dp);
   return i;
}


void write_pinfo(char **pmain)
{
   FILE *fp=NULL;
   char *cdir=NULL;
   char *tmp=NULL;
   struct stat buf;

   MEM_CHECK((cdir = (char *)calloc(LONAME, sizeof(char))));
   MEM_CHECK((tmp = (char *)calloc(SONAME, sizeof(char))));

   if ((fp = fopen("Pinfo", "w")) == NULL) {
      fprintf(stderr, "** Pinfo: %s\n", _PFTP_ERROR_ARRAY_);
      set_tty(0);
      exit(PFTP_FILE_OPEN_ERR);
   }
   for (; *pmain; pmain++) {
      memcpy(tmp, *pmain, SONAME);
      if (*tmp == 'D') {
         sprintf(cdir, "%s/%s", USERSTRING(0), DATESTRING(0));
         buf.st_mode = 0;
         if (stat(cdir, &buf)) continue;
      }
      if (*tmp == 'I') continue;
      else if (*tmp == 'N') *tmp = 'O';
      else if (*tmp != 'O') *tmp = ' ';
      *(tmp+2) = ' ';
      *(tmp+DATECOLUMN-1) = ' ';
      *(tmp+USERCOLUMN-1) = ' ';
      fprintf(fp, "%s\n", tmp);
   }
   fclose(fp);
   free(tmp);
   free(cdir);
}


long read_pinfo(char **pmain, long *incoming, short PFTP_PEEK)
{
   char *dirs[PFTP_MAX_PFM_ENTRY], **dirsptr; /* holds directory entries */
   char *pinfoentries[PFTP_MAX_PFM_ENTRY], **pinfoptr;
   FILE *fp=NULL;
   char *ptmp=NULL, *tmpstr=NULL;
   struct stat buf;
   long i=0, j=0;
   long pinfonum=0;
   *pinfoentries = NULL;
   *dirs = NULL;

   /*
    * Reading all top level and first sublevel directories.
    */
   if ((i = read_cur_dir(dirs, incoming)) < 0) return -1;

   /*
    * Reading Pinfo file if it exists.
    */
   buf.st_mode = 0;
   if (!stat("Pinfo", &buf) && buf.st_size) {
      if ((fp = fopen("Pinfo", "r")) != NULL) {
         MEM_CHECK((ptmp = (char *)calloc(SONAME, sizeof(char))));
         for (pinfonum=0, pinfoptr=pmain; fgets(ptmp, SONAME, fp) && pinfonum < PFTP_MAX_PFM_ENTRY; pinfonum++) {
            if (*pinfoptr) {
               do {
                  *((*pinfoptr)+DATECOLUMN-1) = ' ';
                  *((*pinfoptr)+USERCOLUMN-1) = ' ';
                  memcpy(ptmp, *pinfoptr, SONAME);
                  pinfoptr++;
               } while (**(pinfoptr-1) == 'I');
            }
            else {
               for (tmpstr=ptmp; *tmpstr && *tmpstr != '\n'; tmpstr++);
               *tmpstr = '\0';
            }
            *(pinfoentries+pinfonum) = ptmp;
            MEM_CHECK((ptmp = (char *)calloc(SONAME, sizeof(char))));
         }
         if (ptmp) free(ptmp);
         fclose(fp);
         *(pinfoentries+pinfonum) = NULL;
      }
   }

   /*
    * Comparing Pinfo file entries with directory entries.
    */
   for (i=0, dirsptr=dirs; *dirsptr; dirsptr++) {
      for (pinfoptr=pinfoentries, j=0; j < pinfonum; pinfoptr++, j++) {
         if (**dirsptr == 'I') break;
         if (*pinfoptr && !mycmp((*dirsptr)+DATECOLUMN, (*pinfoptr)+DATECOLUMN)) {
            if (**pinfoptr == 'I') break;
            if (!*(pmain+i)) {
               MEM_CHECK((*(pmain+i) = (char *)calloc(SONAME, sizeof(char))));
            }
            strcpy(*(pmain+i), *pinfoptr);
            free(*dirsptr);
            *dirsptr = NULL;
            free(*pinfoptr);
            *pinfoptr = NULL;
            break;
         }
      }
      if (*dirsptr) {
         if (!*(pmain+i)) {
            MEM_CHECK((*(pmain+i) = (char *)calloc(SONAME, sizeof(char))));
         }
         strcpy(*(pmain+i), *dirsptr);
      }
      *(*(pmain+i)+DATECOLUMN-1) = '\0';
      *(*(pmain+i)+USERCOLUMN-1) = '\0';
      i++;
   }
   if (*(pmain+i)) {
      free(*(pmain+i));
      *(pmain+i) = NULL;
   }
   if (i > 0) bubble_menu(pmain, i);
   if (!PFTP_PEEK) write_pinfo(pmain);
   return i;
}


long set_entry_in_files(char **files, char *str, long *lastentry)
{
   char **tmp;
   struct stat buf;
   long count=0;

   if (lstat(str, &buf)) return -1;

   if ((buf.st_mode & S_IFMT) == S_IFDIR) {
      char *tmpdir[PFTP_MAX_PFM_ENTRY];
#if defined Linux || defined NEXSTEP
      struct direct *dir;
#else
      struct dirent *dir;
#endif
      DIR *dp=NULL;

      tmp = tmpdir;
      if ((dp = opendir(str)) == NULL) {
         fprintf(stderr, "** %s: %s\n", str, _PFTP_ERROR_ARRAY_);
         set_tty(0);
         exit(PFTP_DIR_OPEN_ERR);
      }
      while ((dir = readdir(dp)))
      {
         if (dir->d_ino && mycmp(".", dir->d_name) && mycmp("..", dir->d_name)) {
            MEM_CHECK((*tmp = (char *)calloc(LONAME, sizeof(char))));
            sprintf(*tmp++, "%s/%s", str, dir->d_name);
         }
      }
      closedir(dp);
      for (tmp--; tmp+1 != tmpdir; tmp--) {
         set_entry_in_files(files, *tmp, lastentry);
         free(*tmp);
      }
   }
   /*
    * Check if files were already taged.
    */
   for (tmp=files; tmp - files <= *lastentry; tmp++) {
      if (*tmp && !mycmp(*tmp, str)) return tmp - files;
   }

   /*
    * Search an empty field and set the new entry.
    */
   for (tmp=files; *tmp && count < PFTP_MAX_PFM_ENTRY; count++, tmp++);
   if (count == PFTP_MAX_PFM_ENTRY) {
      fprintf(stderr, "\n** Maximum limit of directory entries exceeded!\n");
      fprintf(stderr, "*** Type any key to continue ***");
      fgetc(stdin);
      return -1;
   }
   if (tmp - files >= *lastentry) *lastentry = tmp - files;
   MEM_CHECK((*tmp = (char *)calloc(LONAME, sizeof(char))));
   strcpy(*tmp, str);
   return -1;
}


void rm_entry_from_files(char **files, char *str, long *lastentry)
{
   char **tmp;
   struct stat buf;

   if (lstat(str, &buf)) return;

   if ((buf.st_mode & S_IFMT) == S_IFDIR) {
      char *tmpdir[PFTP_MAX_PFM_ENTRY];
#if defined Linux || defined NEXSTEP
      struct direct *dir;
#else
      struct dirent *dir;
#endif
      DIR *dp=NULL;

      tmp = tmpdir;
      if ((dp = opendir(str)) == NULL) {
         fprintf(stderr, "** %s: %s\n", str, _PFTP_ERROR_ARRAY_);
         set_tty(0);
         exit(PFTP_DIR_OPEN_ERR);
      }
      while ((dir = readdir(dp)))
      {
         if (dir->d_ino && mycmp(".", dir->d_name) && mycmp("..", dir->d_name)) {
            MEM_CHECK((*tmp = (char *)calloc(LONAME, sizeof(char))));
            sprintf(*tmp++, "%s/%s", str, dir->d_name);
         }
      }
      closedir(dp);
      for (tmp--; tmp+1 != tmpdir; tmp--) {
         rm_entry_from_files(files, *tmp, lastentry);
         free(*tmp);
      }
   }
   for (tmp=files; tmp - files <= *lastentry; tmp++) {
      if (*tmp && !mycmp(*tmp, str)) {
         free(*tmp);
         *tmp=NULL;
         if (tmp - files == *lastentry) {
            for (;!*tmp && tmp != files; tmp--, (*lastentry)--);
            if (*lastentry < 0) *lastentry = 0;
         }
         return;
      }
   }
}


void printlistentry(char *arrow, char *dir, char *file)
{
   int i=0,j=0;
   char perms[10];
   char name[LONAME];
#ifdef FreeBSD
   char date[SONAME];
   char device_numbers[SONAME];
#endif
   char **tmp;
   char type='\0', deletef=' ', tagf=' ';
   struct stat buf;
   char *modes[] = {
      "---", "--x", "-w-", "-wx",
      "r--", "r-x", "rw-", "rwx"
   };

   
   /*
    * Set full path name.
    */
   sprintf(name, "%s/%s", dir, file);

   /*
    * Get inode contents.
    */
   buf.st_mode = 0;
   lstat(name, &buf);
   
   switch(buf.st_mode & S_IFMT) {
      case S_IFREG:
         type = '-';
         break;
      case S_IFDIR:
         type = 'd';
         break;
      case S_IFCHR:
         type = 'c';
         break;
      case S_IFBLK:
         type = 'b';
         break;
      case S_IFIFO:
         type = 'p';
         break;
      case S_IFLNK:
         type = 'l';
         break;
      default:
         type = '?';
         break;
   }

   *perms = '\0';
   for (i=2; i >= 0; i--) {
      j = (buf.st_mode >> (i*3)) & 07;
      strcat(perms, *(modes+j));
   }

   /*
    * Check if the file was taged or will
    * be deleted.
    */
   for (tmp=tfiles; tmp - tfiles <= tlastentry; tmp++) {
      if (*tmp && !mycmp(*tmp, name)) {
         tagf='*';
         break;
      }
   }
   for (tmp=dfiles; tmp - dfiles <= dlastentry; tmp++) {
      if (*tmp && !mycmp(*tmp, name)) {
         deletef='D';
         break;
      }
   }

   /*
    * Finally print out the line.
    */
#ifndef FreeBSD
   if (type == 'c' || type == 'b') {
      sprintf(name, "%s %c%c %c%s %4d, %4d %.12s %s", \
      arrow, deletef, tagf, type, perms, \
      major(buf.st_rdev), minor(buf.st_rdev), \
      ctime(&buf.st_mtime)+4, file);
   }
   else {
      sprintf(name, "%s %c%c %c%s " PFTP_PRINTF_LLD " %.12s %s", \
      arrow, deletef, tagf, type, perms, \
      buf.st_size, ctime(&buf.st_mtime)+4, file);
   }
#else
   /* work around a bug in FreeBSD--tested FreeBSD 3.2-STABLE */
   if (type == 'c' || type == 'b') {
      sprintf(device_numbers, "%4d, %4d " PFTP_PRINTF_LLD, \
      major(buf.st_rdev), minor(buf.st_rdev), (long)buf.st_size);
   }
   else sprintf(device_numbers, PFTP_PRINTF_LLD, (long)buf.st_size);
   strcpy(date, ctime(&buf.st_mtime)+4);
   *(date+12) = 0;
   sprintf(name, "%s %c%c %c%s ", arrow, deletef, tagf, type, perms);
   strcat(name, device_numbers);
   strcat(name, date);
   strcat(name, " ");
   strcat(name, file);
#endif
   *(name+_WINCOLS_-1) = '\0';
   fprintf(stdout, "%s%s%s\n", blankline, name, type == 'd' ? "/": "");
}


long read_subdir(char **direntries, char *pdir)
{
   long endnum=0;
   DIR *dp=NULL;
#if defined Linux || defined NEXSTEP
   struct direct *dir;
#else
   struct dirent *dir;
#endif

   if ((dp = opendir(pdir)) == NULL) {
      print_error_mesg("opendir", pdir);
      return -1;
   }
   for (endnum=0; (dir = readdir(dp)); endnum++)
   {
      if (!dir->d_ino || !mycmp(".", dir->d_name) || !mycmp("..", dir->d_name) \
         || !mycmp(PFTP_INFO_FILE, dir->d_name) || !mycmp(PFTP_LOCK_FILE, dir->d_name)) {
         endnum--;
      }
      else if (endnum < PFTP_MAX_PFM_ENTRY) {
         MEM_CHECK((*(direntries+endnum) = (char *)calloc(LONAME, sizeof(char))));
         strcpy(*(direntries+endnum), dir->d_name);
         fprintf(stderr, "[%dH%4ld", _WINROWS_, endnum+1);
      }
      else break;
   }
   closedir(dp);
   *(direntries+endnum) = NULL;

   if (!endnum) {
      fprintf(stdout, "[%dH%s", _WINROWS_, blankline);
      fprintf(stdout, "** This directory is empty!");
      return -1;
   }

   /*
    * Sort directory entries.
    */
   quick_sort(direntries, pdir, endnum);

   return endnum;
}
