# Copyright (C) 2007-2008 www.stani.be
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see http://www.gnu.org/licenses/

import os, sys

if __name__ == '__main__':
    sys.path.extend(['../..','../../core/lib'])
    __builtins__._=str

import wx
import wx.grid as  gridlib

import droplet, inspectorTag
from formField import IMAGE_READ_EXTENSIONS, is_file
from wildcard import wildcard_list
from clipboard import copy_text
from graphics import bitmap_open
from core.pil import extract_info, pyexiv2, DOT, image_open
from unicoding import ENCODING

SIZE    = (470,510)
SELECT  = _('Select')
ALL     = _('All')
TAGS    = [SELECT, ALL,'Pil']
if pyexiv2:
    TAGS.extend(['Exif','Iptc'])
TAGS.extend(['Pexif','Zexif'])
WX_ENCODING = wx.GetDefaultPyEncoding()

def shortcut(key):
    return ''

def extract_data(filename,info):
    if filename.strip():
        extract_info(info,image_open(filename),filename)
    return [(shortcut(key),'<%s>'%key,value) for key, value in info.items()]

class Grid(droplet.Mixin,inspectorTag.Grid):
    wildcard    = '|'.join([
        wildcard_list(_('Images'),IMAGE_READ_EXTENSIONS),_('All files'),'*'])
    def _init(self):
        super(Grid,self)._init()
        self.info       = {}
        self.SetAsFileDropTarget(self,self.OnDrop)
        self.SetAsFileDropTarget(self.GetEmpty(),self.OnDrop)
        self.SetAsFileDropTarget(self.GetDialog(),self.OnDrop)
        self.Bind(wx.EVT_KEY_DOWN, self.OnKeyDown)
        
    def OnKeyDown(self,event):
        if event.ControlDown() and event.GetKeyCode() == 67:
            self.OnCopy()
        else:
            event.Skip()
            
    def show_message(self,message,title='',style=wx.OK | wx.ICON_EXCLAMATION):
        dlg = wx.MessageDialog(self, 
                message,
                title,
                style,
        )
        answer  = dlg.ShowModal()
        dlg.Destroy()
        return answer

        
    def OnDrop(self,filenames,x,y):
        f   = filenames[0]
        if is_file(f):
            self.OpenImage(f)
            return
        if type(f) is unicode:
            for encoding in [WX_ENCODING,'latin1','utf-8',ENCODING]:
                try:
                    f   = f.encode(encoding)
                    if is_file(f):
                        self.OpenImage(f)
                        return
                except UnicodeEncodeError:
                    pass
        self.show_message(_('Only an image file is allowed.'))
        
    def SetTag(self,tag,autosize=True):
        super(Grid,self).SetTag(tag)
        if tag == ALL:
            self.tag_data    = self.all_data
        elif tag == SELECT:
            self.tag_data    = [row for row in self.all_data\
                                if not(DOT in row[1])]
        else:
            tag             = '<'+tag+DOT
            n               = len(tag)
            self.tag_data   = [row for row in self.all_data if row[1][:n]==tag]
        self.SetFilter()
        
    def SetFilter(self,filter=None):
        if filter is None:
            filter  = self.GetFilter().GetValue()
        if filter.strip():
            self.data   = [row for row in self.tag_data
                if filter.lower() in unicode(row).lower()]
        else:
            self.data   = self.tag_data
        self.data.sort()
        if not self.CheckEmpty():
            self.RefreshAll()
        
    def OpenImage(self,filename):
        self.info       = {}
        self.filename   = filename
        self.SetData(extract_data(filename,self.info),SELECT)
        
    def OnCopy(self,event=None):
        if self.data:
            copy_text(unicode(
                self.data[self.GetGridCursorRow()][self.GetGridCursorCol()]))
        
class SingleGrid(Grid):
    def OnOpen(self,event):
        style   = wx.OPEN | wx.CHANGE_DIR
        if hasattr(wx,'FD_PREVIEW'):
            style   |= wx.FD_PREVIEW
        dlg = wx.FileDialog(self, _("Choose an image"),
            defaultDir      = os.path.dirname(self.filename),
            wildcard        = self.wildcard,
            style           = style,
            )
        if dlg.ShowModal() == wx.ID_OK:
            self.OpenImage(dlg.GetPath())
        dlg.Destroy()
        
    def OnOpenUrl(self,event):
        dlg = wx.TextEntryDialog(self, _("Enter an image url"))
        if dlg.ShowModal() == wx.ID_OK:
            self.OpenImage(dlg.GetValue())
        dlg.Destroy()
        
    def OpenImage(self,filename):
        super(SingleGrid,self).OpenImage(filename)
        self.GetDialog()._OpenImage(filename)

    def SetFilter(self,filter=None):
        super(SingleGrid,self).SetFilter(filter)
        dlg = self.GetDialog().EnableImage(bool(self.data))
            
    def GetDialog(self):
        return wx.GetTopLevelParent(self)

class Browser(inspectorTag.Browser):
    ContentCtrl         = Grid
    
    def _init(self):
        self.GetContent().SetTag(SELECT)
        
    def GetPaintMessage(self):
        content     = self.GetContent()
        if not content.info:
            return _('drag & drop any image here')
        if content.tag_data:
            return _('broaden your search')
        tag = self.tag.GetStringSelection().lower()
        if tag == 'Exif' and not pyexiv2:
            return _('please install pyexiv2')
        return _('no %s tags found')%tag
    
class SingleBrowser(Browser):
    ContentCtrl     = SingleGrid
    
class SingleFrame(inspectorTag.Frame):
    Browser     = SingleBrowser
    def __init__(self,parent,filename='',*args,**keyw):
        super(SingleFrame,self).__init__(parent,[],TAGS, *args,**keyw)
        if not os.path.isfile(filename):filename = ''
        self.GetGrid().OpenImage(filename)
        self.filename   = '' #forces refresh of thumb
        
    def _OpenImage(self,filename):
        """To be called from the grid."""
        if filename.strip():
            di, ba      = os.path.split(filename)
            self.SetTitle('%s %s'%(ba,di))
        else:
            self.SetTitle(_('Image Inspector'))
        if self.thumb.IsShown():
            self.UpdateThumb()
            
    def _thumb(self):
        self.thumb_height   = 128
        size                = (self.thumb_height,self.thumb_height)
        bmp                 = wx.ArtProvider_GetBitmap(wx.ART_MISSING_IMAGE,
            wx.ART_OTHER,size=size)
        self.thumb      = wx.StaticBitmap(self.panel, -1, bmp, size=size)
        self.thumb.Hide()
        grid                = self.GetGrid()
        grid.SetAsFileDropTarget(self.thumb,grid.OnDrop)

        
    def _layout_top(self,main_sizer):
        self._thumb()
        main_sizer.Add(self.thumb, 0, wx.LEFT|wx.TOP, 8)
        
    def _layout_buttons(self,btn_sizer):
        self.preview    = wx.CheckBox(self.panel, -1, _("&Preview"))
        self.preview.Disable()
        self.open       = self.CreateBitmapButton(wx.ART_FILE_OPEN,
            _("Open image file"))
        self.open_url   = self.CreateBitmapButton(wx.ART_GO_HOME,
            _("Open image url"))
        self.copy   = self.CreateBitmapButton(wx.ART_COPY,
            _("Copy cell text"))
        self.copy.Disable()
        btn_sizer.Add(self.preview, 0, wx.ALL|wx.ALIGN_CENTER_VERTICAL, 4)
        btn_sizer.Add(self.open, 0, wx.ALL|wx.ALIGN_CENTER_VERTICAL, 4)
        btn_sizer.Add(self.open_url, 0, wx.ALL|wx.ALIGN_CENTER_VERTICAL, 4)
        btn_sizer.Add(self.copy, 0, wx.ALL|wx.ALIGN_CENTER_VERTICAL, 4)
        super(SingleFrame,self)._layout_buttons(btn_sizer)
        
    def _events(self):
        super(SingleFrame,self)._events()
        grid    = self.GetGrid()
        self.open.Bind(wx.EVT_BUTTON,grid.OnOpen)
        self.open_url.Bind(wx.EVT_BUTTON,grid.OnOpenUrl)
        self.copy.Bind(wx.EVT_BUTTON,grid.OnCopy)
        self.preview.Bind(wx.EVT_CHECKBOX,self.OnPreview)
        
    def OnPreview(self,event):
        state   = event.IsChecked()
        if state: self.UpdateThumb()
        self.thumb.Show(state)
        self.panel.Layout()
        
    def UpdateThumb(self):
        grid    = self.GetGrid()
        if self.filename != grid.filename:
            self.filename = grid.filename
            wx.BeginBusyCursor()
            bmp = bitmap_open(grid.filename,self.thumb_height)
            self.thumb.SetBitmap(bmp)
            size    = (bmp.GetWidth(),bmp.GetHeight())
            self.thumb.SetSize(size)
            self.thumb.SetMinSize(size)
            wx.EndBusyCursor()
        
    def EnableImage(self,state):
        if not hasattr(self,'copy'): return
        for ctrl in [self.copy,self.preview]:
            ctrl.Enable(state)
    
if __name__ == '__main__':
    app = wx.PySimpleApp()
    frame = SingleFrame(
        None,
        '',
        #'/home/stani/sync/photos/20070118-verhuis/IMG_1077.JPG',
        size=SIZE)
    frame.Show(True)
    app.MainLoop()
