# Phatch - Photo Batch Processor
# Copyright (C) 2007-2008 www.stani.be
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see http://www.gnu.org/licenses/
#
# Phatch recommends SPE (http://pythonide.stani.be) for editing python files.

# Embedded icon is taken from:
# http://www.iconlet.com/info/9657_colorscm_128x128
# Icon author: Everaldo Coelho
# Icon license: LGPL

from core import models
from lib.reverse_translation import _t

OPTIONS = [
    _t('Value'),
    _t('Top Left'),
    _t('Top Right'),
    _t('Bottom Left'),
    _t('Bottom Right')
]
#---PIL
def init():
    global Image, ImageOps, ImageMath
    import Image, ImageOps, ImageMath
    global HTMLColorToRGBA
    from lib.colors import HTMLColorToRGBA

def difference1(source, color):
    """When source is bigger than color"""
    return (source - color) / (255.0 - color)

def difference2(source, color):
    """When color is bigger than source"""
    return (color - source) / color


def color_to_alpha(image, color_value=None, select_color_by=None):
    image = image.convert('RGBA')
    width, height = image.size
    select = select_color_by
    color = color_value
    if select == OPTIONS[0]:
        color = HTMLColorToRGBA(color, 255)
    elif select == OPTIONS[1]:
        color = image.getpixel((0, 0))
    elif select == OPTIONS[2]:
        color = image.getpixel((width - 1, 0))
    elif select == OPTIONS[3]:
        color = image.getpixel((0, height - 1))
    elif select == OPTIONS[4]:
        color = image.getpixel((width - 1, height - 1))
    else:
        return image

    if color[3] == 0:
        # The selected color is transparent
        return image

    color = map(float, color)
    img_bands = [band.convert("F") for band in image.split()]

    # Find the maximum difference rate between source and color. I had to use two
    # difference functions because ImageMath.eval only evaluates the expression
    # once.
    alpha = ImageMath.eval(
        """float(
            max(
                max(
                    max(
                        difference1(red_band, cred_band),
                        difference1(green_band, cgreen_band)
                    ),
                    difference1(blue_band, cblue_band)
                ),
                max(
                    max(
                        difference2(red_band, cred_band),
                        difference2(green_band, cgreen_band)
                    ),
                    difference2(blue_band, cblue_band)
                )
            )
        )""",
        difference1=difference1,
        difference2=difference2,
        red_band = img_bands[0],
        green_band = img_bands[1],
        blue_band = img_bands[2],
        cred_band = color[0],
        cgreen_band = color[1],
        cblue_band = color[2]
    )

    # Calculate the new image colors after the removal of the selected color
    new_bands = [
        ImageMath.eval(
            "convert((image - color) / alpha + color, 'L')",
            image = img_bands[i],
            color = color[i],
            alpha = alpha
        )
        for i in xrange(3)
    ]

    # Add the new alpha band
    new_bands.append(ImageMath.eval(
        "convert(alpha_band * alpha, 'L')",
        alpha = alpha,
        alpha_band = img_bands[3]
    ))

    return Image.merge('RGBA', new_bands)

#---Phatch
class Action(models.Action):
    label       = _t('Color to Alpha')
    author      = 'Nadia Alramli'
    email       = 'mail@nadiana.com'
    cache       = False
    init        = staticmethod(init)
    pil         = staticmethod(color_to_alpha)
    version     = '0.1'
    tags        = [_t('color')]
    __doc__     = _t('Make selected color transparent')
    
    def get_relevant_field_labels(self):
        """If this method is present, Phatch will only show relevant
        fields. 
        
        :returns: list of the field labels which are relevant
        :rtype: list of strings
        
        .. note:: 
        
            It is very important that the list of labels has EXACTLY
            the same order as defined in the interface method.
        """
        relevant = ['Select Color By']
        if self.get_field_string('Select Color By') == OPTIONS[0]:
            relevant.append('Color Value')
        return relevant
    
    def interface(self,fields):
        fields[_t('Select Color By')] = self.ChoiceField(OPTIONS[0], choices=OPTIONS)
        fields[_t('Color Value')] = self.ColorField('#FFFFFF')
        
    icon = \
'x\xda\x01\xbf\x0c@\xf3\x89PNG\r\n\x1a\n\x00\x00\x00\rIHDR\x00\x00\x000\x00\
\x00\x000\x08\x06\x00\x00\x00W\x02\xf9\x87\x00\x00\x00\x04sBIT\x08\x08\x08\
\x08|\x08d\x88\x00\x00\x0cvIDATh\x81\xed\x9a{\x8c]\xd5u\xc6\x7fk?\xce\xb9\
\xf3\xf4k\x18\xe3\xd8c\x17cc\x9b\xb8\x10\x1e\xc5&&!\x18\x14\xa5U)\xadZ+\x84\
\x10(-\x8d\t\xa4r_\x11jQT\x11\x89V\xa8m*\x1aBqS\x05)\xb4\xae%@Q\xa2\xa8j\x0c\
\xb1\x12\x1a\x13J\xcb\xa3\xe0\xf2\x90]\xf0\x03<\xb6\x99\xc1\x1e\xcf\xeb\xde\
\xf3\xd8\xab\x7f\xecs\xe7\xdea\xc6\x8f4$\xfc\x93%m\xcf\xb9\xe7\x9c{\xee\xf7\
\xad\xef[k\xefs\x8e\xe1\xe7\xf1\xfe\x86\xbcW\x17\xba\x83rA\x03\xd9ha\xbd\xc0\
Z\x81\xe5@\xff\x93\xeb\x0e\xf0\xc6\x8aw\x8e\n\xbc\xae\xca\xee"\xe8\xd3y\xc1N\
}\xf4\x92\xe1\xf7\xe2w\x7fb\x02\x9fw\xe5-\xaar=\xca\xc7C\x80\x85\x03\xb0\xea\
c\xb0\xec2\x98\xb3\x04\xde\xee\xab3\xde]0\x96\x07v\xee\x1e\xe5\xbf\xf7M\xf2\
\xd2\xff\x8ea\x8d\xec\x10a\xfb\xc87.z\xe8}!p\x1b\xe1N\xd0-5\'\x8b\xae\xfe\
\x02\xac\xbe\x1a\x16_\x08\x1d}m\'\xa9\x82\xb4\xffD\xdc\x1eo\x04^:8\xc9\xaeW\
\xc7\xf8\xca\xbf\r\r\x1e\x1e\xce\xeekl\xbf\xf8\xde\x9f\t\x81\xcd\x84MV\xb8gN\
?+/\xff\x14\xfc\xea_C^\x87P\xcc~~n\x95`t\xd6c\xc6@O\xcd\xf2w;\x86\xd8\xfa\
\xdd\xa3{\x0e\x1c\xcd\xee\x1a\xfb\xa7\x8b\x1e\xf9\xa9\x11\xb8\xc3\x86\x07\
\xcbR7\x7f\xe4F\xe1\x13w\x83M\x80\xd9\xb1ME=-(\xfc\xa9O\x12 \x0f\xca\xbf<5\
\xc2\x9fo{sk\xb6\xfd\x92\xdb\xce\x14\xd3\x19\x11\xb8\x9dr)\x86m\xdd=\xb2a\
\xd3\x97\xe1\xdc\xab \x9f8\xb3\x1f\x18\xeb*\xc8\xd2\xf2\xf4\'*\xa4^\xd8{$\
\xe3\x8e\xaf\x1d\xdcuh8\xbba\xfc\x9f/:p\xba\xaf\x9d\x96\xc0\x93\xeb\xae^\xb3\
\xfd?\x1f\xffV\xff\x80\xae\xfc\xbd\xc7\x84\xb2\xe4\xb4Yo\x8f\x919\x19\x8d\
\x8e3 \xd0\x0c\x85\xdeN\xcb\xcd\xf7\x1f\xd8\xf3\xe2\x9b\xaf\\7\xfe\xd0u\xaf\
\x9c\xea\xf4S\x12\x08\xb7\xb3\x14\xcb\x13/\x8c\xdc\xbe\xb2\xf3\xd3_\xc5\xca\
\x8f\x81\x1c\x10\x85\xe1\x05\r&\xbaNR \xa7\x08c\x02\x7f\xf1\xe2\xa7\xf7|\xf7\
\xd5\xef\\\xc3\x179\xa9\x12\xeeT\x17Q\xcb6zzV^\xf4\xb9\r\x1c;\xfe\x1f\x1c<\
\xb4\x0ek\x02g\\:\n\x13]\xca\xb8\x0b?\x06tA\x99\xa0\xde\xf9#n\xfd\xd0\xcd+_\
\x1b\xde\xbdm\x1f\xfb\xae8\xf9\xd9\'\x89\xf0\xfb<\x88e3\xbf\xf30\x949"\x81c\
\xef\x9c\xc3\x1b\x07\xafB\xd03&1\xb8h\x92\xd19\xd9\x99aWA\xcd\x18\x93\xdd;\
\xa6v\xf5&\xbd|\xe6\x9b\x9f\xd9:\xf9\x85\xc9Y\x0b\xdb\xce\n\xfev6\xa9p\xaf|\
\xf4f\xe8\x98\x0be\x86\x169\x1d\xfe\x08\x89\x0cshp\x15\xa1\x84"?\xf5(3\x18\
\xee\xcc\x18\xb5\x05Y\xa1\xa7\x1d\xf5p\x82\x91\xceo\x92\x95\xd9\xd4\x18\xcdF\
Y\xd5\xbf\xea\xd2\xa7\x96=\xf52\xdf\xe7\xe5wc\x9d\xd5Bj\xb9G\x06\xce\x83\xb3\
\xcf\x87\xfa(\x10\xf3\xad@_\xcf\xb3\x9c\xb78\xe3\xe5\xd7~\x03k\x03\xe8\xa9\
\x95\x98\x9c\x0c\x8cw\x84iz\xcd\xac$!\x98\x11&z\x1fC\xb3l\x86\xb8\xfd\x9d\
\xfd\\\xb1\xfa\x8a{~\xc8\x0fg\xcc\x113\x14\x08\xb7q\xa7$\xe6\x93l\xb8)\xa60\
\xb4\x86\x84\x0c\xca\x9c\xee\xdaA:\xfc o\x1d\xfa\x10e\xae\x94\x85P\xe6\xc4Q\
\xb4F\xc8a\xb87c\xacV\x90\x05\xc8\xabQ\x94J^\x06\xf2R\xc9sh\xe81\x86{\xfe\
\x81Fh\x90\x85\xc64\x05\xb22c\xb2\x98dq\xcf\xe2\x05\xcf\xaf~\xbe^<Q\xec:\xb5\
\x02\x8e-\x9c\xb5\x04\x8a\x0c\xf2\xd9\xbd\x1b\x80\x85\xbd\xcf\xb0\xf6\x9c\
\x8c\xa7_\xb8\x15\x9fTJ\x08-\xa9\x88\xab\x88zC\xa9\xe7:\xb5s*\xfb\x1a\xb7K3\
\xc4\xb1\xce\x07!;}\x87[\xf3\x815[\x9e\xe3\xb9iK\x8ei\n\x94[\xe4\x16\t\xdc\
\xc8\xa5\xd7B\x08U\xe6\xf3\xe9C\x0bDsT\x0bz{\xf6\xd1\xdd1\xc8\xbe\xfd\x97\
\xa1A\tM%\xca6\x05\xfa\xea\x8cu\xe6\x94*\x94A)\x03\x14\x01\xca`\xa8\xf36G\
\xbb\xbeB\xa6u\xf2\x90\x91\x87|\xda(BA \xd0(\x1b\xbc9\xfa&\xa1\x0c=\x07.8p\
\x80\xef\xf3\xc2\xac\nH\xd0\xeb9kaD\x90\x8fM\xa7o\x88\xe95\x80\x11D@\x830p\
\xf6\x0f\xe0\xc2Q\xfe\xfd\x99;\x91f\x8b\xad<,\n\x8d\x0c\xb2\xa2\xa5@L\xbeP\
\xc8 ou|\t\xc9g\xef\xe4\xa5\x96\x9c\xa8\x9f\xe0\xb9\xc1\xe7\x18~g\xb8\xa5\
\xae\xe5z`j\x05;U.\xe5_.[ \x87\xf6\x0f\xb1\xfc\x02\x98\xbf\xa4\x8dU\x0b4\xa6\
:\xddT \x8dD\xe7\x18e\xdf\xc1u\xecx\xf2\xcfH\xd3\xd6<!\xc0\xfe\x0bG8\xb6\xa4\
^\x01\x8f\xffd\xf2\x16\x07:\xeeF\x99>C{\xeb\x19<1\xc8k\x87_c<\x1b\xa7\xc8\
\x8b\x16\xf0\xf6P\xfa\xf8"\xc3\xd3\x14\x90=\xfb7j\x17H\x92B6^}\xb1\x02\xaf&\
\x1a\xdfV$\xa60\nR}>g\xd9\x13\\\xb3\xa1\xc1\xe3O\xde\x8div\'\x85,W\xf2\xa29\
\x91Y\x1a\x1c`o\xfa\xa7hVT\xf9\x11\x8aP0<6\xcc\xde\xa1\xbd\xe4\xf5<\xfe\xa64\
U\x9f56\x02\x8fL#\x80g\xbd@\\\xe3\xe6\x13U\xd6M\x95\xb6\x8aHhS\x00i}\x06T\
\x85s\x7f\xe1{\xa0c|\xe7\x89\xbf\xc1\x98\x80 \xe4e\xec< \xd4e\x1f\xbb\xddg1y\
R%R9:z\x94\x83o\x1fdJ\xb2Yg\xa6w\x85\xb0~&\x01a-\xb5$f\xbf\xe9\xf7\x88v:x\
\x95\x16!\x91V\xf7A\x08\x01\xce]\xfe\x03>\xb1\xf1\x8fx\xf4\xdb_\xa6#U\x8a\
\x12J\x0c\x93\xec\xe5\xc5\xf0yB\xa3\xc0\x1aeh|\x88\xa3\xc7\x8fR\x96e\x0b\xfc\
\x99\xc7\xda\xe6F\x8b\x80e9\x1d)\x94\x931\xeb\xd2\xd4O\xa0\x04\xaci\x81m\x92\
\xd0\x8aD\x9b\xe75\xc0\xaaU\xff\xcao^[\xe7\xb1o?@YZ\xc6\xc2^\x9e\xd7\x9bQ\
\x94F\x91q\xf8\xc4\x11\xb2F1\x1d\xb82\xfd\xf3\xa9cys\xa3\xe52C?\xceC\x98\x84\
P\xaf\x88\xd4[#\xd4A\xab}T\xdbT\xc7\xa8\xb6i\x00\r\xb4PV\x9d\xf7=~\xeb\xda\
\x9b\x18j\xec\xe7G\xe1\xd7\xc8\x99dh\xe2\x08\x87N\xbcE\xa0\xc0&`\x1d\x18G\
\xb4\x8d=c\xf0\x00\xfd\xcd\x8d\x96\x02\xff\x03,\xc8\xa0V\x15\xaa\xablbMU\xbc\
\xc4\xfdV\xaa\x1f\x93X#M+5\x0bO\xa2J*\xb0\xaas\'\xe7^0\xc0\xe3\xe1SL\xd4\x03\
\x85\xcbq\xdd\x11\xa5\x86x\xcb\xac\x80\x84XC\x1a\xe2~\xa6w\xdd\x93\xc4\xd7\
\xdeE\xc0p\x94\xb2\xe8%\xaf\x80\x86\nh\x10\x08\xa6E \x98\xcaR2\x1d|\xb3\xb56\
\xbbS\x17\xf0\xc1\xc0Y\xc7\x96\xb1b\xf2\x1a^r\x8f\xe2m-\x82V\x05+\x04\x8d\
\x80\x8d%N\x84!\x92\x08A\xe29\xa1i\xd3\x19\xe8\x8f\xceT\x00^\xa7\xccWP\x84\
\x08\xae\x99\xc6\xe6\xb7U"\xd8P\xedkz\xdfT\xc7T\x99\x92\xa1\x0b\xb8\x04\x089\
\xc1d,\xf5\xabI\xe5\x93<k\x1f\xc6\x86\x04\xa4\x02\xaa\xb1\x13i)\xa8\x11BEB4\
\x10JC@\xe3\xb5g\xde\x82\xbc\xde\xca{kkw\xccd\x88C\x02\xd8j\xdb(X\x8d\xfbD[\
\xe7\x98\x10\xf7\x9b\xc0\xd4w{\x02\\Z\x1dK\x04\xe7\rI",skX/7"I\x1dg\x15\xe7\
\x03\xde\x05\xbc\x05\xe7\x15k\xc1Zp\x16\x9c\x15\x9c\x0bXW\xf5\x123\xc3R\xbbg\
* <\x8d\xad\xac`MUX\x95\xe7\x9d\xb4f\xe2\xf6\xcf\xed\xb5a\x04\xba\x05~i\xeaz\
`\x14\x9f\x18\x92""X\xc6Z\x12n\xe1Y\xb3\r\xa5e\xa1\x10\x94 J\xd0\xd8\x8a\xa5\
T\x04\x01\xd1(zY\t_6\xbb\xa0>=C\x01\x1d\xe8\xdb9\r\xe0\x14x\xd3\x02\xeb\xab\
\xe1L\xa4\xee\x04\xbc\x05o`\x0epy\x95\x92D\xaaap\x89\x90T\xc3\xf9\xc0\x12\
\xbf\x9a\xf5\xf6F\xb0\x8d\x98}\x07\xde\t\xceIT\xc6*\xce\x0b\xd6)\xd6\x08\xce\
)\xc6)\xc6HT;*\xb1s\x06\x01\xfb\x8fC\xc3X\xd91\xa5\x807m\x9d\xa8\t\xba\xda\
\xe7\xa8\x80W\x17\x9d\x03\xac\xab\x8e\'\x06\xd2\x08\x1eop\xde\x90zC\xe2\ribH\
\x9d\xb049\x9f\x8f\xf8[\x10\xdf\xc0\xbbh\x1b\xef\x14\xe7\x9a\xc0\x03\xce\x81\
\xad\x80[#xO$at\x07\x7f\xf2\xd5\xe1\x19\x04\x00\xd4\x99\xedS\xd6I\xd3V\x0b\
\xf5\x15\xe8\xa6:\xaeM\xa5y\x066\x18H\xa7\x03\'1\x90H\xb4PE"\xad\xea\xc1\xfb\
\xc02\xbf\x96+\xed\xad8\x17\xf0I\xac\x01o[jX\xab8\x1bI\xd4RG\xe2\x1d\xd6*\
\xc6\xca\xf6v\xcc\xd3\x08\xd8\x87\x1b\x0f\xa9\x95A\x8cBZ\x83\xee\xee\x16p+-\
\xdb\xb8\xaa\xc5\xce5\xf1Yt\x93d\x05\xba\xddB\xde[j\x89!\xf5\x96\xc4\xcb\x94\
\x1a\xce\x95,\xf6\xab\xb8\xc2\xff6\xea\xeaQ\x01\xaf1gV\xb1\x15\x89\x9as\xf4\
\xd4j$\xce`\xac\x0e\x86-\xf7?tR\x02\x00\x12\xb2\xfb\x10 q\xd0\xd9\x05\xbd\
\xf3\xda2\xdf\x9c\xd4$\x82_\'1\xdb\xbe\x02\xee\x9a$l\xf5\xb7\xaa\x01/\xa4\t\
\xd1BIS\rK\x9a\x18\x06\xfc\xf9|\xcc}\x16\xe3\x8a\xd8\x81\x9c\xe2\xad\xe0\x8c\
\x92z\xc3Y\xdd\xddtzG\xa6\r\x8c\xe1\xbew\xe3\x9dA\xc0<\x14\xeeU#{\xa8\x9f\
\x80\xb4\x03z\xe6BO\xef\xf4\xc2\x9eo`C\xb3\xa8\xab\xbah\xda\xc6\xb7Y(\x8d\
\x16j\x07\x9e\xb8fQ\x1b\xbc\x17\xbcS\x06\xdc\x07\xb9\xd2o\xae\xc0\x83u\x81\
\x9aw,\x9b\xb3\x90\xce$E\xa5\xc0\x88\xee\x99\xdc\xfc\xc0\x8c\'\xd8\xb3\xae\
\xb8E\xca\xbb\x942\xae\x87\xd2\x0e\x98\xbf\x18\xfa\x06boof^*\x9bx\x89\xde\
\xf7\xa6\xb2Q\xdb\xf0\x0e\xeb,Ie\x9b$1\xa4\xa9!u\x86\xd4C\xe2cq:\x1fX\xecVr\
\x85\xfb]\x82\x9d\xa0\'\xa9\xb1r\xdeB:\x13KW\xe29\x91\x8db-w\xcd\x86uV\x02\
\xe6\xfe\xfa#bd+cC\xb1\x90\xd3\x1at\xf7\xc1\xca_\x84\xcb\xbb\xe2\xc4\xe6\xdb\
\xba\x925S\x9e\xafR\x08\x92\x00\x1e\xef,\xb5f\x07\xf2B\xe2"\x91\xc4\xdbh#g\
\xf0\xce\xe0\x9d0\xe0\xd7\xb0i\xce\x1f\xb2j\xde":}B\x97K821\x0cF\xb7\xbes\
\xd3\x03\xb3>v?\xe9=\x8f\xf9\xab\xd1\xdb\xd4\xc8.\x86\xf6\x82K`^\x07\\\xdc\
\x0b\xbd\x17A\xcfy\xe0}k^hZG*\xe0\x92\x82t\x82t\xe1\x9c\'M\x85\xd4\xc5\x02\
\xae%\x86\xc4K\xb4\x92\x93\xa96\xdb\xdbaX\xde\'\xac\xeeZ\xc9z\xff\xebt\'\tc\
\xe5\x18\x93ab\xd7\x91\xeb\xff\xfe\xa4\x8f\xdbO~\xd3\x06\x98b\xec\x064\xeca\
\xfc%XS\x03:\xc1y\xa8\r@\xf7\x87\xa1\xf3RH\xe6\xc5%\x076\x02\xd7\x1a\xd0\rt\
\x81\xe9\xc1\xfb4\x02O\x9b\xf3A|\x8c\xee\xbd\x90z\xe1\x03=\x86\x15\xf3-K\xe7\
Y:\xacPK`a\xb2\x84\xe5\xc5\xd5\xbc1\xbeo\x8f\xb5\xdcp*\x8c\xa7]\x81\xeb\xdfv\
\xac\t\x97w\x7fKj\xdd+\x99\xbb\x0e|G\xdb\x9c`\xe2\x02F\x05t\xa4\xba7\x08\xd5\
0\x08\x81\xffj\xfc\x01\x87\xf2+1&.\xa3\xe2j\\\xe3\xd4\xa2\x90\x97J\x11\x02e\
\xa9\xd5\xe3\x16\xe5\xd0H\xc9\xe0\x08{\xde\xca_\xb9\xee\xeb\xbf|\xd9\xff\xff\
\xf1z3\xc23}K51\xdb$1\x1b\x98{5\xa4\xbd\xd1\xf7F\xa2m\x8c\x07\x12 \xad\xbc_\
\x03<\x88\xe3\xcd\xf2*F\xc2\nT5.\xf3\x15\xca\x00A\x95\xb2\x80"(eP\x8a2P\x04\
\xd8w\xbc\xe0\xed\xd1\xb0+\xcb\xc3\r_\xda8\xf7\'\x7f\xc1\xd1\x1e\xe5\xab\x8b\
\x1e\x14/\x9b\xe9\\\x02]W\x026.\xe6\xa5\t>\x8d\x7f\xa9U\xfb\x1c\x87\xc3zN\
\xe8\x00\x8a\xc4\x15\xb7B\xa1\x91@(#\x01U\xe5\xed\xf1\x92\x03#\x81\xc9,l\xfd\
\xe3\xcb\xba\xdf\xdbWL\xed\xa1\xfb\xcf\xde\x14\x9c\xbbG\x12Y\x89\xbb\x18\xfc\
Z0\xfd\x80\xab\x08t\xb4\xc8\x88g(\\\xc0\xb8\x9e=u\x93\x154\x1aLC|/v\xa2\xae\
\x1c\x9e(96\xa1{\xf2"\xdc\xf5\xb9\x0b;\x7fz/\xf9\xda#\x0c-\xbb\x13\xd2-Hm\
\x11v1\xf8\x8f\x82Y\x06\xd2\x11;\x105\x90\x94\xe3\xba\x82\x86\xce\x8f\xcfA5\
\x8e\x86\xc2\xf1\x1c\x0eO\x06\xf2\x82\xc1,\x0f\xf7\xdd\xb0<\xf9\xd9\xbcf\x9d\
A\xe4\x9d\x8boQ\x99s\xbdH\xef\xc71\xf3A\xce\x06\xdb\x0f\xd2\x07\xd2\xcb\x90^\
\xca\xb8."(\xd4\x15&K\x98(\xa1\x11\xd8\x91\x07\xdd\xfe+\xfd\xf6\xfdy\xd1\xfd\
\xee(G6-0\xa6w#\xa6o\xbdJ\xffZ\x95\xfe\xe5"g\xf5\x1f\xe7br\xe9;\x9a\x07}}Re\
\xf7x)OO\x04v~x\x8e\xbc\'\xff\xd5\xe0\xe7\xf1~\xc7\xff\x01\xe7GL\xbd\x94q\
\xf9$\x00\x00\x00\x00IEND\xaeB`\x82Y\xe3<\xfc'
