#!/usr/bin/python

# Phatch - Photo Batch Processor
# Copyright (C) 2009 www.stani.be
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see http://www.gnu.org/licenses/
#
# Phatch recommends SPE (http://pythonide.stani.be) for editing python files.
#
# Follows PEP8

import os
import re
from operator import itemgetter

ERROR_KEYS = ('count', 'error', 'description', 'actionlist', 'action',
    'traceback')
LOGS = 'output/logs.txt'
RE_ERROR = re.compile('20[0-9]{2}.+?actionlist:\s*(?P<actionlist>.+?)\s*'\
'generated.+?logs:\n.+?:(?P<description>.+?)\s*Action:(?P<action>.+?)\s*'\
'Traceback.+?:\n\s*(?P<traceback>.+?\n[A-Z].+?:\s*(?P<error>.+?))\n[*]',
re.DOTALL|re.MULTILINE)
SEPARATOR = '-'*40+'\n'
OUTPUT = ''


def output(filename=''):
    """Gives the full path for an output file

    :param filename: filename without path
    :type filename: string
    :returns: full output path
    :rtype: string

    >>> output('report.csv')
    'images/output/report.csv'
    """
    return os.path.join(OUTPUT, filename)


def parse_errors(filename):
    """Parses the log file and convert all errors to dictionaries.

    :param filename: logs filename
    :type filename: string
    :returns: all errors
    :rtype: list of dict
    """
    f = open(filename)
    source = f.read()
    f.close()
    return [match.groupdict() for match in RE_ERROR.finditer(source)]


def count_errors(errors):
    """
    :param errors: errors
    :type errors: list of dictionaries
    :returns: one example of each error with its frequency
    :rtype: dictionary
    """
    count = {}
    for error in errors:
        name = error['error']
        if name in count:
            count[name]['count'] += 1
        else:
            example = error.copy()
            example['count'] = 1
            count[name] = example
    counted_errors = count.values()
    counted_errors.sort(key=itemgetter('count'), reverse=True)
    return counted_errors


def write_csv(filename, counted_errors):
    """Write error in descending frequency to a csv file.

    :param filename: filename of the csv file
    :type filename: string
    :param counted_errors: errors counted by :func:`count_errors`
    :type counted_errors: list of dictionaries
    """
    f = open(filename, 'w')
    f.write('count, error, actionlist\n')
    for error in counted_errors:
        f.write('%(count)02d, %(error)s, %(actionlist)s\n'%error)
    f.close()


def write_txt(filename, counted_errors):
    """Write error in descending frequency to a txt file.

    :param filename: filename of the csv file
    :type filename: string
    :param counted_errors: errors counted by :func:`count_errors`
    :type counted_errors: list of dictionaries
    """
    f = open(filename, 'w')
    f.write('count, error, actionlist\n')
    for error in counted_errors:
        for key in ERROR_KEYS:
            f.write('%s = %s\n\n'%(key.upper(), error[key]))
        f.write(SEPARATOR)
    f.close()

WRITE = {
    'csv': write_csv,
    'txt': write_txt,
}


def main(logs=LOGS):
    errors = parse_errors(logs)
    counted_errors = count_errors(errors)
    for ext, write in WRITE.items():
        logs_filename = output('%s_count.%s'%(logs, ext))
        write(logs_filename, counted_errors)
        print('%d different errors written to "%s"'
            %(len(counted_errors), logs_filename))


if __name__ == '__main__':
    main()
