# ***** BEGIN LICENSE BLOCK *****
# Version: RCSL 1.0/RPSL 1.0/GPL 2.0
#
# Portions Copyright (c) 1995-2002 RealNetworks, Inc. All Rights Reserved.
# Portions Copyright (c) 2006 Lukas Lalinsky. All Rights Reserved.
#
# The contents of this file, and the files included with this file, are
# subject to the current version of the RealNetworks Public Source License
# Version 1.0 (the "RPSL") available at
# http://www.helixcommunity.org/content/rpsl unless you have licensed
# the file under the RealNetworks Community Source License Version 1.0
# (the "RCSL") available at http://www.helixcommunity.org/content/rcsl,
# in which case the RCSL will apply. You may also obtain the license terms
# directly from RealNetworks.  You may not use this file except in
# compliance with the RPSL or, if you have a valid RCSL with RealNetworks
# applicable to this file, the RCSL.  Please see the applicable RPSL or
# RCSL for the rights, obligations and limitations governing use of the
# contents of the file.
#
# This file is part of the Helix DNA Technology. RealNetworks is the
# developer of the Original Code and owns the copyrights in the portions
# it created.
#
# This file, and the files included with this file, is distributed and made
# available on an 'AS IS' basis, WITHOUT WARRANTY OF ANY KIND, EITHER
# EXPRESS OR IMPLIED, AND REALNETWORKS HEREBY DISCLAIMS ALL SUCH WARRANTIES,
# INCLUDING WITHOUT LIMITATION, ANY WARRANTIES OF MERCHANTABILITY, FITNESS
# FOR A PARTICULAR PURPOSE, QUIET ENJOYMENT OR NON-INFRINGEMENT.
#
# Technology Compatibility Kit Test Suite(s) Location:
#    http://www.helixcommunity.org/content/tck
#
# --------------------------------------------------------------------
#
# picard is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# picard is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with picard; if not, write to the Free Software
# Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
#
# Contributor(s):
#   Lukas Lalinsky
#
#
# ***** END LICENSE BLOCK *****

class PlaylistExcetion(Exception):
    pass

class Playlist:
    """Playlist (M3U/PLS/XSPF) generator. 
    """
    
    formats = [
      ("*.m3u", "M3U (*.m3u)"), 
      ("*.pls", "PLS (*.pls)"),
      ("*.xspf", "XSPF (*.xspf)"), 
    ]
    
    def __init__(self, parent, album):
        self.parent = parent
        self.album = album
       
    def saveM3U(self, w):
        """M3U playlist generator
        """
        w(u'#EXTM3U')
        for i in range(self.album.getNumTracks()):
            track = self.album.getTrack(i)
            w(u'#EXTINF:%d,%s - %s' % (track.getDuration() / 1000, track.getArtist().getName(), track.getName()))
            fileId = track.getFileId()
            if fileId >= 0:
                tr = self.parent.tunePimp.getTrack(fileId)
                tr.lock()
                w(u'%s' % tr.getFileName())
                tr.unlock()
                self.parent.tunePimp.releaseTrack(tr)
        
    def savePLS(self, w):
        """PLS playlist generator
        """
        w(u'[playlist]')
        for i in range(self.album.getNumTracks()):
            track = self.album.getTrack(i)
            fileId = track.getFileId()
            if fileId >= 0:
                tr = self.parent.tunePimp.getTrack(fileId)
                tr.lock()
                w(u'File%d=%s' % (i + 1, tr.getFileName()))
                tr.unlock()
                self.parent.tunePimp.releaseTrack(tr)
            w(u'Title%d=%s - %s' % (i + 1, track.getArtist().getName(), track.getName()))
            w(u'Length%d=%d' % (i + 1, track.getDuration() / 1000))
        w(u'NumberOfEntries=%d' % self.album.getNumTracks())    
        w(u'Version=2')    
        
    def saveXSPF(self, w):
        """XSPF playlist generator
        
        See http://www.xspf.org/ for more info. 
        """
        
        def escape(string):
            string = string.replace('&', '&amp;') 
            string = string.replace('<', '&lt;')
            string = string.replace('>', '&gt;')
            return string
        
        album = self.album
        w(u'<?xml version="1.0" encoding="UTF-8"?>')
        w(u'<playlist version="1" xmlns="http://xspf.org/ns/0/">')
        w(u'\t<identifier>http://musicbrainz.org/album/%s</identifier>' % \
          escape(album.getId()))
        w(u'\t<trackList>')
        for i in range(album.getNumTracks()):
            track = album.getTrack(i)
            w(u'\t\t<track>')
            w(u'\t\t\t<title>%s</title>' % \
              escape(track.getName()))
            w(u'\t\t\t<creator>%s</creator>' % \
              escape(track.getArtist().getName()))
            w(u'\t\t\t<album>%s</album>' % \
              escape(album.getName()))
            w(u'\t\t\t<trackNum>%d</trackNum>' % track.getNum())
            w(u'\t\t\t<duration>%d</duration>' % track.getDuration())
            w(u'\t\t\t<identifier>http://musicbrainz.org/track/%s</identifier>' % \
              escape(track.getId()))
            fileId = track.getFileId()
            if fileId >= 0:
                tr = self.parent.tunePimp.getTrack(fileId)
                tr.lock()
                w(u'\t\t\t<location>file://%s</location>' % \
                  tr.getFileName())
                tr.unlock()
                self.parent.tunePimp.releaseTrack(tr)
            w(u'\t\t</track>')
        w(u'\t</trackList>')
        w(u'</playlist>')
        
    def save(self, fileName, format):
        album = self.album
        
        lines = []
        w = lines.append

        if format == 0:
            self.saveM3U(w)
        elif format == 1:
            self.savePLS(w)
        elif format == 2:
            self.saveXSPF(w)

        f = file(fileName, 'wt')
        f.write(u'\n'.join(lines).encode('utf-8'))
        f.close()
            

