# -*- coding: utf-8 -*- 

# vim:set shiftwidth=4 tabstop=4 expandtab textwidth=79:
#Copyright (c) 2005 Ali Afshar aafshar@gmail.com

#Permission is hereby granted, free of charge, to any person obtaining a copy
#of this software and associated documentation files (the "Software"), to deal
#in the Software without restriction, including without limitation the rights
#to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
#copies of the Software, and to permit persons to whom the Software is
#furnished to do so, subject to the following conditions:

#The above copyright notice and this permission notice shall be included in
#all copies or substantial portions of the Software.

#THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
#IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
#FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
#AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
#LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
#OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
#SOFTWARE.

import os
from rat import hig
import pida.core.service as service
import pida.core.actions as actions
import pida.pidagtk.contentview as contentview

defs = service.definitions

try:
    import moo
except ImportError:
    moo = None

import gtk


class MooView(contentview.content_view):

    HAS_TITLE = False

    HAS_CONTROL_BOX = False

    def init(self, filename=None):
        self.widget.set_border_width(6)
        sw = gtk.ScrolledWindow()
        self.widget.pack_start(sw)
        self.editor = self.service.editor_instance.create_doc(filename)
        sw.add(self.editor)

        self._modified = False
        self.editor.connect('doc-status-changed', self.cb_doc_status_changed)

    def cb_doc_status_changed(self, ed, *args):
        self._modified = ed.get_status() & moo.edit.EDIT_MODIFIED
        self.service.set_status_sensitivities(self, ed)
        self.service.set_save_sensitivities(self, ed)


class MooConfigView(contentview.content_view):

    LONG_TITLE = 'Moo Editor Preferences'
    HAS_CONTROL_BOX = False
    HAS_TITLE = False

    def init(self):
        prefs = self.service.editor_instance.prefs_page()
        prefs.set_no_show_all(True)
        prefs.emit('init')
        self.widget.pack_start(prefs)
        self.widget.set_border_width(6)
        self.widget.set_spacing(6)
        self.__prefs = prefs

        bb = gtk.HButtonBox()

        bb.set_spacing(6)
        bb.set_layout(gtk.BUTTONBOX_END)

        self._apply_but = gtk.Button(stock=gtk.STOCK_APPLY)
        self._apply_but.connect('clicked', self.cb_apply_button_clicked)

        self._ok_but = gtk.Button(stock=gtk.STOCK_OK)
        self._ok_but.connect('clicked', self.cb_ok_button_clicked)

        self._cancel_but = gtk.Button(stock=gtk.STOCK_CANCEL)
        self._cancel_but.connect('clicked', self.cb_cancel_button_clicked)

        bb.pack_start(self._apply_but)
        bb.pack_start(self._cancel_but)
        bb.pack_start(self._ok_but)

        self.widget.pack_start(bb, expand=False)

    def cb_ok_button_clicked(self, button):
        self._apply()
        self.remove()

    def cb_apply_button_clicked(self, button):
        self._apply()

    def cb_cancel_button_clicked(self, button):
        self.remove()

    def _apply(self):
        self.__prefs.emit('apply')
        self.service.save_moo_state()

class MooEditor(service.service):

    NAME = 'mooedit'

    display_name = 'Moo'

    window = property(lambda self: self.boss.get_main_window())

    def init(self):
        self._init_moo()
        self.__files = {}
        self.__documents = self.__files
        self.__views = {}

    def save_moo_state(self):
        moo.utils.prefs_save(self._rc_path, self._state_path)

    def _init_moo(self):
        self._rc_path = os.path.join(self.boss.get_pida_home(), 'pida_moo.rc')
        self._state_path = os.path.join(self.boss.get_pida_home(), 'pida_moo.state')
        moo.utils.prefs_load(self._rc_path, self._state_path)
        self.editor_instance = moo.edit.create_editor_instance()
        moo.edit.plugin_read_dirs()

    def reset(self):
        self.__bind_document_actions()

    class Moo(defs.View):
        view_type = MooView
        book_name = 'edit'

    class MooPrefs(defs.View):
        view_type = MooConfigView
        book_name = 'ext'
    
    @classmethod
    def get_sanity_errors(self):
        """I think this should raise"""
        if moo is None:
            return ['Moo is not installed']

    def cmd_edit(self, document=None):
        if document.unique_id not in self.__views:
            self.__load_file(document)
        self.__view_file(document)

    def cmd_revert(self):
        #self.__currentview.editor.reload()
        pass

    def cmd_grab_focus(self):
        if self.__currentview is not None:
            self.__currentview.editor.grab_focus()

    def __load_file(self, document):
        view = self.create_view('Moo', filename=document.filename)
        self.__views[document.unique_id] = view
        self.__documents[view.unique_id] = document
        self.show_view(view=view)

    def __view_file(self, document):
        self.__currentview = self.__views[document.unique_id]
        self.__currentview.raise_page()
        self._set_action_sensitivities(document)

    def _set_action_sensitivities(self, document):
        view = self.__views[document.unique_id]
        if document.is_new:
            self._save_action.set_sensitive(True)
            self.__revertact.set_sensitive(False)
        else:
            self.set_save_sensitivities(view, view.editor)
        self.set_status_sensitivities(view, view.editor)
        

    def set_status_sensitivities(self, view, ed):
        if view is self.__currentview:
            self.__undoact.set_sensitive(ed.props.can_undo)
            self.__redoact.set_sensitive(ed.props.can_redo)

    def set_save_sensitivities(self, view, ed):
        if view is self.__currentview:
            sens = False
            if not ed.props.can_undo:
                sens = False
            else:
                sens = view._modified
            self._save_action.set_sensitive(sens)
            self.__revertact.set_sensitive(sens)
            

    def __bind_document_actions(self):
        doctypes = self.boss.get_service("documenttypes")
        for action in doctypes.cmd_get_document_actions():
            actname = action.get_name()
            if actname == 'documenttypes+document+undo':
                self.__undoact = action
            elif actname == 'documenttypes+document+redo':
                self.__redoact = action
            elif actname == 'documenttypes+document+revert':
                self.__revertact = action
            elif actname == 'DocumentSave':
                self._save_action = action

    def cmd_close(self, document):
        v = self.__views[document.unique_id]
        self.close_view(v)
        self.current_view = None

    def cmd_can_close(self):
        buffs = [view.editor for view in self.__views.values()]
        
        buffs = filter(lambda val: val.get_buffer().get_modified(), buffs)

        # If we have no buffers to save, go on
        if len(buffs) == 0:
            return True
            
        filenames = dict((buff.get_filename(), buff) for buff in buffs)
        files, response = hig.save_changes(filenames.keys(), parent=self.window)
        # Save the files that need to be saved
        errors = []
        for filename in files:
            # XXX: handle new files
            try:
                filenames[filename].save()
            except:
                errors.append(filename)
        if len(errors) > 0:
            # XXX: handle errors
            pass
        return response in (gtk.RESPONSE_CLOSE, gtk.RESPONSE_OK)

    def view_closed(self, view):
        if view.unique_id in self.__documents:
            doc = self.__documents[view.unique_id]
            del self.__documents[view.unique_id]
            del self.__views[doc.unique_id]
            self.boss.call_command('buffermanager', 'document_closed',
                                   document=doc)

    def cmd_start(self):
        self.get_service('editormanager').events.emit('started')

    def cmd_goto_line(self, linenumber):
        self.__currentview.editor.move_cursor(linenumber, 0, False, True)

    def cmd_save(self):
        self.__currentview.editor.save()
        self.boss.call_command('buffermanager', 'reset_current_document')

    def cmd_undo(self):
        self.__currentview.editor.undo()

    def cmd_redo(self):
        self.__currentview.editor.redo()

    def cmd_cut(self):
        self.__currentview.editor.emit('cut-clipboard')

    def cmd_copy(self):
        self.__currentview.editor.emit('copy-clipboard')

    def cmd_paste(self):
        self.__currentview.editor.emit('paste-clipboard')

    def act_editor_preferences(self, action):
        v = self.create_view('MooPrefs')
        self.show_view(view=v)

    @actions.action(default_accel='<Control>f', label='Find')
    def act_find(self, action):
        self.__currentview.editor.emit('find-interactive')

    @actions.action(default_accel='F3', label='Find Next')
    def act_find_next(self, action):
        self.__currentview.editor.emit('find-next-interactive')

    @actions.action(default_accel='<Shift>F3', label='Find Previous')
    def act_find_prev(self, action):
        self.__currentview.editor.emit('find-prev-interactive')

    @actions.action(default_accel='<Ctrl>r', label='Find and Replace')
    def act_replace(self, action):
        self.__currentview.editor.emit('replace-interactive')
    
    @actions.action(default_accel='F4', label='Find Word Under Cursor')
    def act_find_current(self, action):
        self.__currentview.editor.emit('find-word-at-cursor', True)

    @actions.action(default_accel='<Shift>F4', label='Find Word Under Cursor Backwards')
    def act_find_current_back(self, action):
        self.__currentview.editor.emit('find-word-at-cursor', False)

    def get_menu_definition(self):
        return """
                <menubar>
            <menu name="base_edit">
                <placeholder name="EditSearchMenu">
                    <separator />
                    <menuitem name="MooFindToggle"
                        action="mooedit+find" />
                    <menuitem name="MooFindNextToggle"
                        action="mooedit+find_next" />
                    <menuitem name="MooFindPrevToggle"
                        action="mooedit+find_prev" />
                    <menuitem name="MooReplaceToggle"
                        action="mooedit+replace" />
                    <separator/>                
                    <menuitem name="MooFindCurrentToggle"
                        action="mooedit+find_current" />
                    <menuitem name="MooFindCurrentBackToggle"
                        action="mooedit+find_current_back" />
                </placeholder>
            </menu>
                <menu name="base_edit" action="base_edit_menu">
                    <placeholder name="PreferencesMenu">
                        <separator />
                        <menuitem name="mooconf" action="mooedit+editor_preferences" />
                    </placeholder>
                </menu>
                </menubar>
        """
				

Service = MooEditor
