# -*- mode: python; coding: utf-8 -*-
#
# Pigment Python tools
#
# Copyright © 2006, 2007 Fluendo Embedded S.L.
#
# This library is free software; you can redistribute it and/or
# modify it under the terms of the GNU Lesser General Public
# License as published by the Free Software Foundation; either
# version 2 of the License, or (at your option) any later version.
#
# This library is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
# Lesser General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public
# License along with this library; if not, write to the
# Free Software Foundation, Inc., 59 Temple Place - Suite 330,
# Boston, MA 02111-1307, USA.

import pgm
from pypgmtools.widgets.list import List

VERTICAL, HORIZONTAL = 0, 1

class CurvedList(List):

    def __init__(self,
                 canvas,
                 layer,
                 width             = 1.0,
                 height            = 2.0,
                 visible_items     = 7,
                 orientation       = VERTICAL,
                 selected_item     = 0,
                 selector_file     = None,
                 selector_offset   = (0.0, 0.0, 0.0),
                 selector_size     = (1.0, 1.0)):

        self._ratio            = 4.0/3.0

        List.__init__(self, canvas, layer,
                      width, height,
                      visible_items, orientation,
                      selected_item, selector_file,
                      selector_offset, selector_size)



    def _compute_position(self, index):
        # it depends on the zoom
        zoom = self._compute_zoom(index)

        # floor division
        middle_index = self._visible_items // 2 + 1

        tw = self._width
        th = self._height

        # x/y_top_left_offset are the offset between the top left corner and
        # the center of a drawable
        # x and y are the coordinates of the center of the drawable
        if self._orientation == VERTICAL:
            h = self._widget_height*zoom
            w = h*self._ratio
            x_top_left_offset = -w/2.0
            y_top_left_offset = -h/2.0

            if index == middle_index:
                y = self._height / 2.0
                x = 0.0
                z = 1.0
            elif index < middle_index:
                y = sum([self._compute_zoom(i)*self._widget_height for i in xrange(0, index)]) - \
                    self._compute_zoom(0)*self._widget_height + h/2.0
                x = 4.0*(tw+w/2.0)/(th*th)*(y-th/2.0)*(y-th/2.0)
                z = 0.0
            elif index > middle_index:
                index = self._visible_items + 2 - index
                y = self._height - \
                    sum([self._compute_zoom(i)*self._widget_height for i in \
                    xrange(0, index)]) + self._compute_zoom(0)*self._widget_height \
                    + h/2.0

                x = 4.0*(tw+w/2.0)/(th*th)*(y-th/2.0)*(y-th/2.0)
                z = 0.0


        elif self._orientation == HORIZONTAL:
            w = self._widget_width*zoom
            h = w/self._ratio
            x_top_left_offset = -w/2.0
            y_top_left_offset = -h/2.0

            if index == middle_index:
                x = self._width / 2.0
                y = -0.1
                z = 1.0
            elif index < middle_index:
                x = sum([self._compute_zoom(i)*self._widget_width for i in xrange(0, index)]) - \
                    self._compute_zoom(0)*self._widget_width + w/2.0
                y = 4.0*(th+h/2.0)/(tw*tw)*(x-tw/2.0)*(x-tw/2.0)
                z = 0.0
            elif index > middle_index:
                index = self._visible_items + 2 - index
                x = self._width - \
                    sum([self._compute_zoom(i)*self._widget_width for i in \
                    xrange(0, index)]) + self._compute_zoom(0)*self._widget_width \
                    + w/2.0

                y = 4.0*(th+h/2.0)/(tw*tw)*(x-tw/2.0)*(x-tw/2.0)
                z = 0.0

        position = (x+x_top_left_offset, y+y_top_left_offset, z)
        return position

    def _compute_opacity(self, index):
        # floor division
        middle_index = self._visible_items // 2 + 1
        if index == middle_index:
            return 255
        elif index == 0 or index == self._visible_items+1:
            return 0
        else:
            max_opacity = 160
            min_opacity = 100
            step = (max_opacity - min_opacity) / middle_index

            if index > middle_index:
                index = -index + 2*middle_index

            opacity = min_opacity + index * step
            return opacity

    def _compute_zoom(self, index):
        zoom_middle = 1.7
        min_zoom = 0.7
        # floor division
        middle_index = self._visible_items // 2 + 1

        if index == middle_index:
            return zoom_middle + 0.1
        elif index == 0 or index == self._visible_items+1:
            return min_zoom
        else:
            mid = middle_index - 1
            step = (2.0*mid+1.0-zoom_middle-2.0*mid*min_zoom)/(mid*(mid-1))

            if index > middle_index:
                index = -index + 2*middle_index

            zoom = min_zoom + (index-1) * step
            return zoom


    def _update(self):
        # floor division
        start = self._selected_item - self._visible_items // 2 - 1
        stop = min(start + self._visible_items + 2, len(self.widgets))

        i = max(0, -start)

        for widget in self.animated_widgets[max(start, 0):stop]:
            widget.opacity  = self._attributes[i][1]
            widget.position = self._attributes[i][0]
            zoom = self._attributes[i][2]
            if self._orientation == VERTICAL:
                h = self._widget_height*zoom
                w = h*self._ratio
                widget.size = (w, h)

            elif self._orientation == HORIZONTAL:
                w = self._widget_width*zoom
                h = w/self._ratio
                widget.size = (w, h)
            i += 1


if __name__ == "__main__":
    import os
    import pgm
    import gobject
    import gst
    from pypgmtools.graph.text import Text
    from pypgmtools.graph.image import Image
    from pypgmtools.timing import implicit
    
    def create_text(label):
        txt = Text()
        txt.label = label
        txt.font_family = "Bitstream DejaVu"
        txt.font_height = 0.225
        txt.fg_color = (255, 255, 255, 255)
        txt.bg_color = (255, 0, 0, 255)
        txt.ellipsize = pgm.TEXT_ELLIPSIZE_END
        txt.visible = True
        return txt
        
    def create_img(img_file):
        img = Image()
        img.set_from_fd(os.open(img_file, os.O_RDONLY))
        img.fg_color = (255, 255, 255, 255)
        img.bg_color = (100, 200, 100, 155)
        img.visible = True
        return img

    def create_video(video_uri):
        img = Image()
        img.fg_color = (255, 255, 255, 255)
        img.bg_color = (0, 0, 0, 0)
        img.alignment = pgm.IMAGE_LEFT
        img.visible = True

        # GStreamer pipeline setup
        pipeline = gst.element_factory_make('playbin')
        sink = gst.element_factory_make('pgmimagesink')
        pipeline.set_property('uri', video_uri)
        pipeline.set_property('video-sink', sink)
        sink.set_property('image', img)
        pipeline.set_state(gst.STATE_PLAYING)

        return img


    def on_key_press(viewport, event, gl, widget, step_size):
        if event.type == pgm.KEY_PRESS:
            # quit on q or ESC
            if event.keyval == pgm.keysyms.q or event.keyval == pgm.keysyms.Escape:
                pgm.main_quit()

            elif event.keyval == pgm.keysyms.b:
                list_widget.selected_item = 0

            elif event.keyval == pgm.keysyms.g:
                list_widget.selected_item = len(list_widget) - 1

            elif event.keyval == pgm.keysyms.h:
                list_widget.visible_items += 2
                print list_widget.visible_items

            elif event.keyval == pgm.keysyms.n:
                list_widget.visible_items -= 2

            elif event.keyval == pgm.keysyms.s:
                if list_widget.orientation == VERTICAL:
                    list_widget.orientation = HORIZONTAL
                elif list_widget.orientation == HORIZONTAL:
                    list_widget.orientation = VERTICAL

            elif event.keyval == pgm.keysyms.Down or \
                 event.keyval == pgm.keysyms.Right:
                list_widget.selected_item += 1

            elif event.keyval == pgm.keysyms.Up or \
                 event.keyval == pgm.keysyms.Left:
                list_widget.selected_item -= 1

            # add an example entry at the 3rd position
            elif event.keyval == pgm.keysyms.space:
                list_widget.insert(2, create_text("Test"))
                #list_widget.append(create_text("Test"))

            # remove the currently selected item
            elif event.keyval == pgm.keysyms.Return:
                list_widget.pop(list_widget.selected_item)

    def on_delete(viewport, event):
        pgm.main_quit()
    

    
    # OpenGL viewport creation
    factory = pgm.ViewportFactory('opengl')
    gl = factory.create()
    gl.title = 'TextList widget-test'

    # Canvas and image drawable creation
    canvas = pgm.Canvas()
    
    # Bind the canvas to the OpenGL viewport
    gl.set_canvas(canvas)
    gl.show()
    
    list_widget = CurvedList(canvas, pgm.DRAWABLE_MIDDLE, orientation=HORIZONTAL)
    list_widget.position = (0.5, 1.0, 0.0)
    list_widget.width = 3.0
    list_widget.height = 1.0
    list_widget.visible_items = 13

    for i in xrange(50):
        #list_widget.append(create_text("%s" % i))
        list_widget.append(create_img("examples/pictures/fluendo.png"))
        """
        t = create_text("%s" % i)
        i = create_img("examples/pictures/fluendo.png")
        g = Group(canvas, pgm.DRAWABLE_MIDDLE)
        g.add(t)
        g.add(i)
        g.visible = True
        list_widget.append(g)
        """

    #list_widget.append(create_video("file:///home/kaleo/Desktop/media/videos/t_mariogalaxy_e36_h264.wmv"))

    list_widget.selected_item = 3
    list_widget.visible = True
    
    animated_list_widget = implicit.AnimatedObject(list_widget)


    # Let's start a mainloop
    gl.connect('key-press-event',
               on_key_press,
               gl,
               animated_list_widget,
               0.25)
    gl.connect('delete-event', on_delete)
    gobject.timeout_add(10, gl.update)
    pgm.main()

