# -*- mode: python; coding: utf-8 -*-
#
# Pigment Python tools
#
# Copyright © 2006, 2007 Fluendo Embedded S.L.
#
# This library is free software; you can redistribute it and/or
# modify it under the terms of the GNU Lesser General Public
# License as published by the Free Software Foundation; either
# version 2 of the License, or (at your option) any later version.
#
# This library is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
# Lesser General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public
# License along with this library; if not, write to the
# Free Software Foundation, Inc., 59 Temple Place - Suite 330,
# Boston, MA 02111-1307, USA.

import pgm
from pypgmtools.graph.group import Group
from pypgmtools.graph.image import Image
from pypgmtools.timing import implicit

import os


class ProgressBarOsd(Group):

    def __init__(self, canvas, layer, length, bg_image, fg_image):

        Group.__init__(self, canvas, layer)

        if length > 0:
            self._length = length
        else:
            ## FIXME: Raise an Error here?
            self._length = 100
        self._cursor_position = 0

        self._bg = Image()
        self._bg.set_from_fd(os.open(bg_image, os.O_RDONLY))
        self._bg.layout = pgm.IMAGE_FILLED
        self._bg.alignment = pgm.IMAGE_LEFT
        self._bg.height = 1.0
        self._bg.width = 4.0
        self._bg.bg_color = (0, 0, 0, 0)
        self._bg.visible = True

        self._bar = Image()
        self._bar.set_from_fd(os.open(fg_image, os.O_RDONLY))
        self._bar.layout = pgm.IMAGE_FILLED
        self._bar.alignment = pgm.IMAGE_LEFT
        self._bar.height = 1.0
        self._bar.width = 4.0
        self._bar.size = (0, 0)
        self._bar.bg_color = (0, 0, 0, 0)
        self._bar.visible = True
        self._animated_bar = implicit.AnimatedObject(self._bar)
        self._animated_bar.mode = implicit.REPLACE
        self._animated_bar.setup_next_animations(duration = 200)

        self.add(self._bg)
        self.add(self._bar)

        self._offset = 0.16
        self.visible = True


    def size__set(self, size):
        """
        Set the size of the bar. The bar is automatically new painted
        """
        self._bg.size = size
        # Needed for nicer animations
        self._bar.height = size[1]
        self._paint_bar(animated = False)

    def length__set(self, length):
        """
        Set the length. The bar is automatically new painted
        @param length: the length of the process
        @type length: as above: something to do mathematic stuff
        """
        self._length = length
        if self._length <= 0:
            self._length = 1
        if self._cursor_position > self._length:
             self._cursor_position = self._length
        self._paint_bar()

    def length__get(self):
        """
        to get the length:
        @return: returns the length
        """
        return self._length

    def _paint_bar(self, animated = True):
        """
        Here the position is calculated and the bar is painted
        """
        next_pos = self._bg.size[0] / self.length * \
                                    self._cursor_position
        if animated:
            self._animated_bar.width = next_pos
        else:
            self._animated_bar.stop_animations()
            self._bar.width = next_pos

    def cursor_position__set(self, position):
        """
        Set the position of the cursor. The bar is automatically new painted
        (with animation)
        @param position: the position to set to
        @type position: something to do mathematic stuff with
        """
        self._cursor_position = position
        self._cursor_position = max(0,
                                    min(self._cursor_position, self._length)
                                    )
#        if self._cursor_position <= self._length:
        self._paint_bar()

    def cursor_position__get(self):
        """
        returns the position of the cursor
        @return : the position of the cursor
        """
        return self._cursor_position

    def jump_to_position(self, position):
        """
        Jump to this position. This means that the cursor doesn't go there
        with animationm but that it is there immediatly
        @param position: the position to jump to
        @type position: something to do mathematic stuff with
        """
        self._cursor_position = position
        self._cursor_position = max(0,
                                    min(self._cursor_position, self.length)
                                    )
        self._paint_bar(animated=False)

    def bg_color__set(self, color):
        """
        Set the background color. The bar is automatically new painted
        @param color: color of the background
        """
        self._bg.bg_color = color

    def bg_color__get(self):
        """
        returns the current background color
        """
        return self._bg.bg_color

    def bar_color__set(self, color):
        """
        Set the color of the bar. The bar is automatically new painted
        @param color: color of the bar
        """
        self._bar.bg_color = color

    def bar_color__get(self):
        """
        return the current color of the bar
        """
        return self._bar.bg_color
