# -*- mode: python; coding: utf-8 -*-
#
# Pigment media rendering library
#
# Copyright © 2006, 2007 Fluendo Embedded S.L. <www.fluendo.com>
# All rights reserved.
#
# This file is available under one of two license agreements.
#
# This file is licensed under the GPL version 2. See "LICENSE.GPL" in
# the root of this distribution including a special exception to use
# Pigment with Fluendo's plugins.
#
# The GPL part of Pigment is also available under a commercial
# licensing agreement from Fluendo. See "LICENSE.Pigment" in the root
# directory of this distribution package for details on that license.
#
# Author: Mirco Müller <macslow@bangang.de>

from pypgmtools.graph.group import Group
from pypgmtools.graph.image import Image
from pypgmtools.graph.text import Text
from pypgmtools.timing import implicit
from pypgmtools.utils import classinit
import math
import gobject
import gst
import os
import pgm

# widget-modes
CAROUSEL          = 0
CHAIN             = 1

# indices for icon-part
NORMAL            = 0
BLURRED           = 1
REFLECTED         = 2
TEXT              = 3

# indices for attribute-value
X                 = 0
Y                 = 1
WIDTH             = 2
HEIGHT            = 3
NORMAL_OPACITY    = 4
BLURRED_OPACITY   = 5
REFLECTED_OPACITY = 6
TEXT_OPACITY      = 7

class TopLevelMenu(Group):
    """
    """
    
    __metaclass__ = classinit.ClassInitMeta
    __classinit__ = classinit.build_properties

    def __init__(self,
                 canvas            = None,
                 layer             = pgm.DRAWABLE_MIDDLE,
                 width             = 1.0,
                 height            = 1.0,
                 font_family       = "DejaVu Sans",
                 selected_position = (0.0, 0.0, 0.0),
                 selected_size     = (0.0, 0.0),
                 selected_opacity  = 255,
                 mode              = CAROUSEL,
                 path_steps        = 1,
                 duration          = 500,
                 transformation    = implicit.DECELERATE):
        """
        """

        Group.__init__(self, canvas, layer)

        self._canvas            = canvas
        self._layer             = layer
        self._width             = width
        self._height            = height
        self._font_height       = 0.3
        self._font_family       = font_family
        self._selected_position = selected_position
        self._selected_size     = selected_size
        self._selected_opacity  = selected_opacity
        self._mode              = mode
        self._path_steps        = path_steps
        self._duration          = duration
        self._transformation    = transformation
        self._number_of_items   = 0
        self._selected          = False
        self._focused_item      = 0
        self._item_handles      = []
        self._items             = []
        self._values            = []


    def width__get(self):
        return self._width
    
    def height__get(self):
        return self._height

    def size__get(self):
        return (self._width, self._height)

    def width__set(self, width):
        self._width = width
    
    def height__set(self, height):
        self._height = height

    def size__set(self, size):
        self._width = size[0]
        self._height = size[1]

    def _within_bounds(self, index):
        """
        Internal function to make a bounds-check against the current limits of
        the item-list.
        """
        if index > -1 and index < self._number_of_items:
            return True
        else:
            return False

    def _apply_selected_values(self, index):
        """
        Update self._items[index] with the values for the selected state
        """
        item = self._items[index]

        # set values for normal-image
        item[NORMAL].opacity    = self._selected_opacity
        item[NORMAL].position   = self._selected_position
        item[NORMAL].size       = self._selected_size

        # set values for blurred-image
        item[BLURRED].opacity   = 0
        item[BLURRED].position  = self._selected_position
        item[BLURRED].size      = self._selected_size

        reflected_y = self._selected_position[1] + self._selected_size[1]

        # set values for reflected-image
        item[REFLECTED].opacity  = 0
        item[REFLECTED].position = (self._selected_position[0], \
                                    reflected_y, \
                                    self._selected_position[2])
        item[REFLECTED].size     = self._selected_size

        # set values for text-image
        item[TEXT].opacity      = 0
        item[TEXT].position     = (self._selected_position[0], \
                                   reflected_y, \
                                   self._selected_position[2])
        item[TEXT].size         = self._selected_size


    def _apply_hidden_values(self, item_index, value_index):
        """
        """
        item = self._items[item_index]

        start       = value_index
        index       = 0
        opacity     = []
        position    = []
        reflected_position = []
        size        = []

        for index in xrange(self._path_steps):

            opacity.append(1)
            x = self._values[start + index][X] + \
                0.45 * self._values[start + index][WIDTH]
            y = self._values[start + index][Y] + \
                0.45 * self._values[start + index][HEIGHT]
            reflected_y = self._values[start + index][Y] + \
                          0.55 * self._values[start + index][HEIGHT]
            position.append((x, y, item[NORMAL].z))
            reflected_position.append((x, reflected_y, item[NORMAL].z))
            size.append((0.1 * self._values[start + index][WIDTH], \
                         0.1 * self._values[start + index][HEIGHT]))

        # set values for normal-image
        item[NORMAL].opacity    = opacity

        # set values for blurred-image
        item[BLURRED].opacity   = opacity
     
        # set values for reflected-image
        item[REFLECTED].opacity = opacity
      
        # set values for text-image
        item[TEXT].opacity      = opacity

    def _apply_visible_values(self, item_index, value_index):
        """
        """
        item = self._items[item_index]

        start = value_index

        if self._path_steps > 1:
            normal_opacity = []
            blurred_opacity = []
            reflected_opacity = []
            text_opacity = []
            position = []
            reflected_position = []
            text_position = []
            size = []

            # fill the value-lists
            for index in xrange(self._path_steps):
                normal_opacity.append(self._values[start + index][NORMAL_OPACITY])
                blurred_opacity.append(self._values[start + index][BLURRED_OPACITY])
                reflected_opacity.append(self._values[start + index][REFLECTED_OPACITY])
                text_opacity.append(self._values[start + index][TEXT_OPACITY])
                position.append((self._values[start + index][X],
                                 self._values[start + index][Y],
                                 item[NORMAL].z))


                reflected_position.append((self._values[start + index][X], \
                                           self._values[start + index][Y] + \
                                           self._values[start + index][HEIGHT], \
                                           item[REFLECTED].z))

                text_position.append((self._values[start + index][X], \
                                      self._values[start + index][Y] + \
                                      self._values[start + index][HEIGHT], \
                                      item[TEXT].z))

                size.append((self._values[start + index][WIDTH], \
                             self._values[start + index][HEIGHT]))
        else:
            normal_opacity = self._values[start][NORMAL_OPACITY]
            blurred_opacity = self._values[start][BLURRED_OPACITY]
            reflected_opacity = self._values[start][REFLECTED_OPACITY]
            text_opacity = self._values[start][TEXT_OPACITY]
            position = (self._values[start][X], \
                        self._values[start][Y], \
                        item[NORMAL].z)
            reflected_position = (self._values[start][X], \
                                  self._values[start][Y] + \
                                  self._values[start][HEIGHT], \
                                  item[REFLECTED].z)
            text_position = (self._values[start][X], \
                             self._values[start][Y] + \
                             self._values[start][HEIGHT],
                             item[TEXT].z)
            size = (self._values[start][WIDTH], \
                    self._values[start][HEIGHT])


        # set values for normal-image
        item[NORMAL].opacity    = normal_opacity
        item[NORMAL].position   = position
        item[NORMAL].size       = size

        # set values for blurred-image
        item[BLURRED].opacity   = blurred_opacity
        item[BLURRED].position  = position 
        item[BLURRED].size      = size

        # set values for reflected-image
        item[REFLECTED].opacity   = reflected_opacity
        item[REFLECTED].position  = reflected_position
        item[REFLECTED].size      = size

        # set values for text-image
        item[TEXT].opacity    = text_opacity
        item[TEXT].position   = text_position
        item[TEXT].size       = size

    def _calculate_value_list(self, value):
        value_list = []

        x                 = 0
        y                 = 0
        width             = 0
        height            = 0
        normal_opacity    = 0
        blurred_opacity   = 0
        reflected_opacity = 0
        text_opacity      = 0

        if self._mode == CAROUSEL:
            coef              = 1.6
            tmp_x             = (1.0 + math.sin(value))/2.0
            tmp_y             = (1.0 + math.cos(value))/2.0
            width             = self._width/3.0 * (coef-tmp_y)/coef
            height            = self._height/3.0 * (coef-tmp_y)/coef
            x                 = (self._width - width) * tmp_x
            y                 = (self._height - height) * tmp_y
            y                 = abs(y + 2.0*height - self._height)
            normal_opacity    = 255 * (1 - tmp_y)
            blurred_opacity   = 128 - 128 * (1 - tmp_y)
            reflected_opacity = 64 * (1 - tmp_y)
            text_opacity      = 255 * math.exp(40.0 * (-tmp_y))
        elif self._mode == CHAIN:
            # FIXME: values for chain need to computed here
            x                 = self._width / 2.0 * math.cos(value)
            y                 = self._height * math.sin(value)
            width             = y
            height            = y
            normal_opacity    = 255
            blurred_opacity   = 128
            reflected_opacity = 1
            text_opacity      = 255

        value_list.append(x)
        value_list.append(y)
        value_list.append(width)
        value_list.append(height)
        value_list.append(normal_opacity)
        value_list.append(blurred_opacity)
        value_list.append(reflected_opacity)
        value_list.append(text_opacity)

        return value_list

    def layout(self):
        """
        Compute the positions, sizes and opacities for all the items in the
        widget.
        """
        if self._number_of_items == 0:
            return

        self._values = []

        if self._mode == CAROUSEL:
            max_index = self._number_of_items * self._path_steps
        elif self._mode == CHAIN:
            max_index = self._number_of_items

        start = -math.pi
        end = math.pi
        value = start
        value_step = (end - start) / max_index
        for index in xrange(max_index):
            self._values.append(self._calculate_value_list(value))
            value += value_step

        self.regenerate_text()


    def regenerate_text(self):
        # compute the maximum size of the text drawables
        max_values = self._calculate_value_list(math.pi)

        for i in self._items:
            i[TEXT].stop_animations()
        # apply it to all the text drawables
        for i in self._item_handles:
            # store the original size
            prev_size = i[TEXT].size

            i[TEXT].visible = False
            # apply the maximum size
            i[TEXT].size = (max_values[WIDTH], max_values[HEIGHT])

            self._font_height = max_values[HEIGHT] / 3.0
            # FIXME: hack to regenerate the text drawable
            i[TEXT].font_height = self._font_height + 0.1
            i[TEXT].font_height = self._font_height

            # restore original size
            i[TEXT].size = prev_size
            i[TEXT].visible = True

    def update(self):
        """
        Sets the correct positions, sizes and opacities for all
        items in the widget depending on the currently select state or currently
        focused item.
        """
        if self._number_of_items == 0:
            return

        if self._selected:
            self._apply_selected_values(self._focused_item)
            start = self._focused_item + 1
            index = 0
            value_index = 1
            while index < (self._number_of_items - 1):
                real_index = (start + index) % self._number_of_items
                self._apply_hidden_values(real_index, value_index)
                index += 1
                value_index -= self._path_steps
        else:
            start = self._focused_item
            index = 0
            value_index = 0
            while index < self._number_of_items:
                real_index = (start + index) % self._number_of_items
                self._apply_visible_values(real_index, value_index)
                index += 1
                value_index -= self._path_steps


    def update_images(self, index, normal_file, blurred_file, reflected_file):
        if index >= 0 and index < len(self._item_handles):
            normal_image = self._item_handles[index][0]
            blurred_image = self._item_handles[index][1]
            reflected_image = self._item_handles[index][2]
            normal_image.set_from_fd(os.open(normal_file, os.O_RDONLY))
            blurred_image.set_from_fd(os.open(blurred_file, os.O_RDONLY))
            reflected_image.set_from_fd(os.open(reflected_file, os.O_RDONLY))
        
        
    def insert(self, index, normal_file, blurred_file, reflected_file, label):
        """
        This call has no effect if the widget is in state "selected"!
        """
        if self._selected:
            return

        # create images/drawables
        normal_image    = Image()
        blurred_image   = Image()
        reflected_image = Image()
        text_image      = Text()

        # read images from disk
        normal_image.set_from_fd(os.open(normal_file, os.O_RDONLY))
        blurred_image.set_from_fd(os.open(blurred_file, os.O_RDONLY))
        reflected_image.set_from_fd(os.open(reflected_file, os.O_RDONLY))

        # set up the text-label
        #text_image.label         = label
        text_image.label         = ""
        text_image.font_height   = self._font_height
        text_image.font_family   = self._font_family
        text_image.alignment     = pgm.TEXT_ALIGN_CENTER
        text_image.ellipsize     = pgm.TEXT_ELLIPSIZE_MIDDLE

        # set initial opacity
        normal_image.opacity    = 1
        blurred_image.opacity   = 1
        reflected_image.opacity = 1
        text_image.opacity      = 1

        # set initial foreground-color to full opaque white
        normal_image.fg_color    = (255, 255, 255, 255)
        blurred_image.fg_color   = (255, 255, 255, 255)
        reflected_image.fg_color = (255, 255, 255, 255)
        text_image.fg_color      = (255, 255, 255, 255)

        # set initial background-color to transparent black
        normal_image.bg_color    = (0, 0, 0, 0)
        blurred_image.bg_color   = (0, 0, 0, 0)
        reflected_image.bg_color = (0, 0, 0, 0)
        text_image.bg_color      = (0, 0, 0, 0)

        # set inital state of visible-flag
        normal_image.visible    = True
        blurred_image.visible   = True
        reflected_image.visible = True
        text_image.visible      = True

        # make sure the image is scaled
        normal_image.layout       = pgm.IMAGE_SCALED
        blurred_image.layout      = pgm.IMAGE_SCALED
        reflected_image.layout    = pgm.IMAGE_SCALED

        # add each image of the icon to the widgets group so it actually gets
        # added to the canvas
        self.add(normal_image)
        self.add(blurred_image)
        self.add(reflected_image)
        self.add(text_image)

        # create a list of the four images and their "handles" in order to
        # be able to delete it later (in remove())
        item = []
        item.append(normal_image)
        item.append(blurred_image)
        item.append(reflected_image)
        item.append(text_image)
        self._item_handles.insert(index, item)

        # create animated version of the images
        animated_normal_image    = implicit.AnimatedObject(normal_image)
        animated_blurred_image   = implicit.AnimatedObject(blurred_image)
        animated_reflected_image = implicit.AnimatedObject(reflected_image)
        animated_text_image      = implicit.AnimatedObject(text_image)

        # set animation attributes fo the images
        animated_normal_image.setup_next_animations(duration = self._duration,
                                                    transformation = self._transformation)
        animated_normal_image.mode = implicit.REPLACE
        animated_blurred_image.setup_next_animations(duration = self._duration,
                                                     transformation = self._transformation)
        animated_blurred_image.mode = implicit.REPLACE
        animated_reflected_image.setup_next_animations(duration = self._duration,
                                                       transformation = self._transformation)
        animated_reflected_image.mode = implicit.REPLACE
        animated_text_image.setup_next_animations(duration = self._duration,
                                                  transformation = self._transformation)
        animated_text_image.mode = implicit.REPLACE

        # create and add an item of the animated images
        item = []
        item.append(animated_normal_image)
        item.append(animated_blurred_image)
        item.append(animated_reflected_image)
        item.append(animated_text_image)
        self._items.insert(index, item)

        self._number_of_items = len(self._items)

    def append(self, normal_file, blurred_file, reflected_file, label):
        """
        This call has no effect if the widget is in state "selected"!
        """
        if self._selected:
            return

        self.insert(self._number_of_items,
                    normal_file,
                    blurred_file,
                    reflected_file,
                    label)

    def delete(self, index):
        """
        This call has no effect if the widget is in state "selected"!
        """
        if self._number_of_items == 0 or self._selected:
            return

        # retrieve handles from the group
        item = self._item_handles.pop(index)
        self.remove(item[NORMAL])
        self.remove(item[BLURRED])
        self.remove(item[REFLECTED])
        self.remove(item[TEXT])
        item = self._items.pop(index)

        self._number_of_items = len(self._items)

    def prev(self):
        """
        Make the previous item in the widget the currently focused item. This
        function will automatically set the selected state to false and also
        call update() implicitly for you.
        """
        if self._number_of_items == 0:
            return

        if self._selected:
            self._selected = False
        else:
            focused_item = self._focused_item
            self._focused_item = (focused_item - 1) % self._number_of_items
        self.update()

    def next(self):
        """
        Make the next item in the widget the currently focused item. This
        function will automatically set the selected state to false and also
        call update() implicitly for you.
        """
        if self._number_of_items == 0:
            return

        if self._selected:
            self._selected = False
        else:
            focused_item = self._focused_item
            self._focused_item = (focused_item + 1) % self._number_of_items
        self.update()

    def select(self):
        """
        This toggles the selected state of the widget. update() is called
        implicitly for you.
        """
        if self._number_of_items == 0:
            return

        if self._selected:
            self._selected = False
        elif not self._selected:
            self._selected = True
        self.update()

    def is_selected(self):
        """
        Returns if widget is in selected state or not.
        """
        return self._selected

    def focused_item__get(self):
        """
        Get the index of the currently focused item. This does not imply any
        selected state! If you want to know the selected state call
        is_selected()
        """
        return self._focused_item

    def focused_item__set(self, index):
        """
        Directly set the currently focused item. This function has no effect if
        widget is in selected state. If widget is not in selected state update()
        is called implicitly for you.
        """
        if self._selected:
            return

        if self._focused_item != index and self._within_bounds(index):
            self._focused_item = index
            self.update()

if __name__ == "__main__":

    import os, os.path
    import sys
    import random
    import pgm
    import gobject
    import gst
    from pypgmtools.graph.group import Group
    from pypgmtools.graph.text import Text
    from pypgmtools.graph.image import Image
    from pypgmtools.utils import classinit
    from pypgmtools.timing.implicit import AnimatedObject

    def on_delete(viewport,
                  event,
                  shade):
        shade.position = (0.0, 0.0, 0.0)
        gobject.timeout_add(1600, pgm.main_quit)

    def on_key_press(viewport,
                     event,
                     widget,
                     shade):
        if event.type == pgm.KEY_PRESS:
            if event.keyval == pgm.keysyms.q or \
               event.keyval == pgm.keysyms.Escape:
                shade.position = (0.0, 0.0, 0.0)
                gobject.timeout_add(1600, pgm.main_quit)
            elif event.keyval == pgm.keysyms.space or \
                 event.keyval == pgm.keysyms.Return:
                widget.select()
            elif event.keyval == pgm.keysyms.n or \
                 event.keyval == pgm.keysyms.Right:
                widget.next()
            elif event.keyval == pgm.keysyms.p or \
                 event.keyval == pgm.keysyms.Left:
                widget.prev()
            elif event.keyval == pgm.keysyms.a:
                widget.append("./examples/pictures/music-normal_256x256.png",
                              "./examples/pictures/music-blurred_256x256.png",
                              "./examples/pictures/music-reflected_256x256.png",
                              "New")
                widget.layout()
                widget.update()
            elif event.keyval == pgm.keysyms.r:
                widget.delete(1)
                widget.layout()
                widget.update()
            elif event.keyval == pgm.keysyms.f:
                viewport.fullscreen = False
                viewport.update ()
            elif event.keyval == pgm.keysyms.v:
                viewport.fullscreen = True
                viewport.update ()

    # OpenGL viewport creation
    factory = pgm.ViewportFactory('opengl')
    gl = factory.create()
    gl.title = 'TopLevelMenu widget-test'
    size = gl.get_screen_resolution()
    aspect_ratio = float(size[0]) / float(size[1])
    gl.size = (aspect_ratio * 400, 400)
    window_height = 3.0
    window_width = aspect_ratio * window_height
    gl.fullscreen = True
    gl.show()

    # Canvas and image drawable creation
    canvas = pgm.Canvas()
    canvas.size = (window_width, window_height)

    # Bind the canvas to the OpenGL viewport
    gl.set_canvas(canvas)
    
    bg_image          = Image()
    bg_image.set_from_fd(os.open("./examples/pictures/bg.png",
                                 os.O_RDONLY))
    bg_image.opacity  = 255
    bg_image.visible  = True
    bg_image.fg_color = (255, 255, 255, 255)
    bg_image.bg_color = (0, 0, 0, 0)
    bg_image.position = (0.0, 0.0, 0.0)
    bg_image.width    = window_width
    bg_image.height   = window_height
    bg_image.layout   = pgm.IMAGE_FILLED

    bg_group = Group(canvas,
                     pgm.DRAWABLE_FAR)
    bg_group.add(bg_image)
    bg_group.opacity = 255
    bg_group.visible = True

    shade_image          = Image()
    shade_image.set_from_fd(os.open("./examples/pictures/shade.png",
                                    os.O_RDONLY))
    shade_image.opacity  = 255
    shade_image.visible  = True
    shade_image.fg_color = (255, 255, 255, 255)
    shade_image.bg_color = (0, 0, 0, 0)
    shade_image.position = (0.0, .0, 0.0)
    shade_image.width    = window_width
    shade_image.height   = window_height
    shade_image.layout   = pgm.IMAGE_FILLED
    animated_shade_image = AnimatedObject(shade_image)
    animated_shade_image.setup_next_animations(duration = 1500)
    animated_shade_image.position = (0.0, -6.0, 0.0)
    
    shade_group = Group(canvas, pgm.DRAWABLE_NEAR)
    shade_group.add(shade_image)
    shade_group.opacity = 255
    shade_group.visible = True

    widget = TopLevelMenu(canvas,
                          pgm.DRAWABLE_MIDDLE,
                          width                 = window_width,
                          height                = window_height,
                          selected_position     = (0.1, -0.75, 0.0),
                          selected_size         = (4.5, 4.5),
                          selected_opacity      = 32,
                          mode                  = CAROUSEL,
                          path_steps            = 1,
                          duration              = 500,
                          transformation        = implicit.DECELERATE)

    widget.position = ((window_width-widget.width)/2.0,
                       (window_height-widget.height)/2.0,
                       0.0)
    widget.opacity = 255
    widget.visible = True

    """
    frame = Image()
    frame.opacity  = 255
    frame.bg_color = (0, 0, 255, 255)
    frame.position = widget.position
    frame.size = widget.size
    frame.visible  = True
    canvas.add(pgm.DRAWABLE_MIDDLE, frame)
    """

    widget.append("./examples/pictures/music-normal_256x256.png",
                  "./examples/pictures/music-blurred_256x256.png",
                  "./examples/pictures/music-reflected_256x256.png",
                  "音樂")

    widget.append("./examples/pictures/video-normal_256x256.png",
                  "./examples/pictures/video-blurred_256x256.png",
                  "./examples/pictures/video-reflected_256x256.png",
                  "視頻")

    widget.append("./examples/pictures/dvd-normal_256x256.png",
                  "./examples/pictures/dvd-blurred_256x256.png",
                  "./examples/pictures/dvd-reflected_256x256.png",
                  "DVD")

    widget.append("./examples/pictures/photo-normal_256x256.png",
                  "./examples/pictures/photo-blurred_256x256.png",
                  "./examples/pictures/photo-reflected_256x256.png",
                  "圖片")

    widget.append("./examples/pictures/service-normal_256x256.png",
                  "./examples/pictures/service-blurred_256x256.png",
                  "./examples/pictures/service-reflected_256x256.png",
                  "服務")

    widget.layout()
    widget.update()

    gl.connect('key-press-event',
               on_key_press,
               widget,
               animated_shade_image)
    gl.connect('delete-event',
               on_delete,
               animated_shade_image)
    gobject.timeout_add(15,
                        gl.update)

    pgm.main()
