/* -*- Mode: C; c-basic-offset: 2; indent-tabs-mode: nil -*-
 *
 * Pigment media rendering library
 *
 * Copyright © 2006, 2007 Fluendo Embedded S.L.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * Author(s): Loïc Molinari <loic@fluendo.com>
 *            Julien Moutte <julien@fluendo.com>
 */

/**
 * SECTION:pgmplugin
 * @short_description: Various structs and macros used for plugins handling.
 *
 * Various structs and macros used by Pigment for plugins handling.
 *
 * Last reviewed on 2007-04-12 (0.1.5)
 */

#ifndef __PGM_PLUGIN_H__
#define __PGM_PLUGIN_H__

#include "pgmviewport.h"

G_BEGIN_DECLS

/**
 * PGM_PLUGIN_PATH_NAME:
 *
 * The name of the plugin path environment variable name.
 */
#define PGM_PLUGIN_PATH_NAME "PGM_PLUGIN_PATH"

typedef struct _PgmPluginDesc PgmPluginDesc;

/**
 * PgmPluginInitFunc:
 *
 * A plugin should provide a pointer to a function of this type in the
 * #PgmPluginDesc struct. This function will be called to initialize the
 * plugin.
 *
 * Returns: %TRUE if the initialization successes, %FALSE otherwise.
 */
typedef gboolean (*PgmPluginInitFunc) (void);

/**
 * PgmPluginShutdownFunc:
 *
 * A plugin should provide a pointer to a function of this type in the
 * #PgmPluginDesc  struct. This function will be called to shutdown the plugin.
 *
 * Returns: %TRUE if the deinitialization successes, %FALSE otherwise.
 */
typedef gboolean (*PgmPluginShutdownFunc) (void);

/**
 * PgmPluginCreateFunc:
 *
 * A plugin should provide a pointer to a function of this prototype in the
 * %PgmPluginDesc struct. This function will be called by the user to create
 * the %PgmViewport.
 *
 * Returns: a new #PgmViewport.
 */
typedef PgmViewport* (*PgmPluginCreateFunc) (void);

/**
 * PgmPluginDesc:
 * @init: the plugin initialization function pointer.
 * @shutdown: the plugin shutdown function pointer.
 * @create: the plugin create function pointer (returning a %PgmViewport).
 * @name: the plugin name.
 * @description: the plugin description.
 * @license: the plugin license.
 * @origin: the plugin origin URL.
 * @author: the plugin author.
 *
 * Pigment Plugin description structure.
 */
struct _PgmPluginDesc {
  PgmPluginInitFunc     init;
  PgmPluginShutdownFunc shutdown;
  PgmPluginCreateFunc   create;
  gchar                 *name;
  gchar                 *description;
  gchar                 *license;
  gchar                 *origin;
  gchar                 *author;
};

/**
 * PGM_PLUGIN_DEFINE:
 * @init: the plugin initialization function pointer.
 * @shutdown: the plugin shutdown function pointer.
 * @create: the plugin creation function pointer (returning a %PgmViewport).
 * @name: the plugin name.
 * @description: the plugin description.
 * @license: the plugin license.
 * @origin: the plugin origin URL.
 * @author: the plugin author.
 *
 * Utility macro to create a #PgmPluginDesc plugin description structure.
 * This is the entry point for every Pigment plugin and it is highly
 * recommended to use this macro to avoid common mistakes making entry point
 * unusable.
 */
#define PGM_PLUGIN_DEFINE(init,shutdown,create,name,description,license,origin,author) \
  PgmPluginDesc pgm_plugin_desc =                                                    \
    { init, shutdown, create, name, description, license, origin, author };

G_END_DECLS

#endif /* __PGM_PLUGIN_H__ */
