/* -*- Mode: C; c-basic-offset: 2; indent-tabs-mode: nil -*-
 *
 * Pigment OpenGL plugin
 *
 * Copyright © 2006, 2007 Fluendo Embedded S.L.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * Author: Loïc Molinari <loic@fluendo.com>
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif /* HAVE_CONFIG_H */

#include <gdk-pixbuf/gdk-pixbuf.h>
#include "pgmglimage.h"

GST_DEBUG_CATEGORY_EXTERN (pgm_gl_debug);
#define GST_CAT_DEFAULT pgm_gl_debug

static PgmGlDrawableClass *parent_class = NULL;

/* Private methods */

/* Computes the image pixel aspect ratio */
static void
update_image_ratio (PgmGlImage *glimage)
{
  PgmGlDrawable *gldrawable = PGM_GL_DRAWABLE (glimage);
  PgmImage *image = PGM_IMAGE (gldrawable->drawable);

  GST_OBJECT_LOCK (image);
  GST_OBJECT_LOCK (glimage);

  glimage->image_ratio = (gfloat) image->par_n / image->par_d;

  GST_OBJECT_UNLOCK (image);
  GST_OBJECT_UNLOCK (glimage);
}

/* Computes the drawable pixel aspect ratio */
static void
update_drawable_ratio (PgmGlImage *glimage)
{
  PgmGlDrawable *gldrawable = PGM_GL_DRAWABLE (glimage);

  GST_OBJECT_LOCK (glimage);

  if (G_LIKELY (gldrawable->height != 0.0))
    glimage->drawable_ratio = gldrawable->width / gldrawable->height;
  else
    glimage->drawable_ratio = 1.0f;

  GST_OBJECT_UNLOCK (glimage);
}

/* Updates the last position fields with the one from the drawable */
static void
update_last_position (PgmGlImage *glimage)
{
  PgmGlDrawable *gldrawable = PGM_GL_DRAWABLE (glimage);

  GST_OBJECT_LOCK (glimage);

  glimage->last_x = gldrawable->x;
  glimage->last_y = gldrawable->y;
  glimage->last_z = gldrawable->z;

  GST_OBJECT_UNLOCK (glimage);
}

/* Updates the alignment */
static void
update_alignment (PgmGlImage *glimage)
{
  PgmDrawable *drawable = PGM_GL_DRAWABLE (glimage)->drawable;
  PgmImage *image = PGM_IMAGE (drawable);
  PgmImageAlignment align;

  GST_OBJECT_LOCK (image);
  align = image->align;
  GST_OBJECT_UNLOCK (image);

  GST_OBJECT_LOCK (glimage);

  /* Horizontal alignment */
  if (align & PGM_IMAGE_LEFT)
    glimage->h_align = 0.0f;
  else if (align & PGM_IMAGE_RIGHT)
    glimage->h_align = 1.0f;
  else
    glimage->h_align = 0.5f;

  /* Vertical alignment */
  if (align & PGM_IMAGE_TOP)
    glimage->v_align = 0.0f;
  else if (align & PGM_IMAGE_BOTTOM)
    glimage->v_align = 1.0f;
  else
    glimage->v_align = 0.5f;

  GST_OBJECT_UNLOCK (glimage);
}

/* Updates the interpolation */
static void
update_interp (PgmGlImage *glimage)
{
  PgmGlDrawable *gldrawable = PGM_GL_DRAWABLE (glimage);
  PgmImage *image = PGM_IMAGE (gldrawable->drawable);
  PgmTexture *texture = glimage->texture;

  GST_OBJECT_LOCK (image);
  PGM_TEXTURE_LOCK (texture);

  if (image->interp == PGM_IMAGE_BILINEAR)
    texture->filter = PGM_GL_LINEAR;
  else if (image->interp == PGM_IMAGE_NEAREST)
    texture->filter = PGM_GL_NEAREST;

  PGM_TEXTURE_UNLOCK (texture);
  GST_OBJECT_UNLOCK (image);
}

/* Simply copies gldrawable background vertices */
static void
set_standard_vertices (PgmGlImage *glimage)
{
  PgmGlDrawable *gldrawable = PGM_GL_DRAWABLE (glimage);
  PgmGlFloat *drb_vertex = gldrawable->bg_vertex;
  PgmGlFloat *img_vertex = glimage->vertex;
  guint i;

  GST_OBJECT_LOCK (glimage);

  for (i = 0; i < 12; i++)
    img_vertex[i] = drb_vertex[i];

  GST_OBJECT_UNLOCK (glimage);
}

/* Modifies vertices for the scaled layout */
static void
set_scaled_vertices (PgmGlImage *glimage)
{
  PgmGlDrawable *gldrawable = PGM_GL_DRAWABLE (glimage);
  PgmGlFloat *drb_vertex = gldrawable->bg_vertex;
  PgmGlFloat *img_vertex = glimage->vertex;
  gfloat offset;

  GST_OBJECT_LOCK (glimage);

  /* Drawable width larger than the image width */
  if (glimage->drawable_ratio >= glimage->image_ratio)
    {
      offset = gldrawable->width - gldrawable->height * glimage->image_ratio;
      img_vertex[0] = drb_vertex[0] + offset * glimage->h_align;
      img_vertex[1] = drb_vertex[1];
      img_vertex[3] = drb_vertex[3] - offset * (1.0f - glimage->h_align);
      img_vertex[4] = drb_vertex[4];
      img_vertex[6] = img_vertex[3];
      img_vertex[7] = drb_vertex[7];
      img_vertex[9] = img_vertex[0];
      img_vertex[10] = drb_vertex[10];
    }
  /* Drawable height larger than the image height */
  else
    {
      offset = gldrawable->height - gldrawable->width / glimage->image_ratio;
      img_vertex[0] = drb_vertex[0];
      img_vertex[1] = drb_vertex[1] + offset * glimage->v_align;
      img_vertex[3] = drb_vertex[3];
      img_vertex[4] = img_vertex[1];
      img_vertex[6] = drb_vertex[6];
      img_vertex[7] = drb_vertex[7] - offset * (1.0f - glimage->v_align);
      img_vertex[9] = drb_vertex[9];
      img_vertex[10] = img_vertex[7];
    }

  /* Then simply copy the z position from the background one */
  img_vertex[2] = drb_vertex[2];
  img_vertex[5] = drb_vertex[5];
  img_vertex[8] = drb_vertex[8];
  img_vertex[11] = drb_vertex[11];

  GST_OBJECT_UNLOCK (glimage);
}

/* Modifies texture coordinates for the zoomed layout */
static void
set_zoomed_coordinates (PgmGlImage *glimage)
{
  PgmGlDrawable *gldrawable = PGM_GL_DRAWABLE (glimage);
  PgmImage *image = PGM_IMAGE (gldrawable->drawable);
  PgmTexture *texture = glimage->texture;
  gfloat pixel_s, pixel_t;
  gfloat max_s, max_t;
  gfloat offset;

  if (image->storage_type != PGM_IMAGE_IMAGE)
    texture = glimage->texture;
  else
    texture = glimage->master;

  PGM_TEXTURE_LOCK (texture);

  /* Remove the thin one pixel lines on bottom and right of the texture */
  pixel_s = 1.0f / texture->width_pot;
  pixel_t = 1.0f / texture->height_pot;

  max_s = (gfloat) texture->width / texture->width_pot - pixel_s;
  max_t = (gfloat) texture->height / texture->height_pot - pixel_t;

  PGM_TEXTURE_UNLOCK (texture);

  GST_OBJECT_LOCK (glimage);

  /* Adapt coordinates when drawable width is larger than image width */
  if (glimage->drawable_ratio >= glimage->image_ratio)
    {
      gfloat image_height = gldrawable->width / glimage->image_ratio;

      offset = ((image_height - gldrawable->height) * max_t) / image_height;
      glimage->coord[0] = 0.0f;
      glimage->coord[1] = offset * glimage->v_align;
      glimage->coord[2] = max_s;
      glimage->coord[3] = glimage->coord[1];
      glimage->coord[4] = max_s;
      glimage->coord[5] = max_t - offset * (1.0f - glimage->v_align);
      glimage->coord[6] = 0.0f;
      glimage->coord[7] = glimage->coord[5];
    }
  /* Adapt coordinates when drawable height is larger than image height */
  else
    {
      gfloat image_width = gldrawable->height * glimage->image_ratio;

      offset = ((image_width - gldrawable->width) * max_s) / image_width;
      glimage->coord[0] = offset * glimage->h_align;
      glimage->coord[1] = 0.0f;
      glimage->coord[2] = max_s - offset * (1.0f - glimage->h_align);
      glimage->coord[3] = 0.0f;
      glimage->coord[4] = glimage->coord[2];
      glimage->coord[5] = max_t;
      glimage->coord[6] = glimage->coord[0];
      glimage->coord[7] = max_t;
    }

  GST_OBJECT_UNLOCK (glimage);
}

/* Modifies texture coordinates to fill the whole stored texture */
static void
set_standard_coordinates (PgmGlImage *glimage)
{
  PgmGlDrawable *gldrawable = PGM_GL_DRAWABLE (glimage);
  PgmImage *image = PGM_IMAGE (gldrawable->drawable);
  PgmTexture *texture;
  gfloat pixel_s, pixel_t;

  if (image->storage_type != PGM_IMAGE_IMAGE)
    texture = glimage->texture;
  else
    texture = glimage->master;

  GST_OBJECT_LOCK (glimage);
  PGM_TEXTURE_LOCK (texture);

  /* Remove the thin one pixel lines on bottom and right of the texture */
  pixel_s = 1.0f / texture->width_pot;
  pixel_t = 1.0f / texture->height_pot;

  glimage->coord[0] = 0.0f;
  glimage->coord[1] = 0.0f;
  glimage->coord[2] = (gfloat) texture->width / texture->width_pot - pixel_s;
  glimage->coord[3] = 0.0f;
  glimage->coord[4] = glimage->coord[2];
  glimage->coord[5] = (gfloat) texture->height / texture->height_pot - pixel_t;
  glimage->coord[6] = 0.0f;
  glimage->coord[7] = glimage->coord[5];

  PGM_TEXTURE_UNLOCK (texture);
  GST_OBJECT_UNLOCK (glimage);
}

/* Adapts properties to a size change depending on the layout */
static void
update_layout (PgmGlImage *glimage)
{
  PgmDrawable *drawable = PGM_GL_DRAWABLE (glimage)->drawable;
  PgmImage *image = PGM_IMAGE (drawable);
  PgmImageLayoutType layout;

  GST_OBJECT_LOCK (image);
  layout = image->layout;
  GST_OBJECT_UNLOCK (image);

  switch (layout)
    {
    case PGM_IMAGE_SCALED:
    case PGM_IMAGE_CENTERED: /* FIXME: Not implemented yet */
    case PGM_IMAGE_TILED:    /* FIXME: Not implemented yet */
      set_scaled_vertices (glimage);
      set_standard_coordinates (glimage);
      break;

    case PGM_IMAGE_ZOOMED:
      set_standard_vertices (glimage);
      set_zoomed_coordinates (glimage);
      break;

    case PGM_IMAGE_FILLED:
      set_standard_vertices (glimage);
      set_standard_coordinates (glimage);
      break;

    default:
      break;
    }
}

/* Adapts to a drawable position change */
static void
update_vertices_position (PgmGlImage *glimage)
{
  PgmGlDrawable *gldrawable = PGM_GL_DRAWABLE (glimage);
  PgmGlFloat *img_vertex = glimage->vertex;
  guint i;

  GST_OBJECT_LOCK (glimage);

  /* Add the offsets to the current vertices */
  for (i = 0; i < 12; i += 3)
    {
      img_vertex[i+0] += gldrawable->x - glimage->last_x;
      img_vertex[i+1] += gldrawable->y - glimage->last_y;
      img_vertex[i+2] += gldrawable->z - glimage->last_z;
    }

  GST_OBJECT_UNLOCK (glimage);

  /* And update the last position with the new one */
  update_last_position (glimage);
}

/* Update the image ratio and layout of all the slaves of the given image */
static void
update_slaves (PgmGlImage *glimage)
{
  PgmGlDrawable *gldrawable = PGM_GL_DRAWABLE (glimage);
  PgmGlViewport *glviewport = gldrawable->glviewport;
  PgmImage *master = PGM_IMAGE (gldrawable->drawable);
  PgmGlImage *slave;
  GList *walk;

  GST_OBJECT_LOCK (master);

  walk = master->slaves;
  while (walk)
    {
      /* Retrieve the slave glimage */
      GST_OBJECT_LOCK (glviewport);
      slave = g_hash_table_lookup (glviewport->drawable_hash, walk->data);
      GST_OBJECT_UNLOCK (glviewport);

      /* Update slave glimage parameters */
      if (slave)
        {
          GST_OBJECT_UNLOCK (master);
          update_image_ratio (slave);
          update_layout (slave);
          GST_OBJECT_LOCK (master);
        }

      walk = walk->next;
    }

  GST_OBJECT_UNLOCK (master);
}

/* PgmGlDrawable methods */

static void
pgm_gl_image_draw (PgmGlDrawable *gldrawable)
{
  PgmImage *image = PGM_IMAGE (gldrawable->drawable);
  PgmGlImage *glimage = PGM_GL_IMAGE (gldrawable);
  PgmContextProcAddress *gl;
  PgmTexture *texture;

  GST_DEBUG_OBJECT (gldrawable, "draw");

  PGM_GL_IMAGE_RETURN_IF_EMPTY (glimage);

  gl = gldrawable->glviewport->context->gl;

  if (image->storage_type != PGM_IMAGE_IMAGE)
    texture = glimage->texture;
  else
    texture = glimage->master;

  pgm_texture_bind (texture);

  gl->enable_client_state (PGM_GL_VERTEX_ARRAY);
  gl->enable_client_state (PGM_GL_TEXTURE_COORD_ARRAY);

  GST_OBJECT_LOCK (glimage);
  gl->vertex_pointer (3, PGM_GL_FLOAT, 0, glimage->vertex);
  gl->tex_coord_pointer (2, PGM_GL_FLOAT, 0, glimage->coord);
  gl->color_4fv (glimage->fg_color);
  GST_OBJECT_UNLOCK (glimage);

  gl->draw_arrays (PGM_GL_QUADS, 0, 4);
  gl->disable_client_state (PGM_GL_VERTEX_ARRAY);
  gl->disable_client_state (PGM_GL_TEXTURE_COORD_ARRAY);

  pgm_texture_unbind (texture);
}

static void
pgm_gl_image_regenerate (PgmGlDrawable *gldrawable)
{
  GST_DEBUG_OBJECT (gldrawable, "regenerate");
}

static void
pgm_gl_image_update_projection (PgmGlDrawable *gldrawable)
{
  PgmGlImage *glimage = PGM_GL_IMAGE (gldrawable);

  GST_DEBUG_OBJECT (gldrawable, "update_projection");

  update_image_ratio (glimage);
  update_layout (glimage);
}

static void
pgm_gl_image_set_size (PgmGlDrawable *gldrawable)
{
  PgmGlImage *glimage = PGM_GL_IMAGE (gldrawable);

  GST_DEBUG_OBJECT (gldrawable, "set_size");

  update_drawable_ratio (glimage);
  update_layout (glimage);
}

static void
pgm_gl_image_set_position (PgmGlDrawable *gldrawable)
{
  PgmGlImage *glimage = PGM_GL_IMAGE (gldrawable);

  GST_DEBUG_OBJECT (gldrawable, "set_position");

  update_vertices_position (glimage);
}

static void
pgm_gl_image_set_fg_color (PgmGlDrawable *gldrawable)
{
  PgmGlImage *glimage = PGM_GL_IMAGE (gldrawable);
  PgmDrawable *drawable;

  GST_DEBUG_OBJECT (gldrawable, "set_fg_color");

  drawable = gldrawable->drawable;

  GST_OBJECT_LOCK (drawable);
  GST_OBJECT_LOCK (glimage);

  glimage->fg_color[0] = drawable->fg_r * INV_255;
  glimage->fg_color[1] = drawable->fg_g * INV_255;
  glimage->fg_color[2] = drawable->fg_b * INV_255;
  glimage->fg_color[3] = drawable->fg_a * drawable->opacity * SQR_INV_255;

  GST_OBJECT_UNLOCK (glimage);
  GST_OBJECT_UNLOCK (drawable);
}

static void
pgm_gl_image_set_opacity (PgmGlDrawable *gldrawable)
{
  PgmGlImage *glimage = PGM_GL_IMAGE (gldrawable);
  PgmDrawable *drawable;

  GST_DEBUG_OBJECT (gldrawable, "set_opacity");

  drawable = gldrawable->drawable;

  GST_OBJECT_LOCK (drawable);
  GST_OBJECT_LOCK (glimage);

  glimage->fg_color[3] = drawable->fg_a * drawable->opacity * SQR_INV_255;

  GST_OBJECT_UNLOCK (glimage);
  GST_OBJECT_UNLOCK (drawable);
}

static void
pgm_gl_image_sync (PgmGlDrawable *gldrawable)
{
  PgmGlImage *glimage = PGM_GL_IMAGE (gldrawable);
  PgmImage *image = PGM_IMAGE (gldrawable->drawable);
  PgmImageStorageType type;

  GST_DEBUG_OBJECT (gldrawable, "sync");

  /* Synchronize various properties */
  pgm_gl_image_set_fg_color (gldrawable);
  update_interp (glimage);
  update_alignment (glimage);
  update_last_position (glimage);
  update_drawable_ratio (glimage);
  update_image_ratio (glimage);

  GST_OBJECT_LOCK (image);
  type = image->storage_type;
  GST_OBJECT_UNLOCK (image);

  /* Synchronize the buffer */
  switch (type)
    {
    case PGM_IMAGE_BUFFER:
      pgm_gl_image_set_from_buffer (glimage);
      break;
    case PGM_IMAGE_GST_BUFFER:
      pgm_gl_image_set_from_gst_buffer (glimage);
      break;
    case PGM_IMAGE_FD:
      pgm_gl_image_set_from_fd (glimage);
      break;
    case PGM_IMAGE_IMAGE:
      pgm_gl_image_set_from_image (glimage);
      break;
    case PGM_IMAGE_EMPTY:
      break;
    default:
      break;
    }
}

/* GObject stuff */

G_DEFINE_TYPE (PgmGlImage, pgm_gl_image, PGM_TYPE_GL_DRAWABLE);

static void
pgm_gl_image_dispose (GObject *object)
{
  PgmGlDrawable *gldrawable = PGM_GL_DRAWABLE (object);
  PgmGlImage *glimage = PGM_GL_IMAGE (object);
  PgmContextTask *task;

  GST_DEBUG_OBJECT (glimage, "dispose");

  gst_object_unref (gldrawable->drawable);
  pgm_context_remove_tasks_with_data (gldrawable->glviewport->context,
                                      glimage->texture);
  task = pgm_context_task_new (PGM_CONTEXT_FREE_TEXTURE, glimage->texture);
  pgm_context_push_immediate_task (gldrawable->glviewport->context, task);
  glimage->texture = NULL;

  GST_CALL_PARENT (G_OBJECT_CLASS, dispose, (object));
}

static void
pgm_gl_image_class_init (PgmGlImageClass *klass)
{
  GObjectClass *gobject_class;
  PgmGlDrawableClass *drawable_class;

  parent_class = g_type_class_peek_parent (klass);

  gobject_class = G_OBJECT_CLASS (klass);
  drawable_class = PGM_GL_DRAWABLE_CLASS (klass);

  /* GObject virtual table */
  gobject_class->dispose = GST_DEBUG_FUNCPTR (pgm_gl_image_dispose);

  /* PgmGlDrawable virtual table */
  drawable_class->sync = GST_DEBUG_FUNCPTR (pgm_gl_image_sync);
  drawable_class->draw = GST_DEBUG_FUNCPTR (pgm_gl_image_draw);
  drawable_class->regenerate = GST_DEBUG_FUNCPTR (pgm_gl_image_regenerate);
  drawable_class->update_projection =
    GST_DEBUG_FUNCPTR (pgm_gl_image_update_projection);
  drawable_class->set_size = GST_DEBUG_FUNCPTR (pgm_gl_image_set_size);
  drawable_class->set_position = GST_DEBUG_FUNCPTR (pgm_gl_image_set_position);
  drawable_class->set_fg_color = GST_DEBUG_FUNCPTR (pgm_gl_image_set_fg_color);
  drawable_class->set_opacity = GST_DEBUG_FUNCPTR (pgm_gl_image_set_opacity);
}

static void
pgm_gl_image_init (PgmGlImage *glimage)
{
  GST_DEBUG_OBJECT (glimage, "init");

  glimage->empty = TRUE;
}

/* Public methods */

PgmGlDrawable *
pgm_gl_image_new (PgmDrawable *drawable,
                  PgmGlViewport *glviewport)
{
  PgmGlImage *glimage;
  PgmGlDrawable *gldrawable;

  glimage = g_object_new (PGM_TYPE_GL_IMAGE, NULL);

  GST_DEBUG_OBJECT (glimage, "created new glimage");

  glimage->texture = pgm_texture_new (glviewport->context);
  glimage->master = NULL;

  gldrawable = PGM_GL_DRAWABLE (glimage);
  gldrawable->drawable = gst_object_ref (drawable);
  gldrawable->glviewport = glviewport;
  pgm_gl_drawable_sync (gldrawable);

  return gldrawable;
}

void
pgm_gl_image_clear (PgmGlImage *glimage)
{
  PgmGlDrawable *gldrawable = PGM_GL_DRAWABLE (glimage);
  PgmContextTask *task = NULL;

  GST_DEBUG_OBJECT (glimage, "clear");

  PGM_GL_IMAGE_RETURN_IF_EMPTY (glimage);

  /* Request texture clean up */
  task = pgm_context_task_new (PGM_CONTEXT_CLEAN_TEXTURE, glimage->texture);
  pgm_context_push_immediate_task (gldrawable->glviewport->context, task);

  GST_OBJECT_LOCK (glimage);

  glimage->empty = TRUE;
  glimage->image_ratio = 0.0f;

  GST_OBJECT_UNLOCK (glimage);
}

void
pgm_gl_image_set_from_buffer (PgmGlImage *glimage)
{
  PgmGlDrawable *gldrawable = PGM_GL_DRAWABLE (glimage);
  PgmImage *image = PGM_IMAGE (gldrawable->drawable);
  PgmContextTask *task;

  GST_DEBUG_OBJECT (glimage, "set_from_buffer");

  GST_OBJECT_LOCK (image);
  if (G_UNLIKELY (!(image->storage_type == PGM_IMAGE_BUFFER
                    || image->storage_type == PGM_IMAGE_IMAGE)))
    {
      GST_OBJECT_UNLOCK (image);
      return;
    }
  pgm_texture_set_buffer (glimage->texture, image->data.buffer.buffer,
                          image->data.buffer.format, image->data.buffer.width,
                          image->data.buffer.height, image->data.buffer.size,
                          image->data.buffer.stride);
  GST_OBJECT_UNLOCK (image);

  GST_OBJECT_LOCK (glimage);
  glimage->empty = FALSE;
  GST_OBJECT_UNLOCK (glimage);

  update_image_ratio (glimage);
  update_layout (glimage);
  update_slaves (glimage);

  task = pgm_context_task_new (PGM_CONTEXT_GEN_TEXTURE, glimage->texture);
  pgm_context_push_immediate_task (gldrawable->glviewport->context, task);
  task = pgm_context_task_new (PGM_CONTEXT_UPLOAD_TEXTURE, glimage->texture);
  pgm_context_push_deferred_task (gldrawable->glviewport->context, task);
}

void
pgm_gl_image_set_from_gst_buffer (PgmGlImage *glimage)
{
  PgmGlDrawable *gldrawable = PGM_GL_DRAWABLE (glimage);
  PgmImage *image = PGM_IMAGE (gldrawable->drawable);
  PgmContextTask *task;
  gboolean empty;

  GST_DEBUG_OBJECT (glimage, "set_from_gst_buffer");

  /* Get the empty state with locking */
  GST_OBJECT_LOCK (glimage);
  empty = glimage->empty;
  GST_OBJECT_UNLOCK (glimage);

  /* It's the first time the glimage receives a GstBuffer, let's set the
   * texture buffer with its different informations */
  if (G_UNLIKELY (empty))
    {
      GST_OBJECT_LOCK (image);

      if (G_UNLIKELY (!(image->storage_type == PGM_IMAGE_GST_BUFFER
                        || image->storage_type == PGM_IMAGE_IMAGE)))
        {
          GST_OBJECT_UNLOCK (image);
          return;
        }

      pgm_texture_set_gst_buffer (glimage->texture,
                                  image->data.gst_buffer.gst_buffer,
                                  image->data.gst_buffer.format,
                                  image->data.gst_buffer.width,
                                  image->data.gst_buffer.height,
                                  image->data.gst_buffer.stride);
      GST_OBJECT_UNLOCK (image);

      GST_OBJECT_LOCK (glimage);
      glimage->empty = FALSE;
      GST_OBJECT_UNLOCK (glimage);

      update_image_ratio (glimage);
      update_layout (glimage);
      update_slaves (glimage);

      task = pgm_context_task_new (PGM_CONTEXT_GEN_TEXTURE, glimage->texture);
      pgm_context_push_immediate_task (gldrawable->glviewport->context, task);
    }

  /* It's not the first time, so we just need to send the updated buffer */
  else
    {
      GST_OBJECT_LOCK (image);
      if (G_UNLIKELY (!(image->storage_type == PGM_IMAGE_GST_BUFFER
                        || image->storage_type == PGM_IMAGE_IMAGE)))
        {
          GST_OBJECT_UNLOCK (image);
          return;
        }
      pgm_texture_update_gst_buffer (glimage->texture,
                                     image->data.gst_buffer.gst_buffer);
      GST_OBJECT_UNLOCK (image);
    }

  task = pgm_context_task_new (PGM_CONTEXT_UPLOAD_TEXTURE, glimage->texture);
  pgm_context_push_deferred_task (gldrawable->glviewport->context, task);
}

void
pgm_gl_image_set_from_fd (PgmGlImage *glimage)
{
  PgmGlDrawable *gldrawable = PGM_GL_DRAWABLE (glimage);
  PgmImage *image = PGM_IMAGE (gldrawable->drawable);
  PgmContextTask *task;

  GST_DEBUG_OBJECT (glimage, "set_from_fd");

  GST_OBJECT_LOCK (image);
  /*  */
  if (G_UNLIKELY (!(image->storage_type == PGM_IMAGE_FD
                    || image->storage_type == PGM_IMAGE_IMAGE)))
    {
      GST_OBJECT_UNLOCK (image);
      return;
    }
  /* */
  pgm_texture_set_pixbuf (glimage->texture, image->data.fd.pixbuf);
  GST_OBJECT_UNLOCK (image);

  GST_OBJECT_LOCK (glimage);
  glimage->empty = FALSE;
  GST_OBJECT_UNLOCK (glimage);

  update_image_ratio (glimage);
  update_layout (glimage);
  update_slaves (glimage);

  task = pgm_context_task_new (PGM_CONTEXT_GEN_TEXTURE, glimage->texture);
  pgm_context_push_immediate_task (gldrawable->glviewport->context, task);
  task = pgm_context_task_new (PGM_CONTEXT_UPLOAD_TEXTURE, glimage->texture);
  pgm_context_push_deferred_task (gldrawable->glviewport->context, task);
}

void
pgm_gl_image_set_from_image (PgmGlImage *glimage)
{
  PgmGlDrawable *gldrawable = PGM_GL_DRAWABLE (glimage);
  PgmImage *image = PGM_IMAGE (gldrawable->drawable);
  PgmGlViewport *glviewport = gldrawable->glviewport;
  PgmGlImage *master;

  GST_DEBUG_OBJECT (glimage, "set_from_image");

  /* Retrieve the master glimage */
  GST_OBJECT_LOCK (image);
  if (G_UNLIKELY (image->storage_type != PGM_IMAGE_IMAGE))
    {
      GST_OBJECT_UNLOCK (image);
      return;
    }
  GST_OBJECT_LOCK (glviewport);
  master = g_hash_table_lookup (glviewport->drawable_hash, image->master);
  GST_OBJECT_UNLOCK (glviewport);
  GST_OBJECT_UNLOCK (image);

  /* And use its texture */
  GST_OBJECT_LOCK (glimage);
  glimage->master = master->texture;
  glimage->empty = FALSE;
  GST_OBJECT_UNLOCK (glimage);

  update_image_ratio (glimage);
  update_layout (glimage);
}

void
pgm_gl_image_set_alignment (PgmGlImage *glimage)
{
  GST_DEBUG_OBJECT (glimage, "set_alignment");

  update_alignment (glimage);
  update_layout (glimage);
}

void
pgm_gl_image_set_layout (PgmGlImage *glimage)
{
  GST_DEBUG_OBJECT (glimage, "set_layout");

  update_layout (glimage);
}

void
pgm_gl_image_set_interp (PgmGlImage *glimage)
{
  PgmGlDrawable *gldrawable = PGM_GL_DRAWABLE (glimage);
  PgmContextTask *task;

  GST_DEBUG_OBJECT (glimage, "set_interp");

  update_interp (glimage);

  task = pgm_context_task_new (PGM_CONTEXT_UPDATE_TEXTURE, glimage->texture);
  pgm_context_push_immediate_task (gldrawable->glviewport->context, task);
}

void
pgm_gl_image_set_aspect_ratio (PgmGlImage *glimage)
{
  GST_DEBUG_OBJECT (glimage, "set_aspect_ratio");

  update_image_ratio (glimage);
  update_layout (glimage);
}

/* FIXME */
void
pgm_gl_image_alloc_gst_buffer (PgmGlImage *glimage)
{
  GST_DEBUG_OBJECT (glimage, "alloc_gst_buffer");
}
