/* -*- Mode: C; c-basic-offset: 2; indent-tabs-mode: nil -*-
 *
 * Pigment OpenGL plugin
 *
 * Copyright © 2006, 2007 Fluendo Embedded S.L.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * Author: Loïc Molinari <loic@fluendo.com>
 */

/*
 * The plugin duplicates all "generic" Pigment drawables in GL drawables. The
 * GL drawables are stored in layer (lists) inside the GL viewport object. The
 * generic layers are duplicated when the user calls pgm_viewport_set_canvas().
 * The duplicated GL drawables are deleted when the canvas is disposed, when
 * another canvas is bound or when NULL is bound. The GL Viewport is connected
 * to the Canvas 'added' and 'removed' signals so that it can update the GL
 * drawable layers to reflect the "generic" one.
 *
 * The link between the "generic" and the GL drawable is done through a
 * dedicated hash-table using the "generic" drawable address as the key and the
 * GL drawable address as the value. When a "generic" drawable property changes,
 * it emits the 'changed' signal to which the Gl Viewport is connected. The
 * handler stores the change as a task in an update queue which is flushed
 * when the user request a viewport update through pgm_viewport_update().
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif /* HAVE_CONFIG_H */

#include "pgmglviewport.h"
#include "pgmglimage.h"
#include "pgmgltext.h"

GST_DEBUG_CATEGORY_EXTERN (pgm_gl_debug);
#define GST_CAT_DEFAULT pgm_gl_debug

/* Task types */
typedef enum {
  TASK_CHANGE,
  TASK_REORDER,
  TASK_ADD,
  TASK_REMOVE,
  TASK_REGENERATE,
  LAST_TASK
} TaskType;

/* Change task */
typedef struct {
  TaskType             type;
  PgmGlDrawable       *gldrawable;
  PgmDrawableProperty  property;
} TaskChange;

/* Reorder task */
typedef struct {
  TaskType          type;
  PgmGlDrawable    *drawable;
  PgmDrawableLayer  layer;
  guint             order;
} TaskReorder;

/* Add task */
typedef struct {
  TaskType          type;
  PgmDrawable      *drawable;
  PgmDrawableLayer  layer;
  guint             order;
} TaskAdd;

/* Remove task */
typedef struct {
  TaskType          type;
  PgmDrawable      *drawable;
  PgmDrawableLayer  layer;
} TaskRemove;

/* Any task */
typedef struct {
  TaskType type;
} TaskAny;

/* Task */
typedef union {
  TaskType    type;
  TaskAny     any;
  TaskChange  change;
  TaskReorder reorder;
  TaskAdd     add;
  TaskRemove  remove;
} Task;

/* Task function type definition */
typedef void (*TaskFunc) (PgmGlViewport *glviewport, Task *task);

/* Task function prototypes */
static void do_task_change     (PgmGlViewport *glviewport, Task *task);
static void do_task_reorder    (PgmGlViewport *glviewport, Task *task);
static void do_task_add        (PgmGlViewport *glviewport, Task *task);
static void do_task_remove     (PgmGlViewport *glviewport, Task *task);
static void do_task_regenerate (PgmGlViewport *glviewport, Task *task);

/* Callback prototypes */
static void changed_cb (PgmDrawable *drawable,
                        PgmDrawableProperty property,
                        gpointer data);

/* Task function array */
static TaskFunc task_func[LAST_TASK] = {
  do_task_change,
  do_task_reorder,
  do_task_add,
  do_task_remove,
  do_task_regenerate
};

static PgmViewportClass *parent_class = NULL;

/* Private functions */

/* Create a change task */
static Task*
task_change_new (PgmGlDrawable *gldrawable,
                 PgmDrawableProperty property)
{
  Task *task;

  task = g_slice_new (Task);
  task->change.type = TASK_CHANGE;
  task->change.gldrawable = gst_object_ref (gldrawable);
  task->change.property = property;

  return task;
}

/* Free a change task */
static void
task_change_free (Task *task)
{
  g_return_if_fail (task != NULL);

  gst_object_unref (task->change.gldrawable);

  g_slice_free (Task, task);
}

/* Create a reorder task */
static Task*
task_reorder_new (PgmDrawable *drawable,
                  PgmDrawableLayer layer,
                  guint order)
{
  Task *task;

  task = g_slice_new (Task);
  task->reorder.type = TASK_REORDER;
  task->reorder.drawable = gst_object_ref (drawable);
  task->reorder.layer = layer;
  task->reorder.order = order;

  return task;
}

/* Free a reorder task */
static void
task_reorder_free (Task *task)
{
  g_return_if_fail (task != NULL);

  gst_object_unref (task->reorder.drawable);

  g_slice_free (Task, task);
}

/* Create a add task */
static Task*
task_add_new (PgmDrawable *drawable,
              PgmDrawableLayer layer,
              guint order)
{
  Task *task;

  task = g_slice_new (Task);
  task->add.type = TASK_ADD;
  task->add.drawable = gst_object_ref (drawable);
  task->add.layer = layer;
  task->add.order = order;

  return task;
}

/* Free a add task */
static void
task_add_free (Task *task)
{
  g_return_if_fail (task != NULL);

  gst_object_unref (task->add.drawable);

  g_slice_free (Task, task);
}

/* Create a remove task */
static Task*
task_remove_new (PgmDrawable *drawable,
                 PgmDrawableLayer layer)
{
  Task *task;

  task = g_slice_new (Task);
  task->remove.type = TASK_REMOVE;
  task->remove.drawable = gst_object_ref (drawable);
  task->remove.layer = layer;

  return task;
}

/* Free a remove task */
static void
task_remove_free (Task *task)
{
  g_return_if_fail (task != NULL);

  gst_object_unref (task->remove.drawable);

  g_slice_free (Task, task);
}

/* Create a new generic task */
static Task*
task_any_new (TaskType type)
{
  Task *task;

  task = g_slice_new (Task);
  task->any.type = type;

  return task;
}

/* Create a new generic task */
static void
task_any_free (Task *task)
{
  g_slice_free (Task, task);
}

/* Free a generic task guessing the type */
static void
task_free (Task *task)
{
  switch (task->type)
    {
    case TASK_CHANGE:
      task_change_free (task);
      break;
    case TASK_REORDER:
      task_reorder_free (task);
      break;
    case TASK_ADD:
      task_add_free (task);
      break;
    case TASK_REMOVE:
      task_remove_free (task);
      break;
    default:
      task_any_free (task);
      break;
    }
}

/* Create a GL drawable from a generic drawable */
static PgmGlDrawable*
gl_drawable_new (PgmGlViewport *glviewport,
                 PgmDrawable *drawable)
{
  PgmGlDrawable *gldrawable = NULL;

  /* Create the GlDrawable depending on the type */
  if (PGM_IS_IMAGE (drawable))
    {
      gldrawable = pgm_gl_image_new (drawable, glviewport);
      GST_DEBUG_OBJECT (glviewport, "created %s", GST_OBJECT_NAME (gldrawable));
    }
  else if (PGM_IS_TEXT (drawable))
    {
      gldrawable = pgm_gl_text_new (drawable, glviewport);
      GST_DEBUG_OBJECT (glviewport, "created %s", GST_OBJECT_NAME (gldrawable));
    }
  else
    {
      gldrawable = NULL;
      GST_DEBUG_OBJECT (glviewport, "cannot create object from this type");
    }

  /* Make it easily retrievable and connect to the 'changed' signal so that
   * changes could be reflected in the plugin */
  if (gldrawable)
    {
      GST_OBJECT_LOCK (glviewport);
      GST_OBJECT_LOCK (drawable);

      g_hash_table_insert (glviewport->drawable_hash, drawable, gldrawable);
      gldrawable->change_handler = g_signal_connect (drawable, "changed",
                                                     G_CALLBACK (changed_cb),
                                                     (gpointer) gldrawable);

      GST_OBJECT_UNLOCK (drawable);
      GST_OBJECT_UNLOCK (glviewport);
    }

  return gldrawable;
}

/* Free a GL drawable */
static void
gl_drawable_free (PgmGlViewport *glviewport,
                  PgmGlDrawable *gldrawable)
{
  GSList *walk, *tmp;
  Task *task;

  /* Disconnect from the 'changed' signal and remove it from the hash */
  GST_OBJECT_LOCK (glviewport);
  GST_OBJECT_LOCK (gldrawable->drawable);

  g_signal_handler_disconnect (gldrawable->drawable, gldrawable->change_handler);
  g_hash_table_remove (glviewport->drawable_hash, gldrawable->drawable);

  GST_OBJECT_UNLOCK (gldrawable->drawable);
  GST_OBJECT_UNLOCK (glviewport);

  /* Remove the gldrawable from the update queue */
  g_mutex_lock (glviewport->update_lock);
  walk = glviewport->update_queue;
  while (walk)
    {
      tmp = walk->next;
      task = walk->data;
      /* Remove it only if it's a task dealing with gldrawable */
      if (task->type == TASK_CHANGE)
        {
          if (task->change.gldrawable == gldrawable)
            {
              glviewport->update_queue =
                g_slist_delete_link (glviewport->update_queue, walk);
              task_change_free (task);
            }
        }
      walk = tmp;
    }
  g_mutex_unlock (glviewport->update_lock);

  /* Then delete it */
  GST_DEBUG_OBJECT (glviewport, "deleting %s", GST_OBJECT_NAME (gldrawable));
  gst_object_unref (gldrawable);
}

/* Complete a change task and free it */
static void
do_task_change (PgmGlViewport *glviewport,
                Task *task)
{
  PgmGlViewportClass *klass = PGM_GL_VIEWPORT_GET_CLASS (glviewport);

  klass->changed_func[task->change.property] (task->change.gldrawable);

  task_change_free (task);
}

/* Complete a reorder task and free it */
static void
do_task_reorder (PgmGlViewport *glviewport,
                 Task *task)
{
  PgmGlDrawable *gldrawable;

  GST_OBJECT_LOCK (glviewport);
  gldrawable = g_hash_table_lookup (glviewport->drawable_hash,
                                    task->reorder.drawable);
  GST_OBJECT_UNLOCK (glviewport);

  if (!gldrawable)
    goto removed;

  g_mutex_lock (glviewport->layer_lock);

  switch (task->reorder.layer)
    {
    case PGM_DRAWABLE_NEAR:
      glviewport->near = g_list_remove (glviewport->near, gldrawable);
      glviewport->near = g_list_insert (glviewport->near, gldrawable,
                                        task->reorder.order);
      break;

    case PGM_DRAWABLE_MIDDLE:
      glviewport->middle = g_list_remove (glviewport->middle, gldrawable);
      glviewport->middle = g_list_insert (glviewport->middle, gldrawable,
                                          task->reorder.order);
      break;

    case PGM_DRAWABLE_FAR:
      glviewport->far = g_list_remove (glviewport->far, gldrawable);
      glviewport->far = g_list_insert (glviewport->far, gldrawable,
                                       task->reorder.order);
      break;

    default:
      break;
    }

  g_mutex_unlock (glviewport->layer_lock);

 removed:
  task_reorder_free (task);
}

/* Complete a add task and free it */
static void
do_task_add (PgmGlViewport *glviewport,
             Task *task)
{
  PgmGlDrawable *gldrawable;

  gldrawable = gl_drawable_new (glviewport, task->add.drawable);

  g_mutex_lock (glviewport->layer_lock);
  switch (task->add.layer)
    {
    case PGM_DRAWABLE_NEAR:
      glviewport->near = g_list_insert (glviewport->near, gldrawable,
                                        task->add.order);
      break;

    case PGM_DRAWABLE_MIDDLE:
      glviewport->middle = g_list_insert (glviewport->middle, gldrawable,
                                          task->add.order);
      break;

    case PGM_DRAWABLE_FAR:
      glviewport->far = g_list_insert (glviewport->far, gldrawable,
                                       task->add.order);
      break;

    default:
      break;
    }
  g_mutex_unlock (glviewport->layer_lock);

  task_add_free (task);
}

/* Complete a remove task and free it */
static void
do_task_remove (PgmGlViewport *glviewport,
                Task *task)
{
  PgmGlDrawable *gldrawable;

  /* Get the GlDrawable */
  GST_OBJECT_LOCK (glviewport);
  gldrawable = g_hash_table_lookup (glviewport->drawable_hash,
                                    task->remove.drawable);
  GST_OBJECT_UNLOCK (glviewport);

  /* Remove it from the list */
  g_mutex_lock (glviewport->layer_lock);
  switch (task->remove.layer)
    {
    case PGM_DRAWABLE_NEAR:
      glviewport->near = g_list_remove (glviewport->near, gldrawable);
      break;

    case PGM_DRAWABLE_MIDDLE:
      glviewport->middle = g_list_remove (glviewport->middle, gldrawable);
      break;

    case PGM_DRAWABLE_FAR:
      glviewport->far = g_list_remove (glviewport->far, gldrawable);
      break;

    default:
      break;
    }
  g_mutex_unlock (glviewport->layer_lock);

  /* And delete it */
  gl_drawable_free (glviewport, gldrawable);

  task_remove_free (task);
}

/* Complete a regenerate task and free it */
static void
do_task_regenerate (PgmGlViewport *glviewport,
                    Task *task)
{
  g_mutex_lock (glviewport->layer_lock);

  g_list_foreach (glviewport->near, (GFunc) pgm_gl_drawable_regenerate, NULL);
  g_list_foreach (glviewport->middle, (GFunc) pgm_gl_drawable_regenerate, NULL);
  g_list_foreach (glviewport->far, (GFunc) pgm_gl_drawable_regenerate, NULL);

  g_mutex_unlock (glviewport->layer_lock);

  task_any_free (task);
}

/* Drawable 'changed' handler */
static void
changed_cb (PgmDrawable *drawable,
            PgmDrawableProperty property,
            gpointer data)
{
  PgmGlDrawable *gldrawable = PGM_GL_DRAWABLE (data);
  PgmGlViewport *glviewport = gldrawable->glviewport;

  GST_DEBUG_OBJECT (glviewport, "drawable_changed_cb");

  if (glviewport)
    {
      Task *task;

      /* Add the change task to the update queue */
      g_mutex_lock (glviewport->update_lock);
      task = task_change_new (gldrawable, property);
      glviewport->update_queue = g_slist_prepend (glviewport->update_queue,
                                                  task);
      g_mutex_unlock (glviewport->update_lock);
    }
}

/* Canvas 'drawable-added' handler */
static void
drawable_added_cb (PgmCanvas *canvas,
                   PgmDrawable *drawable,
                   PgmDrawableLayer layer,
                   gint order,
                   gpointer data)
{
  PgmGlViewport *glviewport = PGM_GL_VIEWPORT (data);
  Task *task;

  GST_DEBUG_OBJECT (glviewport, "drawable_added_cb");

  /* Add the add task in the update queue */
  task = task_add_new (drawable, layer, order);
  g_mutex_lock (glviewport->update_lock);
  glviewport->update_queue = g_slist_prepend (glviewport->update_queue, task);
  g_mutex_unlock (glviewport->update_lock);
}

/* Canvas 'drawable-removed' handler */
static void
drawable_removed_cb (PgmCanvas *canvas,
                     PgmDrawable *drawable,
                     PgmDrawableLayer layer,
                     gpointer data)
{
  PgmGlViewport *glviewport = PGM_GL_VIEWPORT (data);
  Task *task;

  GST_DEBUG_OBJECT (glviewport, "drawable_removed_cb");

  /* Add the remove task in the update queue */
  task = task_remove_new (drawable, layer);
  g_mutex_lock (glviewport->update_lock);
  glviewport->update_queue = g_slist_prepend (glviewport->update_queue, task);
  g_mutex_unlock (glviewport->update_lock);
}

/* Canvas 'drawable-reordered' handler */
static void
drawable_reordered_cb (PgmCanvas *canvas,
                       PgmDrawable *drawable,
                       PgmDrawableLayer layer,
                       gint order,
                       gpointer data)
{
  PgmGlViewport *glviewport = PGM_GL_VIEWPORT (data);
  Task *task;

  GST_DEBUG_OBJECT (glviewport, "drawable_reordered_cb");

  /* Add the reorder task in the update queue */
  task = task_reorder_new (drawable, layer, order);
  g_mutex_lock (glviewport->update_lock);
  glviewport->update_queue = g_slist_prepend (glviewport->update_queue, task);
  g_mutex_unlock (glviewport->update_lock);
}

/* Canvas 'regenerated' handler */
static void
regenerated_cb (PgmCanvas *canvas,
                gpointer data)
{
  PgmGlViewport *glviewport = PGM_GL_VIEWPORT (data);
  Task *task;

  GST_DEBUG_OBJECT (glviewport, "drawable_regenerated_cb");

  /* Add the regenerate task in the update queue */
  task = task_any_new (TASK_REGENERATE);
  g_mutex_lock (glviewport->update_lock);
  glviewport->update_queue = g_slist_prepend (glviewport->update_queue, task);
  g_mutex_unlock (glviewport->update_lock);
}

/* Delete all GL drawable from the given GL viewport layer list and free it */
static void
delete_all_gl_drawable_from_layer (PgmGlViewport *glviewport,
                                   GList **layer)
{
  GList *walk = *layer;
  PgmGlDrawable *gldrawable;

  while (walk)
    {
      gldrawable = (PgmGlDrawable*) walk->data;
      gl_drawable_free (glviewport, gldrawable);
      walk = walk->next;
    }

  GST_OBJECT_LOCK (glviewport);
  g_list_free (*layer);
  *layer = NULL;
  GST_OBJECT_UNLOCK (glviewport);
}

/* Synchronize the drawables from the given canvas creating and adding the
 * the corresponding GL drawables in the layers */
static void
sync_gl_drawable_from_canvas (PgmGlViewport *glviewport,
                              PgmCanvas *canvas)
{
  PgmGlDrawable *gldrawable;
  GList *walk;

  /* Sync near layer */
  walk = canvas->near;
  while (walk)
    {
      gldrawable = gl_drawable_new (glviewport, (PgmDrawable*) walk->data);
      g_mutex_lock (glviewport->layer_lock);
      glviewport->near = g_list_append (glviewport->near, gldrawable);
      g_mutex_unlock (glviewport->layer_lock);
      walk = walk->next;
    }

  /* Sync middle layer */
  walk = canvas->middle;
  while (walk)
    {
      gldrawable = gl_drawable_new (glviewport, (PgmDrawable*) walk->data);
      g_mutex_lock (glviewport->layer_lock);
      glviewport->middle = g_list_append (glviewport->middle, gldrawable);
      g_mutex_unlock (glviewport->layer_lock);
      walk = walk->next;
    }

  /* Sync far layer */
  walk = canvas->far;
  while (walk)
    {
      gldrawable = gl_drawable_new (glviewport, (PgmDrawable*) walk->data);
      g_mutex_lock (glviewport->layer_lock);
      glviewport->far = g_list_append (glviewport->far, gldrawable);
      g_mutex_unlock (glviewport->layer_lock);
      walk = walk->next;
    }
}

/* PgmViewport methods */

static PgmError
pgm_gl_viewport_update (PgmViewport *viewport)
{
  PgmGlViewport *glviewport = PGM_GL_VIEWPORT (viewport);

  GST_DEBUG_OBJECT (glviewport, "update");

  /* Request an update in the rendering thread */
  pgm_context_update (glviewport->context);

  return PGM_ERROR_OK;
}

static PgmError
pgm_gl_viewport_set_title (PgmViewport *viewport,
                           const gchar *title)
{
  PgmGlViewport *glviewport = PGM_GL_VIEWPORT (viewport);
  PgmContextTask *task;

  GST_DEBUG_OBJECT (glviewport, "set_title");

  task = pgm_context_task_new (PGM_CONTEXT_TITLE, NULL);
  pgm_context_push_immediate_task (glviewport->context, task);

  return PGM_ERROR_OK;
}

static PgmError
pgm_gl_viewport_show (PgmViewport *viewport)
{
  PgmGlViewport *glviewport = PGM_GL_VIEWPORT (viewport);

  GST_DEBUG_OBJECT (glviewport, "show");

  pgm_backend_show (glviewport->context->backend);

  return PGM_ERROR_OK;
}

static PgmError
pgm_gl_viewport_hide (PgmViewport *viewport)
{
  PgmGlViewport *glviewport = PGM_GL_VIEWPORT (viewport);

  GST_DEBUG_OBJECT (glviewport, "hide");

  pgm_backend_hide (glviewport->context->backend);

  return PGM_ERROR_OK;
}

static PgmError
pgm_gl_viewport_set_cursor (PgmViewport *viewport,
                            PgmViewportCursor cursor)
{
  PgmGlViewport *glviewport = PGM_GL_VIEWPORT (viewport);
  PgmContextTask *task;

  GST_DEBUG_OBJECT (glviewport, "set_cursor");

  task = pgm_context_task_new (PGM_CONTEXT_CURSOR, NULL);
  pgm_context_push_immediate_task (glviewport->context, task);

  return PGM_ERROR_OK;
}

static PgmError
pgm_gl_viewport_set_icon (PgmViewport *viewport,
                          GdkPixbuf *icon)
{
  PgmGlViewport *glviewport = PGM_GL_VIEWPORT (viewport);
  PgmContextTask *task;

  GST_DEBUG_OBJECT (glviewport, "set_icon");

  task = pgm_context_task_new (PGM_CONTEXT_ICON, NULL);
  pgm_context_push_immediate_task (glviewport->context, task);

  return PGM_ERROR_OK;
}

static PgmError
pgm_gl_viewport_set_size (PgmViewport *viewport,
                          gint width,
                          gint height)
{
  PgmGlViewport *glviewport = PGM_GL_VIEWPORT (viewport);
  PgmContextTask *task;

  GST_DEBUG_OBJECT (glviewport, "set_size");

  task = pgm_context_task_new (PGM_CONTEXT_SIZE, NULL);
  pgm_context_push_immediate_task (glviewport->context, task);

  return PGM_ERROR_OK;
}

static PgmError
pgm_gl_viewport_set_fullscreen (PgmViewport *viewport,
                                gboolean fullscreen)
{
  PgmGlViewport *glviewport = PGM_GL_VIEWPORT (viewport);
  PgmContextTask *task;

  GST_DEBUG_OBJECT (glviewport, "set_fullscreen");

  task = pgm_context_task_new (PGM_CONTEXT_FULLSCREEN, NULL);
  pgm_context_push_immediate_task (glviewport->context, task);

  return PGM_ERROR_OK;
}

static PgmError
pgm_gl_viewport_get_screen_resolution (PgmViewport *viewport,
                                       gint *width,
                                       gint *height)
{
  PgmGlViewport *glviewport = PGM_GL_VIEWPORT (viewport);

  GST_DEBUG_OBJECT (glviewport, "get_screen_resolution");

  pgm_backend_get_screen_resolution (glviewport->context->backend, width,
                                     height);

  return PGM_ERROR_OK;
}

static PgmError
pgm_gl_viewport_get_screen_size_mm (PgmViewport *viewport,
                                    gint *width,
                                    gint *height)
{
  PgmGlViewport *glviewport = PGM_GL_VIEWPORT (viewport);

  GST_DEBUG_OBJECT (glviewport, "get_screen_size_mm");

  pgm_backend_get_screen_size_mm (glviewport->context->backend, width, height);

  return PGM_ERROR_OK;
}

static PgmError
pgm_gl_viewport_set_canvas (PgmViewport *viewport,
                            PgmCanvas *canvas)
{
  PgmGlViewport *glviewport = PGM_GL_VIEWPORT (viewport);

  GST_DEBUG_OBJECT (glviewport, "set_canvas");

  /* There's a canvas bound, let's clear associated data */
  if (glviewport->canvas)
    {
      GST_OBJECT_LOCK (glviewport);
      GST_OBJECT_LOCK (glviewport->canvas);

      /* Disconnect handlers */
      g_signal_handler_disconnect (glviewport->canvas,
                                   glviewport->add_handler);
      g_signal_handler_disconnect (glviewport->canvas,
                                   glviewport->remove_handler);
      g_signal_handler_disconnect (glviewport->canvas,
                                   glviewport->reorder_handler);
      g_signal_handler_disconnect (glviewport->canvas,
                                   glviewport->regenerated_handler);

      GST_OBJECT_UNLOCK (glviewport->canvas);
      GST_OBJECT_UNLOCK (glviewport);

      /* Delete our hierarchy */
      delete_all_gl_drawable_from_layer (glviewport, &glviewport->near);
      delete_all_gl_drawable_from_layer (glviewport, &glviewport->middle);
      delete_all_gl_drawable_from_layer (glviewport, &glviewport->far);

      GST_OBJECT_LOCK (glviewport);
      glviewport->canvas = NULL;
      GST_OBJECT_UNLOCK (glviewport);
    }

  /* Bind the new canvas if any */
  if (canvas)
    {
      sync_gl_drawable_from_canvas (glviewport, canvas);

      GST_OBJECT_LOCK (glviewport);
      GST_OBJECT_LOCK (canvas);

      /* Connect handlers */
      glviewport->add_handler =
        g_signal_connect (G_OBJECT (canvas), "drawable-added",
                          G_CALLBACK (drawable_added_cb),
                          (gpointer) glviewport);
      glviewport->remove_handler =
        g_signal_connect (G_OBJECT (canvas), "drawable-removed",
                          G_CALLBACK (drawable_removed_cb),
                          (gpointer) glviewport);
      glviewport->reorder_handler =
        g_signal_connect (G_OBJECT (canvas), "drawable-reordered",
                          G_CALLBACK (drawable_reordered_cb),
                          (gpointer) glviewport);
      glviewport->regenerated_handler =
        g_signal_connect (G_OBJECT (canvas), "regenerated",
                          G_CALLBACK (regenerated_cb),
                          (gpointer) glviewport);

      glviewport->canvas = canvas;

      GST_OBJECT_UNLOCK (canvas);
      GST_OBJECT_UNLOCK (glviewport);
    }

  return PGM_ERROR_OK;
}

static PgmError
pgm_gl_viewport_update_projection (PgmViewport *viewport)
{
  PgmGlViewport *glviewport = PGM_GL_VIEWPORT (viewport);
  PgmContextTask *task;

  GST_DEBUG_OBJECT (glviewport, "update_projection");

  task = pgm_context_task_new (PGM_CONTEXT_PROJECTION, NULL);
  pgm_context_push_immediate_task (glviewport->context, task);

  return PGM_ERROR_OK;
}

static PgmError
pgm_gl_viewport_get_pixel_formats (PgmViewport *viewport,
                                   gulong *formats_mask)
{
  PgmGlViewport *glviewport = PGM_GL_VIEWPORT (viewport);
  gchar *variable;

  GST_DEBUG_OBJECT (glviewport, "get_pixel_formats");

  GST_OBJECT_LOCK (glviewport);

  /* The classic formats supported by all hardware */
  *formats_mask = PGM_IMAGE_RGB
                | PGM_IMAGE_RGBA
                | PGM_IMAGE_BGR
                | PGM_IMAGE_BGRA;

  /* I420 and YV12 are supported through fragment programs and
   * multi-texturing with a minimum texture units number of three. The
   * PGM_GL_CSP_PROGRAM environment variable allows to disable the use
   * fragment programs to do the color space conversion. */
  if (glviewport->context->feature_mask & PGM_GL_FEAT_PER_PLANE_YCBCR_PROGRAM)
    {
      *formats_mask |= PGM_IMAGE_I420 | PGM_IMAGE_YV12;

      variable = getenv ("PGM_GL_CSP_PROGRAM");
      if (variable && variable[0] == '0')
        *formats_mask &= ~(PGM_IMAGE_I420 | PGM_IMAGE_YV12);
    }

  /* FIXME: YUYV and UYVY are not supported */

  GST_OBJECT_UNLOCK (glviewport);

  return PGM_ERROR_OK;
}

static PgmError
pgm_gl_viewport_get_caps_mask (PgmViewport *viewport,
                               gulong *caps_mask)
{
  PgmGlViewport *glviewport = PGM_GL_VIEWPORT (viewport);

  GST_DEBUG_OBJECT (glviewport, "get_caps_mask");

  GST_OBJECT_LOCK (glviewport);

  /* Init mask */
  *caps_mask = 0x0;

  /* And set the acceleration flag */
  if (pgm_backend_is_accelerated (glviewport->context->backend))
    *caps_mask |= PGM_VIEWPORT_HARDWARE_ACCELERATION;

  GST_OBJECT_UNLOCK (glviewport);

  return PGM_ERROR_OK;
}

/* GObject stuff */

G_DEFINE_TYPE (PgmGlViewport, pgm_gl_viewport, PGM_TYPE_VIEWPORT);

static void
pgm_gl_viewport_dispose (GObject *object)
{
  PgmGlViewport *glviewport = PGM_GL_VIEWPORT (object);

  GST_DEBUG_OBJECT (glviewport, "dispose");

  g_slist_foreach (glviewport->update_queue, (GFunc) task_free, NULL);
  g_slist_free (glviewport->update_queue);

  if (glviewport->canvas)
    {
      /* Disconnect handlers */
      g_signal_handler_disconnect (glviewport->canvas,
                                   glviewport->add_handler);
      g_signal_handler_disconnect (glviewport->canvas,
                                   glviewport->remove_handler);
      g_signal_handler_disconnect (glviewport->canvas,
                                   glviewport->reorder_handler);
      g_signal_handler_disconnect (glviewport->canvas,
                                   glviewport->regenerated_handler);

      /* Delete our hierarchy */
      delete_all_gl_drawable_from_layer (glviewport, &glviewport->near);
      delete_all_gl_drawable_from_layer (glviewport, &glviewport->middle);
      delete_all_gl_drawable_from_layer (glviewport, &glviewport->far);
    }

  pgm_context_free (glviewport->context);

  g_mutex_free (glviewport->layer_lock);
  g_mutex_free (glviewport->update_lock);

  GST_CALL_PARENT (G_OBJECT_CLASS, dispose, (object));
}

static void
pgm_gl_viewport_class_init (PgmGlViewportClass *klass)
{
  GObjectClass *gobject_class;
  PgmViewportClass *viewport_class;

  parent_class = g_type_class_peek_parent (klass);

  gobject_class = G_OBJECT_CLASS (klass);
  viewport_class = PGM_VIEWPORT_CLASS (klass);

  /* GObject virtual table */
  gobject_class->dispose = GST_DEBUG_FUNCPTR (pgm_gl_viewport_dispose);

  /* PgmViewport virtual table */
  viewport_class->update = GST_DEBUG_FUNCPTR (pgm_gl_viewport_update);
  viewport_class->set_title = GST_DEBUG_FUNCPTR (pgm_gl_viewport_set_title);
  viewport_class->show = GST_DEBUG_FUNCPTR (pgm_gl_viewport_show);
  viewport_class->hide = GST_DEBUG_FUNCPTR (pgm_gl_viewport_hide);
  viewport_class->set_cursor = GST_DEBUG_FUNCPTR (pgm_gl_viewport_set_cursor);
  viewport_class->set_icon = GST_DEBUG_FUNCPTR (pgm_gl_viewport_set_icon);
  viewport_class->set_size = GST_DEBUG_FUNCPTR (pgm_gl_viewport_set_size);
  viewport_class->set_fullscreen =
    GST_DEBUG_FUNCPTR (pgm_gl_viewport_set_fullscreen);
  viewport_class->get_screen_resolution =
    GST_DEBUG_FUNCPTR (pgm_gl_viewport_get_screen_resolution);
  viewport_class->get_screen_size_mm =
    GST_DEBUG_FUNCPTR (pgm_gl_viewport_get_screen_size_mm);
  viewport_class->set_canvas = GST_DEBUG_FUNCPTR (pgm_gl_viewport_set_canvas);
  viewport_class->update_projection =
    GST_DEBUG_FUNCPTR (pgm_gl_viewport_update_projection);
  viewport_class->get_pixel_formats =
    GST_DEBUG_FUNCPTR (pgm_gl_viewport_get_pixel_formats);
  viewport_class->get_caps_mask =
    GST_DEBUG_FUNCPTR (pgm_gl_viewport_get_caps_mask);

  /* PgmGlDrawable changed table */
  klass->changed_func[PGM_DRAWABLE_VISIBILITY] =
    GST_DEBUG_FUNCPTR (pgm_gl_drawable_set_visibility);
  klass->changed_func[PGM_DRAWABLE_SIZE] =
    GST_DEBUG_FUNCPTR (pgm_gl_drawable_set_size);
  klass->changed_func[PGM_DRAWABLE_POSITION] =
    GST_DEBUG_FUNCPTR (pgm_gl_drawable_set_position);
  klass->changed_func[PGM_DRAWABLE_BG_COLOR] =
    GST_DEBUG_FUNCPTR (pgm_gl_drawable_set_bg_color);
  klass->changed_func[PGM_DRAWABLE_FG_COLOR] =
    GST_DEBUG_FUNCPTR (pgm_gl_drawable_set_fg_color);
  klass->changed_func[PGM_DRAWABLE_OPACITY] =
    GST_DEBUG_FUNCPTR (pgm_gl_drawable_set_opacity);

  /* PgmGlImage changed table */
  klass->changed_func[PGM_IMAGE_DATA_EMPTY] =
    GST_DEBUG_FUNCPTR ((PgmGlDrawableChangedFunc) pgm_gl_image_clear);
  klass->changed_func[PGM_IMAGE_DATA_BUFFER] =
    GST_DEBUG_FUNCPTR ((PgmGlDrawableChangedFunc) pgm_gl_image_set_from_buffer);
  klass->changed_func[PGM_IMAGE_DATA_GST_BUFFER] =
    GST_DEBUG_FUNCPTR ((PgmGlDrawableChangedFunc) pgm_gl_image_set_from_gst_buffer);
  klass->changed_func[PGM_IMAGE_DATA_FD] =
    GST_DEBUG_FUNCPTR ((PgmGlDrawableChangedFunc) pgm_gl_image_set_from_fd);
  klass->changed_func[PGM_IMAGE_DATA_IMAGE] =
    GST_DEBUG_FUNCPTR ((PgmGlDrawableChangedFunc) pgm_gl_image_set_from_image);
  klass->changed_func[PGM_IMAGE_ALIGNMENT] =
    GST_DEBUG_FUNCPTR ((PgmGlDrawableChangedFunc) pgm_gl_image_set_alignment);
  klass->changed_func[PGM_IMAGE_LAYOUT] =
    GST_DEBUG_FUNCPTR ((PgmGlDrawableChangedFunc) pgm_gl_image_set_layout);
  klass->changed_func[PGM_IMAGE_INTERP] =
    GST_DEBUG_FUNCPTR ((PgmGlDrawableChangedFunc) pgm_gl_image_set_interp);
  klass->changed_func[PGM_IMAGE_ASPECT_RATIO] =
    GST_DEBUG_FUNCPTR ((PgmGlDrawableChangedFunc) pgm_gl_image_set_aspect_ratio);

  /* PgmGlText changed table */
  klass->changed_func[PGM_TEXT_LABEL] =
    GST_DEBUG_FUNCPTR ((PgmGlDrawableChangedFunc) pgm_gl_text_set_label);
  klass->changed_func[PGM_TEXT_MARKUP] =
    GST_DEBUG_FUNCPTR ((PgmGlDrawableChangedFunc) pgm_gl_text_set_markup);
  klass->changed_func[PGM_TEXT_FONT_FAMILY] =
    GST_DEBUG_FUNCPTR ((PgmGlDrawableChangedFunc) pgm_gl_text_set_font_family);
  klass->changed_func[PGM_TEXT_HEIGHT] =
    GST_DEBUG_FUNCPTR ((PgmGlDrawableChangedFunc) pgm_gl_text_set_font_height);
  klass->changed_func[PGM_TEXT_ELLIPSIZE] =
    GST_DEBUG_FUNCPTR ((PgmGlDrawableChangedFunc) pgm_gl_text_set_ellipsize);
  klass->changed_func[PGM_TEXT_JUSTIFY] =
    GST_DEBUG_FUNCPTR ((PgmGlDrawableChangedFunc) pgm_gl_text_set_justify);
  klass->changed_func[PGM_TEXT_ALIGNMENT] =
    GST_DEBUG_FUNCPTR ((PgmGlDrawableChangedFunc) pgm_gl_text_set_alignment);
  klass->changed_func[PGM_TEXT_WRAP] =
    GST_DEBUG_FUNCPTR ((PgmGlDrawableChangedFunc) pgm_gl_text_set_wrap);
  klass->changed_func[PGM_TEXT_GRAVITY] =
    GST_DEBUG_FUNCPTR ((PgmGlDrawableChangedFunc) pgm_gl_text_set_gravity);
  klass->changed_func[PGM_TEXT_STRETCH] =
    GST_DEBUG_FUNCPTR ((PgmGlDrawableChangedFunc) pgm_gl_text_set_stretch);
  klass->changed_func[PGM_TEXT_STYLE] =
    GST_DEBUG_FUNCPTR ((PgmGlDrawableChangedFunc) pgm_gl_text_set_style);
  klass->changed_func[PGM_TEXT_VARIANT] =
    GST_DEBUG_FUNCPTR ((PgmGlDrawableChangedFunc) pgm_gl_text_set_variant);
  klass->changed_func[PGM_TEXT_WEIGHT] =
    GST_DEBUG_FUNCPTR ((PgmGlDrawableChangedFunc) pgm_gl_text_set_weight);
  klass->changed_func[PGM_TEXT_LINE_SPACING] =
    GST_DEBUG_FUNCPTR ((PgmGlDrawableChangedFunc) pgm_gl_text_set_line_spacing);
  klass->changed_func[PGM_TEXT_OUTLINE_COLOR] =
    GST_DEBUG_FUNCPTR ((PgmGlDrawableChangedFunc) pgm_gl_text_set_outline_color);
  klass->changed_func[PGM_TEXT_OUTLINE_WIDTH] =
    GST_DEBUG_FUNCPTR ((PgmGlDrawableChangedFunc) pgm_gl_text_set_outline_width);
}

static void
pgm_gl_viewport_init (PgmGlViewport *glviewport)
{
  GST_DEBUG_OBJECT (glviewport, "init");

  /* Attached canvas */
  glviewport->canvas = NULL;

  /* Empty lists */
  glviewport->far = NULL;
  glviewport->middle = NULL;
  glviewport->near = NULL;
  glviewport->layer_lock = g_mutex_new ();

  /* Update task queue */
  glviewport->update_queue = NULL;
  glviewport->update_lock = g_mutex_new ();

  /* Create the drawable hash using pointers as keys */
  glviewport->drawable_hash = g_hash_table_new (NULL, NULL);

  /* Create context */
  glviewport->context = pgm_context_new (glviewport);
}

/* Public methods */

PgmViewport *
pgm_gl_viewport_new (void)
{
  PgmGlViewport *glviewport;

  glviewport = g_object_new (PGM_TYPE_GL_VIEWPORT, NULL);
  GST_DEBUG_OBJECT (glviewport, "created new glviewport");

  return PGM_VIEWPORT (glviewport);
}

void
pgm_gl_viewport_flush_update_queue (PgmGlViewport *glviewport)
{
  GSList *copy, *walk;
  Task *task;

  /* Get a reverse copy of the update task list, and NULLify it */
  g_mutex_lock (glviewport->update_lock);
  copy = g_slist_reverse (glviewport->update_queue);
  glviewport->update_queue = NULL;
  g_mutex_unlock (glviewport->update_lock);

  /* Flush the update task list */
  walk = copy;
  while (walk)
    {
      task = walk->data;
      task_func[task->type] (glviewport, task);
      walk = walk->next;
    }

  g_slist_free (copy);
  copy = NULL;
}

void
pgm_gl_viewport_update_drawable_projection (PgmGlViewport *glviewport)
{
  g_mutex_lock (glviewport->layer_lock);

  /* Adapt position and size of all the drawables */
  g_list_foreach (glviewport->near,
                  (GFunc) pgm_gl_drawable_update_projection, NULL);
  g_list_foreach (glviewport->middle,
                  (GFunc) pgm_gl_drawable_update_projection, NULL);
  g_list_foreach (glviewport->far,
                  (GFunc) pgm_gl_drawable_update_projection, NULL);

  g_mutex_unlock (glviewport->layer_lock);
}
