/* -*- Mode: C; c-basic-offset: 2; indent-tabs-mode: nil -*-
 *
 * Pigment unit test for pgmcanvas.c
 *
 * Copyright © 2006, 2007 Fluendo Embedded S.L.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include <gst/check/gstcheck.h>
#include <pgm/pgm.h>

/* test drawable floating reference on canvas_add/del_* */
GST_START_TEST (test_canvas_add_del_drawable)
{
  PgmError ret;
  gboolean success;
  PgmCanvas *canvas;
  PgmDrawable *drawable;

  success = pgm_init_check (NULL, NULL);
  fail_unless (success, "init_check returned FALSE");

  canvas = pgm_canvas_new ();

  /* create a new floating drawable with refcount 1 */
  drawable = pgm_image_new ();
  ASSERT_OBJECT_REFCOUNT (drawable, "drawable", 1);
  fail_unless (GST_OBJECT_IS_FLOATING (drawable),
               "drawable object is not floating at creation");

  /* ref it for ourselves */
  gst_object_ref (GST_OBJECT (drawable));
  ASSERT_OBJECT_REFCOUNT (drawable, "drawable", 2);

  /* adding it in the canvas sinks the drawable */
  ret = pgm_canvas_add (canvas, PGM_DRAWABLE_NEAR, drawable);
  fail_if (ret != PGM_ERROR_OK, "canvas_add error");
  ASSERT_OBJECT_REFCOUNT (drawable, "drawable", 2);
  fail_if (GST_OBJECT_IS_FLOATING (drawable),
           "drawable object still floating after being added to the canvas");

  /* removing it reduces the refcount */
  ret = pgm_canvas_remove (canvas, drawable);
  fail_if (ret != PGM_ERROR_OK, "canvas_del error");
  ASSERT_OBJECT_REFCOUNT (drawable, "drawable", 1);

  /* clean up our own reference */
  gst_object_unref (GST_OBJECT_CAST (drawable));
  gst_object_unref (GST_OBJECT_CAST (canvas));
  pgm_deinit ();
}
GST_END_TEST

/* test drawable floating reference on canvas unref */
GST_START_TEST (test_canvas_add_drawable_unref_object)
{
  PgmError ret;
  gboolean success;
  PgmCanvas *canvas;
  PgmDrawable *drawable;

  success = pgm_init_check (NULL, NULL);
  fail_unless (success, "init_check returned FALSE");

  canvas = pgm_canvas_new ();

  /* create a new floating drawable with refcount 1 */
  drawable = pgm_image_new ();
  ASSERT_OBJECT_REFCOUNT (drawable, "drawable", 1);
  fail_unless (GST_OBJECT_IS_FLOATING (drawable),
               "drawable object is not floating at creation");

  /* ref it for ourselves */
  gst_object_ref (GST_OBJECT (drawable));
  ASSERT_OBJECT_REFCOUNT (drawable, "drawable", 2);

  /* adding it in the canvas sinks the drawable */
  ret = pgm_canvas_add (canvas, PGM_DRAWABLE_NEAR, drawable);
  fail_if (ret != PGM_ERROR_OK, "canvas_add_drawable error");
  ASSERT_OBJECT_REFCOUNT (drawable, "drawable", 2);
  fail_if (GST_OBJECT_IS_FLOATING (drawable),
           "drawable object still floating after being added to the canvas");

  /* unreffing the canvas should clean it up */
  gst_object_unref (GST_OBJECT (canvas));
  ASSERT_OBJECT_REFCOUNT (drawable, "drawable", 1);

  /* clean up our own reference */
  gst_object_unref (GST_OBJECT_CAST (drawable));
  pgm_deinit ();
}
GST_END_TEST

/* test insertion of z-ordered drawables */
GST_START_TEST (test_canvas_z_ordered_insertion)
{
  PgmCanvas *canvas = NULL;
  PgmDrawable *img1, *img2, *img3, *img4, *img5;
  gboolean success;
  PgmDrawableLayer layer;
  gint count;
  gint order;
  PgmError ret;

  success = pgm_init_check (NULL, NULL);
  fail_unless (success, "init_check returned FALSE");

  canvas = pgm_canvas_new ();
  img1 = pgm_image_new ();
  pgm_drawable_set_position (img1, 0.0f, 0.0f, 0.0f);
  img2 = pgm_image_new ();
  pgm_drawable_set_position (img2, 0.0f, 0.0f, 1.0f);
  img3 = pgm_image_new ();
  pgm_drawable_set_position (img3, 0.0f, 0.0f, 2.0f);
  img4 = pgm_image_new ();
  pgm_drawable_set_position (img4, 0.0f, 0.0f, 3.0f);
  img5 = pgm_image_new ();
  pgm_drawable_set_position (img5, 0.0f, 0.0f, 4.0f);

  pgm_canvas_add (canvas, PGM_DRAWABLE_MIDDLE, img1);
  ret = pgm_canvas_get_layer_count (canvas, PGM_DRAWABLE_MIDDLE, &count);
  fail_if (count != 1, "bad layer count");
  ret = pgm_canvas_get_order (canvas, img1, &layer, &order);
  fail_if (ret != PGM_ERROR_OK, "canvas_get_order error");
  fail_if (layer != PGM_DRAWABLE_MIDDLE, "bad drawable layer");
  fail_if (order != 0, "bad drawable order");

  pgm_canvas_add (canvas, PGM_DRAWABLE_MIDDLE, img2);
  ret = pgm_canvas_get_layer_count (canvas, PGM_DRAWABLE_MIDDLE, &count);
  fail_if (ret != PGM_ERROR_OK, "canvas_get_layer error");
  fail_if (count != 2, "bad layer count");
  ret = pgm_canvas_get_order (canvas, img1, &layer, &order);
  fail_if (ret != PGM_ERROR_OK, "canvas_get_order error");
  fail_if (layer != PGM_DRAWABLE_MIDDLE, "bad drawable layer");
  fail_if (order != 0, "bad drawable order");
  ret = pgm_canvas_get_order (canvas, img2, &layer, &order);
  fail_if (ret != PGM_ERROR_OK, "canvas_get_order error");
  fail_if (layer != PGM_DRAWABLE_MIDDLE, "bad drawable layer");
  fail_if (order != 1, "bad drawable order");

  pgm_canvas_add (canvas, PGM_DRAWABLE_MIDDLE, img3);
  ret = pgm_canvas_get_layer_count (canvas, PGM_DRAWABLE_MIDDLE, &count);
  fail_if (ret != PGM_ERROR_OK, "canvas_get_layer error");
  fail_if (count != 3, "bad layer count");
  ret = pgm_canvas_get_order (canvas, img1, &layer, &order);
  fail_if (ret != PGM_ERROR_OK, "canvas_get_order error");
  fail_if (layer != PGM_DRAWABLE_MIDDLE, "bad drawable layer");
  fail_if (order != 0, "bad drawable order");
  ret = pgm_canvas_get_order (canvas, img2, &layer, &order);
  fail_if (ret != PGM_ERROR_OK, "canvas_get_order error");
  fail_if (layer != PGM_DRAWABLE_MIDDLE, "bad drawable layer");
  fail_if (order != 1, "bad drawable order");
  ret = pgm_canvas_get_order (canvas, img3, &layer, &order);
  fail_if (ret != PGM_ERROR_OK, "canvas_get_order error");
  fail_if (layer != PGM_DRAWABLE_MIDDLE, "bad drawable layer");
  fail_if (order != 2, "bad drawable order");

  pgm_canvas_add (canvas, PGM_DRAWABLE_MIDDLE, img4);
  ret = pgm_canvas_get_layer_count (canvas, PGM_DRAWABLE_MIDDLE, &count);
  fail_if (ret != PGM_ERROR_OK, "canvas_get_layer error");
  fail_if (count != 4, "bad layer count");
  ret = pgm_canvas_get_order (canvas, img1, &layer, &order);
  fail_if (ret != PGM_ERROR_OK, "canvas_get_order error");
  fail_if (layer != PGM_DRAWABLE_MIDDLE, "bad drawable layer");
  fail_if (order != 0, "bad drawable order");
  ret = pgm_canvas_get_order (canvas, img2, &layer, &order);
  fail_if (ret != PGM_ERROR_OK, "canvas_get_order error");
  fail_if (layer != PGM_DRAWABLE_MIDDLE, "bad drawable layer");
  fail_if (order != 1, "bad drawable order");
  ret = pgm_canvas_get_order (canvas, img3, &layer, &order);
  fail_if (ret != PGM_ERROR_OK, "canvas_get_order error");
  fail_if (layer != PGM_DRAWABLE_MIDDLE, "bad drawable layer");
  fail_if (order != 2, "bad drawable order");
  ret = pgm_canvas_get_order (canvas, img4, &layer, &order);
  fail_if (ret != PGM_ERROR_OK, "canvas_get_order error");
  fail_if (layer != PGM_DRAWABLE_MIDDLE, "bad drawable layer");
  fail_if (order != 3, "bad drawable order");

  pgm_canvas_add (canvas, PGM_DRAWABLE_MIDDLE, img5);
  ret = pgm_canvas_get_layer_count (canvas, PGM_DRAWABLE_MIDDLE, &count);
  fail_if (ret != PGM_ERROR_OK, "canvas_get_layer error");
  fail_if (count != 5, "bad layer count");
  ret = pgm_canvas_get_order (canvas, img1, &layer, &order);
  fail_if (ret != PGM_ERROR_OK, "canvas_get_order error");
  fail_if (layer != PGM_DRAWABLE_MIDDLE, "bad drawable layer");
  fail_if (order != 0, "bad drawable order");
  ret = pgm_canvas_get_order (canvas, img2, &layer, &order);
  fail_if (ret != PGM_ERROR_OK, "canvas_get_order error");
  fail_if (layer != PGM_DRAWABLE_MIDDLE, "bad drawable layer");
  fail_if (order != 1, "bad drawable order");
  ret = pgm_canvas_get_order (canvas, img3, &layer, &order);
  fail_if (ret != PGM_ERROR_OK, "canvas_get_order error");
  fail_if (layer != PGM_DRAWABLE_MIDDLE, "bad drawable layer");
  fail_if (order != 2, "bad drawable order");
  ret = pgm_canvas_get_order (canvas, img4, &layer, &order);
  fail_if (ret != PGM_ERROR_OK, "canvas_get_order error");
  fail_if (layer != PGM_DRAWABLE_MIDDLE, "bad drawable layer");
  fail_if (order != 3, "bad drawable order");
  ret = pgm_canvas_get_order (canvas, img5, &layer, &order);
  fail_if (ret != PGM_ERROR_OK, "canvas_get_order error");
  fail_if (layer != PGM_DRAWABLE_MIDDLE, "bad drawable layer");
  fail_if (order != 4, "bad drawable order");

  gst_object_unref (GST_OBJECT_CAST (canvas));
  pgm_deinit ();
}
GST_END_TEST

/* test insertion of z-misordered drawables */
GST_START_TEST (test_canvas_z_misordered_insertion)
{
  PgmCanvas *canvas = NULL;
  PgmDrawable *img1, *img2, *img3, *img4, *img5;
  gboolean success;
  PgmDrawableLayer layer;
  gint count;
  gint order;
  PgmError ret;

  success = pgm_init_check (NULL, NULL);
  fail_unless (success, "init_check returned FALSE");

  canvas = pgm_canvas_new ();
  img1 = pgm_image_new ();
  pgm_drawable_set_position (img1, 0.0f, 0.0f, 1.0f);
  img2 = pgm_image_new ();
  pgm_drawable_set_position (img2, 0.0f, 0.0f, 3.0f);
  img3 = pgm_image_new ();
  pgm_drawable_set_position (img3, 0.0f, 0.0f, 2.0f);
  img4 = pgm_image_new ();
  pgm_drawable_set_position (img4, 0.0f, 0.0f, 0.0f);
  img5 = pgm_image_new ();
  pgm_drawable_set_position (img5, 0.0f, 0.0f, 4.0f);

  pgm_canvas_add (canvas, PGM_DRAWABLE_MIDDLE, img1);
  ret = pgm_canvas_get_layer_count (canvas, PGM_DRAWABLE_MIDDLE, &count);
  fail_if (count != 1, "bad layer count");
  ret = pgm_canvas_get_order (canvas, img1, &layer, &order);
  fail_if (ret != PGM_ERROR_OK, "canvas_get_order error");
  fail_if (layer != PGM_DRAWABLE_MIDDLE, "bad drawable layer");
  fail_if (order != 0, "bad drawable order");

  pgm_canvas_add (canvas, PGM_DRAWABLE_MIDDLE, img2);
  ret = pgm_canvas_get_layer_count (canvas, PGM_DRAWABLE_MIDDLE, &count);
  fail_if (ret != PGM_ERROR_OK, "canvas_get_layer error");
  fail_if (count != 2, "bad layer count");
  ret = pgm_canvas_get_order (canvas, img1, &layer, &order);
  fail_if (ret != PGM_ERROR_OK, "canvas_get_order error");
  fail_if (layer != PGM_DRAWABLE_MIDDLE, "bad drawable layer");
  fail_if (order != 0, "bad drawable order");
  ret = pgm_canvas_get_order (canvas, img2, &layer, &order);
  fail_if (ret != PGM_ERROR_OK, "canvas_get_order error");
  fail_if (layer != PGM_DRAWABLE_MIDDLE, "bad drawable layer");
  fail_if (order != 1, "bad drawable order");

  pgm_canvas_add (canvas, PGM_DRAWABLE_MIDDLE, img3);
  ret = pgm_canvas_get_layer_count (canvas, PGM_DRAWABLE_MIDDLE, &count);
  fail_if (ret != PGM_ERROR_OK, "canvas_get_layer error");
  fail_if (count != 3, "bad layer count");
  ret = pgm_canvas_get_order (canvas, img1, &layer, &order);
  fail_if (ret != PGM_ERROR_OK, "canvas_get_order error");
  fail_if (layer != PGM_DRAWABLE_MIDDLE, "bad drawable layer");
  fail_if (order != 0, "bad drawable order");
  ret = pgm_canvas_get_order (canvas, img2, &layer, &order);
  fail_if (ret != PGM_ERROR_OK, "canvas_get_order error");
  fail_if (layer != PGM_DRAWABLE_MIDDLE, "bad drawable layer");
  fail_if (order != 2, "bad drawable order");
  ret = pgm_canvas_get_order (canvas, img3, &layer, &order);
  fail_if (ret != PGM_ERROR_OK, "canvas_get_order error");
  fail_if (layer != PGM_DRAWABLE_MIDDLE, "bad drawable layer");
  fail_if (order != 1, "bad drawable order");

  pgm_canvas_add (canvas, PGM_DRAWABLE_MIDDLE, img4);
  ret = pgm_canvas_get_layer_count (canvas, PGM_DRAWABLE_MIDDLE, &count);
  fail_if (ret != PGM_ERROR_OK, "canvas_get_layer error");
  fail_if (count != 4, "bad layer count");
  ret = pgm_canvas_get_order (canvas, img1, &layer, &order);
  fail_if (ret != PGM_ERROR_OK, "canvas_get_order error");
  fail_if (layer != PGM_DRAWABLE_MIDDLE, "bad drawable layer");
  fail_if (order != 1, "bad drawable order");
  ret = pgm_canvas_get_order (canvas, img2, &layer, &order);
  fail_if (ret != PGM_ERROR_OK, "canvas_get_order error");
  fail_if (layer != PGM_DRAWABLE_MIDDLE, "bad drawable layer");
  fail_if (order != 3, "bad drawable order");
  ret = pgm_canvas_get_order (canvas, img3, &layer, &order);
  fail_if (ret != PGM_ERROR_OK, "canvas_get_order error");
  fail_if (layer != PGM_DRAWABLE_MIDDLE, "bad drawable layer");
  fail_if (order != 2, "bad drawable order");
  ret = pgm_canvas_get_order (canvas, img4, &layer, &order);
  fail_if (ret != PGM_ERROR_OK, "canvas_get_order error");
  fail_if (layer != PGM_DRAWABLE_MIDDLE, "bad drawable layer");
  fail_if (order != 0, "bad drawable order");

  pgm_canvas_add (canvas, PGM_DRAWABLE_MIDDLE, img5);
  ret = pgm_canvas_get_layer_count (canvas, PGM_DRAWABLE_MIDDLE, &count);
  fail_if (ret != PGM_ERROR_OK, "canvas_get_layer error");
  fail_if (count != 5, "bad layer count");
  ret = pgm_canvas_get_order (canvas, img1, &layer, &order);
  fail_if (ret != PGM_ERROR_OK, "canvas_get_order error");
  fail_if (layer != PGM_DRAWABLE_MIDDLE, "bad drawable layer");
  fail_if (order != 1, "bad drawable order");
  ret = pgm_canvas_get_order (canvas, img2, &layer, &order);
  fail_if (ret != PGM_ERROR_OK, "canvas_get_order error");
  fail_if (layer != PGM_DRAWABLE_MIDDLE, "bad drawable layer");
  fail_if (order != 3, "bad drawable order");
  ret = pgm_canvas_get_order (canvas, img3, &layer, &order);
  fail_if (ret != PGM_ERROR_OK, "canvas_get_order error");
  fail_if (layer != PGM_DRAWABLE_MIDDLE, "bad drawable layer");
  fail_if (order != 2, "bad drawable order");
  ret = pgm_canvas_get_order (canvas, img4, &layer, &order);
  fail_if (ret != PGM_ERROR_OK, "canvas_get_order error");
  fail_if (layer != PGM_DRAWABLE_MIDDLE, "bad drawable layer");
  fail_if (order != 0, "bad drawable order");
  ret = pgm_canvas_get_order (canvas, img5, &layer, &order);
  fail_if (ret != PGM_ERROR_OK, "canvas_get_order error");
  fail_if (layer != PGM_DRAWABLE_MIDDLE, "bad drawable layer");
  fail_if (order != 4, "bad drawable order");

  gst_object_unref (GST_OBJECT_CAST (canvas));
  pgm_deinit ();
}
GST_END_TEST

/* test insertion of a drawable in a layer having drawables with the
 * same z-position */
GST_START_TEST (test_canvas_add_with_same_z_position)
{
  PgmCanvas *canvas = NULL;
  PgmDrawable *img1, *img2, *img3, *img4, *img5;
  gboolean success;
  PgmDrawableLayer layer;
  gint order;

  success = pgm_init_check (NULL, NULL);
  fail_unless (success, "init_check returned FALSE");

  canvas = pgm_canvas_new ();
  img1 = pgm_image_new ();
  pgm_drawable_set_position (img1, 0.0f, 0.0f, 0.0f);
  img2 = pgm_image_new ();
  pgm_drawable_set_position (img2, 0.0f, 0.0f, 1.0f);
  img3 = pgm_image_new ();
  pgm_drawable_set_position (img3, 0.0f, 0.0f, 1.0f);
  img4 = pgm_image_new ();
  pgm_drawable_set_position (img4, 0.0f, 0.0f, 1.0f);
  img5 = pgm_image_new ();
  pgm_drawable_set_position (img5, 0.0f, 0.0f, 2.0f);

  pgm_canvas_add (canvas, PGM_DRAWABLE_MIDDLE, img1);
  pgm_canvas_add (canvas, PGM_DRAWABLE_MIDDLE, img2);
  pgm_canvas_add (canvas, PGM_DRAWABLE_MIDDLE, img3);
  pgm_canvas_add (canvas, PGM_DRAWABLE_MIDDLE, img5);

  /* The order should be 3 after the other drawables with a z-position
   * of 1.Of */
  pgm_canvas_add (canvas, PGM_DRAWABLE_MIDDLE, img4);
  pgm_canvas_get_order (canvas, img4, &layer, &order);
  fail_if (order != 3, "bad drawable order");

  gst_object_unref (GST_OBJECT_CAST (canvas));
  pgm_deinit ();
}
GST_END_TEST

/* test canvas reordering */
GST_START_TEST (test_canvas_reordering)
{
  PgmCanvas *canvas = NULL;
  PgmDrawable *img1, *img2, *img3, *img4, *img5, *img6, *img7, *img8;
  gboolean success;
  PgmError ret;
  PgmDrawableLayer layer;
  gint order;

  success = pgm_init_check (NULL, NULL);
  fail_unless (success, "init_check returned FALSE");

  canvas = pgm_canvas_new ();
  img1 = pgm_image_new ();
  pgm_drawable_set_position (img1, 0.0f, 0.0f, 0.0f);
  img2 = pgm_image_new ();
  pgm_drawable_set_position (img2, 0.0f, 0.0f, 1.0f);
  img3 = pgm_image_new ();
  pgm_drawable_set_position (img3, 0.0f, 0.0f, 2.0f);
  img4 = pgm_image_new ();
  pgm_drawable_set_position (img4, 0.0f, 0.0f, 2.0f);
  img5 = pgm_image_new ();
  pgm_drawable_set_position (img5, 0.0f, 0.0f, 2.0f);
  img6 = pgm_image_new ();
  pgm_drawable_set_position (img6, 0.0f, 0.0f, 3.0f);
  img7 = pgm_image_new ();
  pgm_drawable_set_position (img7, 0.0f, 0.0f, 4.0f);

  pgm_canvas_add (canvas, PGM_DRAWABLE_MIDDLE, img1);
  pgm_canvas_add (canvas, PGM_DRAWABLE_MIDDLE, img2);
  pgm_canvas_add (canvas, PGM_DRAWABLE_MIDDLE, img3);
  pgm_canvas_add (canvas, PGM_DRAWABLE_MIDDLE, img4);
  pgm_canvas_add (canvas, PGM_DRAWABLE_MIDDLE, img5);
  pgm_canvas_add (canvas, PGM_DRAWABLE_MIDDLE, img6);
  pgm_canvas_add (canvas, PGM_DRAWABLE_MIDDLE, img7);

  /* Test different z */
  pgm_canvas_set_order (canvas, img6, 1);
  pgm_canvas_get_order (canvas, img6, &layer, &order);
  fail_if (order != 5, "drawable order should have not changed");

  /* Test same position */
  pgm_canvas_set_order (canvas, img6, 5);
  pgm_canvas_get_order (canvas, img6, &layer, &order);
  fail_if (order != 5, "drawable order should have not changed");

  /* Test out of bound order */
  ret = pgm_canvas_set_order (canvas, img6, 10);
  fail_if (ret != PGM_ERROR_X, "setting an outbound order should return an "
           "error");
  pgm_canvas_get_order (canvas, img6, &layer, &order);
  fail_if (order != 5, "drawable order should have not changed");

  /* Test unbinded drawable */
  img8 = pgm_image_new ();
  ret = pgm_canvas_set_order (canvas, img8, 2);
  fail_if (ret != PGM_ERROR_X, "setting order with an unbinded drawable "
           "should return an error");
  gst_object_unref (img8);

  /* Put the last drawable with z=2 at the beginning */
  pgm_canvas_set_order (canvas, img5, 2);
  pgm_canvas_get_order (canvas, img5, &layer, &order);
  fail_if (order != 2, "drawable order should be 2");
  pgm_canvas_get_order (canvas, img3, &layer, &order);
  fail_if (order != 3, "drawable order should be 3");
  pgm_canvas_get_order (canvas, img4, &layer, &order);
  fail_if (order != 4, "drawable order should be 4");

  /* Put the first drawable with z=2 at the end */
  pgm_canvas_set_order (canvas, img5, 4);
  pgm_canvas_get_order (canvas, img5, &layer, &order);
  fail_if (order != 4, "drawable order should be 4");
  pgm_canvas_get_order (canvas, img3, &layer, &order);
  fail_if (order != 2, "drawable order should be 2");
  pgm_canvas_get_order (canvas, img4, &layer, &order);
  fail_if (order != 3, "drawable order should be 3");

  /* Remove drawables with z-position different than 2, to test reordering
   * at beginning and end of the list */
  pgm_canvas_remove (canvas, img1);
  pgm_canvas_remove (canvas, img2);
  pgm_canvas_remove (canvas, img6);
  pgm_canvas_remove (canvas, img7);

  /* Test correct ordering after remove operations */
  pgm_canvas_get_order (canvas, img3, &layer, &order);
  fail_if (order != 0, "drawable order should be 2");
  pgm_canvas_get_order (canvas, img4, &layer, &order);
  fail_if (order != 1, "drawable order should be 3");
  pgm_canvas_get_order (canvas, img5, &layer, &order);
  fail_if (order != 2, "drawable order should be 4");

  /* Put the last drawable at the beginning */
  pgm_canvas_set_order (canvas, img5, 0);
  pgm_canvas_get_order (canvas, img5, &layer, &order);
  fail_if (order != 0, "drawable order should be 0");
  pgm_canvas_get_order (canvas, img3, &layer, &order);
  fail_if (order != 1, "drawable order should be 1");
  pgm_canvas_get_order (canvas, img4, &layer, &order);
  fail_if (order != 2, "drawable order should be 2");

  /* Put the first drawable at the end */
  pgm_canvas_set_order (canvas, img5, 2);
  pgm_canvas_get_order (canvas, img5, &layer, &order);
  fail_if (order != 2, "drawable order should be 2");
  pgm_canvas_get_order (canvas, img3, &layer, &order);
  fail_if (order != 0, "drawable order should be 0");
  pgm_canvas_get_order (canvas, img4, &layer, &order);
  fail_if (order != 1, "drawable order should be 1");

  gst_object_unref (GST_OBJECT_CAST (canvas));
  pgm_deinit ();
}
GST_END_TEST

/* test canvas reordering */
GST_START_TEST (test_canvas_z_reordering)
{
  PgmCanvas *canvas = NULL;
  PgmDrawable *img1, *img2, *img3, *img4, *img5;
  gboolean success;
  PgmError ret;
  PgmDrawableLayer layer;
  gint order;

  success = pgm_init_check (NULL, NULL);
  fail_unless (success, "init_check returned FALSE");

  canvas = pgm_canvas_new ();
  img1 = pgm_image_new ();
  pgm_drawable_set_position (img1, 0.0f, 0.0f, 0.0f);
  img2 = pgm_image_new ();
  pgm_drawable_set_position (img2, 0.0f, 0.0f, 2.0f);
  img3 = pgm_image_new ();
  pgm_drawable_set_position (img3, 0.0f, 0.0f, 2.0f);
  img4 = pgm_image_new ();
  pgm_drawable_set_position (img4, 0.0f, 0.0f, 4.0f);
  img5 = pgm_image_new ();
  pgm_drawable_set_position (img5, 0.0f, 0.0f, 6.0f);

  pgm_canvas_add (canvas, PGM_DRAWABLE_MIDDLE, img1);
  pgm_canvas_add (canvas, PGM_DRAWABLE_MIDDLE, img2);
  pgm_canvas_add (canvas, PGM_DRAWABLE_MIDDLE, img3);
  pgm_canvas_add (canvas, PGM_DRAWABLE_MIDDLE, img4);
  pgm_canvas_add (canvas, PGM_DRAWABLE_MIDDLE, img5);

  /* Change the z-position to put img5 at order 0 */
  pgm_drawable_set_position (img5, 0.0f, 0.0f, -1.0f);
  pgm_canvas_get_order (canvas, img5, &layer, &order);
  fail_if (order != 0, "drawable order should be 0");
  pgm_canvas_get_order (canvas, img1, &layer, &order);
  fail_if (order != 1, "drawable order should be 1");
  pgm_canvas_get_order (canvas, img2, &layer, &order);
  fail_if (order != 2, "drawable order should be 2");
  pgm_canvas_get_order (canvas, img3, &layer, &order);
  fail_if (order != 3, "drawable order should be 3");
  pgm_canvas_get_order (canvas, img4, &layer, &order);
  fail_if (order != 4, "drawable order should be 4");

  /* Change the z-position to put img5 at order 4 */
  pgm_drawable_set_position (img5, 0.0f, 0.0f, 6.0f);
  pgm_canvas_get_order (canvas, img1, &layer, &order);
  fail_if (order != 0, "drawable order should be 0");
  pgm_canvas_get_order (canvas, img2, &layer, &order);
  fail_if (order != 1, "drawable order should be 1");
  pgm_canvas_get_order (canvas, img3, &layer, &order);
  fail_if (order != 2, "drawable order should be 2");
  pgm_canvas_get_order (canvas, img4, &layer, &order);
  fail_if (order != 3, "drawable order should be 3");
  pgm_canvas_get_order (canvas, img5, &layer, &order);
  fail_if (order != 4, "drawable order should be 4");

  /* Change the z-position to put img5 at order 3 */
  pgm_drawable_set_position (img5, 0.0f, 0.0f, 2.0f);
  pgm_canvas_get_order (canvas, img1, &layer, &order);
  fail_if (order != 0, "drawable order should be 0");
  pgm_canvas_get_order (canvas, img2, &layer, &order);
  fail_if (order != 1, "drawable order should be 1");
  pgm_canvas_get_order (canvas, img3, &layer, &order);
  fail_if (order != 2, "drawable order should be 2");
  pgm_canvas_get_order (canvas, img5, &layer, &order);
  fail_if (order != 3, "drawable order should be 3");
  pgm_canvas_get_order (canvas, img4, &layer, &order);
  fail_if (order != 4, "drawable order should be 4");

  gst_object_unref (GST_OBJECT_CAST (canvas));
  pgm_deinit ();
}
GST_END_TEST

/* test default values */
GST_START_TEST (test_canvas_default_values)
{
  PgmCanvas *canvas;
  PgmError ret;
  gboolean success;
  gfloat width, height;

  success = pgm_init_check (NULL, NULL);
  fail_unless (success, "init_check returned FALSE");

  canvas = pgm_canvas_new ();

  /* size */
  ret = pgm_canvas_get_size (canvas, &width, &height);
  fail_if (ret != PGM_ERROR_OK, "canvas_get_size error");
  fail_if (width != 4.0f || height != 3.0f, "bad default size");

  gst_object_unref (GST_OBJECT_CAST (canvas));
  pgm_deinit ();
}
GST_END_TEST;

/* test set/get correctness */
GST_START_TEST (test_canvas_set_get_correctness)
{
  PgmCanvas *canvas;
  PgmError ret;
  gboolean success;
  gfloat width, height;

  success = pgm_init_check (NULL, NULL);
  fail_unless (success, "init_check returned FALSE");

  canvas = pgm_canvas_new ();

  /* size */
  ret = pgm_canvas_set_size (canvas, 16.0f, 9.0f);
  fail_if (ret != PGM_ERROR_OK, "canvas_set_size error");
  ret = pgm_canvas_get_size (canvas, &width, &height);
  fail_if (ret != PGM_ERROR_OK, "canvas_get_size error");
  fail_if (width != 16.0f || height != 9.0f, "size not set");

  gst_object_unref (GST_OBJECT_CAST (canvas));
  pgm_deinit ();
}
GST_END_TEST

Suite *
pgm_canvas_suite (void)
{
  Suite *s = suite_create ("PgmCanvas");
  TCase *tc_chain = tcase_create ("pgmcanvas tests");

  suite_add_tcase (s, tc_chain);
  tcase_add_test (tc_chain, test_canvas_add_del_drawable);
  tcase_add_test (tc_chain, test_canvas_add_drawable_unref_object);
  tcase_add_test (tc_chain, test_canvas_z_ordered_insertion);
  tcase_add_test (tc_chain, test_canvas_z_misordered_insertion);
  tcase_add_test (tc_chain, test_canvas_add_with_same_z_position);
  tcase_add_test (tc_chain, test_canvas_reordering);
  tcase_add_test (tc_chain, test_canvas_z_reordering);
  tcase_add_test (tc_chain, test_canvas_default_values);
  tcase_add_test (tc_chain, test_canvas_set_get_correctness);

  return s;
}

GST_CHECK_MAIN (pgm_canvas);
