/* -*- Mode: C; c-basic-offset: 2; indent-tabs-mode: nil -*-
 *
 * Pigment unit test for pgmlinalg.c
 *
 * Copyright © 2007 Fluendo Embedded S.L.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include <gst/check/gstcheck.h>
#include <pgm/pgm.h>

/* test macros */
GST_START_TEST (test_linalg_macros)
{
  PgmVec3 vec3, _vec3;
  PgmVec4 vec4, _vec4;
  PgmMat4x4 mat4, _mat4;

  /* Vec3 initialization */
  PGM_VEC3_INIT (vec3, 1.0f, 2.0f, 3.0f);
  fail_if (vec3.v[0] != 1.0f || vec3.v[1] != 2.0f || vec3.v[2] != 3.0f,
           "PGM_VEC3_INIT error");

  /* Vec3 copy */
  PGM_VEC3_COPY (&_vec3, &vec3);
  fail_if (_vec3.v[0] != 1.0f || _vec3.v[1] != 2.0f || _vec3.v[2] != 3.0f,
           "PGM_VEC3_COPY error");

  /* Vec4 initialization */
  PGM_VEC4_INIT (vec4, 1.0f, 2.0f, 3.0f, 4.0f);
  fail_if (vec4.v[0] != 1.0f || vec4.v[1] != 2.0f || vec4.v[2] != 3.0f
           || vec4.v[3] != 4.0f, "PGM_VEC4_INIT error");

  /* Vec4 copy */
  PGM_VEC4_COPY (&_vec4, &vec4);
  fail_if (_vec4.v[0] != 1.0f || _vec4.v[1] != 2.0f || _vec4.v[2] != 3.0f,
           "PGM_VEC4_COPY error");

  /* Mat4x4 initialization */
  PGM_MAT4X4_INIT (mat4, 1.0f, 2.0f, 3.0f, 4.0f, 5.0f, 6.0f, 7.0f, 8.0f,
                   9.0f, 10.0f, 11.0f, 12.0f, 13.0f, 14.0f, 15.0f, 16.0f);
  fail_if (mat4.m[0][0] != 1.0f || mat4.m[0][1] != 2.0f
           || mat4.m[0][2] != 3.0f || mat4.m[0][3] != 4.0f
           || mat4.m[1][0] != 5.0f || mat4.m[1][1] != 6.0f
           || mat4.m[1][2] != 7.0f || mat4.m[1][3] != 8.0f
           || mat4.m[2][0] != 9.0f || mat4.m[2][1] != 10.0f
           || mat4.m[2][2] != 11.0f || mat4.m[2][3] != 12.0f
           || mat4.m[3][0] != 13.0f || mat4.m[3][1] != 14.0f
           || mat4.m[3][2] != 15.0f || mat4.m[3][3] != 16.0f,
           "PGM_MAT4X4_INIT error");

  /* Mat4x4 identity initialization */
  PGM_MAT4X4_IDENTITY (_mat4);
  fail_if (_mat4.m[0][0] != 1.0f || _mat4.m[0][1] != 0.0f
           || _mat4.m[0][2] != 0.0f || _mat4.m[0][3] != 0.0f
           || _mat4.m[1][0] != 0.0f || _mat4.m[1][1] != 1.0f
           || _mat4.m[1][2] != 0.0f || _mat4.m[1][3] != 0.0f
           || _mat4.m[2][0] != 0.0f || _mat4.m[2][1] != 0.0f
           || _mat4.m[2][2] != 1.0f || _mat4.m[2][3] != 0.0f
           || _mat4.m[3][0] != 0.0f || _mat4.m[3][1] != 0.0f
           || _mat4.m[3][2] != 0.0f || _mat4.m[3][3] != 1.0f,
           "PGM_MAT4X4_IDENTITY error");

  /* Mat4x4 copy */
  PGM_MAT4X4_COPY (&_mat4, &mat4);
  fail_if (_mat4.m[0][0] != 1.0f || _mat4.m[0][1] != 2.0f
           || _mat4.m[0][2] != 3.0f || _mat4.m[0][3] != 4.0f
           || _mat4.m[1][0] != 5.0f || _mat4.m[1][1] != 6.0f
           || _mat4.m[1][2] != 7.0f || _mat4.m[1][3] != 8.0f
           || _mat4.m[2][0] != 9.0f || _mat4.m[2][1] != 10.0f
           || _mat4.m[2][2] != 11.0f || _mat4.m[2][3] != 12.0f
           || _mat4.m[3][0] != 13.0f || _mat4.m[3][1] != 14.0f
           || _mat4.m[3][2] != 15.0f || _mat4.m[3][3] != 16.0f,
           "PGM_MAT4X4_COPY error");
}
GST_END_TEST;

/* test functions
 *
 * FIXME: Tests are needed for these functions:
 * pgm_vec4_mult_mat4x4_vec4, pgm_mat4x4_rotate_x,
 * pgm_mat4x4_rotate_y, pgm_mat4x4_rotate_z, pgm_mat4x4_rotate,
 * pgm_mat4x4_translate, pgm_mat4x4_scale,
 * pgm_mat4x4_scale_along_axis, pgm_mat4x4_mult_mat4x4_mat4x4
 */
GST_START_TEST (test_linalg_functions)
{
  PgmVec3 vec3;
  PgmMat4x4 mat4, _mat4, m;
  gfloat length;
  PgmVec3 l1, l2;
  PgmVec3 p, pu, pv;
  PgmVec3 v;
  gboolean belongs;

  PGM_VEC3_INIT (vec3, 13.0f, 34.0f, 25.0f);

  /* 3 components vector length */
  length = pgm_vec3_length (&vec3);
  fail_if (length <= 44.1588f || length >= 44.1589f, "pgm_vec3_length error");

  /* 3 components vector normalization */
  pgm_vec3_normalize (&vec3);
  fail_if (vec3.v[0] <= 0.2943f || vec3.v[0] >= 0.2944f,
           vec3.v[1] <= 0.7699f || vec3.v[1] >= 0.77f,
           vec3.v[2] <= 0.5661f || vec3.v[2] >= 0.5662f,
           "pgm_vec3_normalize error");

  /* intersection line-plane */
  PGM_VEC3_INIT (l1, 0.0f, 0.0f, 0.0f);
  PGM_VEC3_INIT (l2, 0.0f, 0.0f, 1.0f);
  PGM_VEC3_INIT (p, -0.5f, -0.5f, 1.0f);
  PGM_VEC3_INIT (pu, 1.0f, 0.0f, 0.0f);
  PGM_VEC3_INIT (pv, 0.0f, 1.0f, 0.0f);
  pgm_intersection_line_plane (&v, &l1, &l2, &p, &pu, &pv);
  fail_if (v.v[0] != 0.0f || v.v[1] != 0.0f || v.v[2] != 1.0f,
           "pgm_intersection_line_plane error");

  PGM_VEC3_INIT (l1, 1.0f, 1.0f, 0.5f);
  PGM_VEC3_INIT (l2, 1.0f, 0.0f, 0.5f);
  PGM_VEC3_INIT (p,  0.0f, 0.0f, 0.0f);
  PGM_VEC3_INIT (pu, 2.0f, 0.0f, 0.0f);
  PGM_VEC3_INIT (pv, 0.0f, 0.0f, 1.0f);
  pgm_intersection_line_plane (&v, &l1, &l2, &p, &pu, &pv);
  fail_if (v.v[0] != 1.0f || v.v[1] != 0.0f || v.v[2] != 0.5f,
           "pgm_intersection_line_plane error");

  PGM_VEC3_INIT (l1, 1.0f, 1.0f, 0.5f);
  PGM_VEC3_INIT (l2, 1.0f, 0.0f, 0.5f);
  PGM_VEC3_INIT (p,  0.0f, 0.0f, 0.0f);
  PGM_VEC3_INIT (pu, 20.0f, 0.0f, 0.0f);
  PGM_VEC3_INIT (pv, 0.0f, 0.0f, 5.0f);
  pgm_intersection_line_plane (&v, &l1, &l2, &p, &pu, &pv);
  fail_if (v.v[0] != 1.0f || v.v[1] != 0.0f || v.v[2] != 0.5f,
           "pgm_intersection_line_plane error");

  /* point in a rectangle */
  PGM_VEC3_INIT (p, -0.5f, -0.5f, 0.0f);
  PGM_VEC3_INIT (pu, 1.0f, 0.0f, 0.0f);
  PGM_VEC3_INIT (pv, 0.0f, 1.0f, 0.0f);

  PGM_VEC3_INIT (vec3, 0.0f, 0.0f, 0.0f);
  belongs = pgm_vec3_belongs_rectangle (&vec3, &p, &pu, &pv);
  fail_if (belongs != TRUE, "pgm_vec3_belongs_rectangle error");

  PGM_VEC3_INIT (vec3, 1.0f, 0.0f, 0.0f);
  belongs = pgm_vec3_belongs_rectangle (&vec3, &p, &pu, &pv);
  fail_if (belongs != FALSE, "pgm_vec3_belongs_rectangle error");
  
  PGM_VEC3_INIT (vec3, 0.5f, 0.0f, 0.0f);
  belongs = pgm_vec3_belongs_rectangle (&vec3, &p, &pu, &pv);
  fail_if (belongs != TRUE, "pgm_vec3_belongs_rectangle error");

  PGM_VEC3_INIT (vec3, 0.5f, 0.5f, 0.0f);
  belongs = pgm_vec3_belongs_rectangle (&vec3, &p, &pu, &pv);
  fail_if (belongs != TRUE, "pgm_vec3_belongs_rectangle error");


  PGM_VEC3_INIT (p, 1.0f, -1.0f, 0.0f);
  PGM_VEC3_INIT (pu, 0.0f, 0.0f, 2.0f);
  PGM_VEC3_INIT (pv, 0.0f, 1.0f, 0.0f);

  PGM_VEC3_INIT (vec3, 1.0f, 0.0f, 0.0f);
  belongs = pgm_vec3_belongs_rectangle (&vec3, &p, &pu, &pv);
  fail_if (belongs != TRUE, "pgm_vec3_belongs_rectangle error");

  PGM_VEC3_INIT (vec3, 1.0f, -0.5f, 1.9f);
  belongs = pgm_vec3_belongs_rectangle (&vec3, &p, &pu, &pv);
  fail_if (belongs != TRUE, "pgm_vec3_belongs_rectangle error");

  PGM_VEC3_INIT (vec3, 1.0f, -0.5f, 2.1f);
  belongs = pgm_vec3_belongs_rectangle (&vec3, &p, &pu, &pv);
  fail_if (belongs != FALSE, "pgm_vec3_belongs_rectangle error");

  /*
  PGM_VEC3_INIT (vec3, 0.0f, 0.0f, 0.0f);
  belongs = pgm_vec3_belongs_rectangle (&vec3, &p, &pu, &pv);
  fail_if (belongs != FALSE, "pgm_vec3_belongs_rectangle error");
  */

  PGM_VEC3_INIT (vec3, 1.0f, -0.5f, 1.0f);
  belongs = pgm_vec3_belongs_rectangle (&vec3, &p, &pu, &pv);
  fail_if (belongs != TRUE, "pgm_vec3_belongs_rectangle error");
}
GST_END_TEST;

Suite *
pgm_linalg_suite (void)
{
  Suite *s = suite_create ("PgmLinAlg");
  TCase *tc_chain = tcase_create ("pgmlinalg tests");

  suite_add_tcase (s, tc_chain);
  tcase_add_test (tc_chain, test_linalg_macros);
  tcase_add_test (tc_chain, test_linalg_functions);

  return s;
}

GST_CHECK_MAIN (pgm_linalg);
