/* -*- Mode: C; c-basic-offset: 2; indent-tabs-mode: nil -*-
 *
 * Pigment unit test for pgmviewportfactory.c
 *
 * Copyright © 2006, 2007 Fluendo Embedded S.L.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include <gst/check/gstcheck.h>
#include <pgm/pgm.h>

/* test plugin loading */
GST_START_TEST (test_factory_load)
{
  PgmViewportFactory *factory = NULL;
  gboolean success;

  success = pgm_init_check (NULL, NULL);
  fail_unless (success, "init_check returned FALSE");

  /* Try to load an unavailable plugin */
  factory = pgm_viewport_factory_new ("blah blah");
  fail_if (factory != NULL, "viewport_factory_new returned TRUE with an "
           "unavailable plugin");

  /* Try to load an available plugin */
  factory = pgm_viewport_factory_new ("boilerplate");
  fail_unless (factory != NULL, "viewport_factory_new returned FALSE with the "
               "boilerplate plugin");
  gst_object_unref (factory);

  pgm_deinit ();
}
GST_END_TEST;

/* test available plugin */
GST_START_TEST (test_factory_desc)
{
  PgmViewportFactory *factory = NULL;
  gchar *string = NULL;
  gboolean success;
  PgmError ret;

  success = pgm_init_check (NULL, NULL);
  fail_unless (success, "init_check returned FALSE");

  factory = pgm_viewport_factory_new ("boilerplate");
  fail_unless (factory != NULL, "viewport_factory_new returned FALSE with the "
               "boilerplate plugin");

  /* Check the description */
  ret = pgm_viewport_factory_get_description (factory, &string);
  fail_if (ret != PGM_ERROR_OK, "viewport_factory_get_description error");
  success = g_ascii_strcasecmp ("A boilerplate viewport used for tests",
                                string);
  fail_unless (!success, "boilerplate plugin returns a bad description");
  g_free (string);

  /* Check the license */
  ret = pgm_viewport_factory_get_license (factory, &string);
  fail_if (ret != PGM_ERROR_OK, "viewport_factory_get_license error");
  success = g_ascii_strcasecmp ("LGPL", string);
  fail_unless (!success, "boilerplate plugin returns a bad license");
  g_free (string);

  /* Check the origin */
  ret = pgm_viewport_factory_get_origin (factory, &string);
  fail_if (ret != PGM_ERROR_OK, "viewport_factory_get_origin error");
  success = g_ascii_strcasecmp ("https://code.fluendo.com/pigment/trac",
                                string);
  fail_unless (!success, "boilerplate plugin returns a bad origin");
  g_free (string);

  /* Check the author */
  ret = pgm_viewport_factory_get_author (factory, &string);
  fail_if (ret != PGM_ERROR_OK, "viewport_factory_get_author error");
  success = g_ascii_strcasecmp ("Loïc Molinari <loic@fluendo.com>",
                                string);
  fail_unless (!success, "boilerplate plugin returns a bad author");
  g_free (string);

  gst_object_unref (factory);
  pgm_deinit ();
}
GST_END_TEST;

/* test viewport creation with make */
GST_START_TEST (test_factory_make)
{
  PgmViewport *viewport = NULL;
  gboolean success;
  PgmError ret;

  success = pgm_init_check (NULL, NULL);
  fail_unless (success, "init_check returned FALSE");

  /* Create a viewport with factory_make */
  ret = pgm_viewport_factory_make ("boilerplate", &viewport);
  fail_if (ret != PGM_ERROR_OK, "viewport_factory_create error");
  fail_unless (PGM_IS_VIEWPORT (viewport), "viewport_factory_create returns "
               "a bad viewport");
  ASSERT_OBJECT_REFCOUNT (viewport, "viewport", 1);

  gst_object_unref (viewport);
  pgm_deinit ();
}
GST_END_TEST;


/* test 2 viewport creations with new/create */
GST_START_TEST (test_factory_create)
{
  PgmViewportFactory *factory = NULL;
  PgmViewport *vp1 = NULL, *vp2 = NULL;
  gboolean success;
  PgmError ret;

  success = pgm_init_check (NULL, NULL);
  fail_unless (success, "init_check returned FALSE");

  /* Create the factory */
  factory = pgm_viewport_factory_new ("boilerplate");
  fail_unless (factory != NULL, "viewport_factory_new returned FALSE with the "
               "boilerplate plugin");
  ASSERT_OBJECT_REFCOUNT (factory, "factory", 1);

  /* Instantiate a first viewport, increase the factory ref count */
  ret = pgm_viewport_factory_create (factory, &vp1);
  fail_if (ret != PGM_ERROR_OK, "viewport_factory_create error");
  fail_unless (PGM_IS_VIEWPORT (vp1), "viewport_factory_create returns a bad "
               "viewport");
  ASSERT_OBJECT_REFCOUNT (vp1, "vp1", 1);
  ASSERT_OBJECT_REFCOUNT (factory, "factory", 2);

  /* Instantiate a second viewport, increase the factory ref count */
  ret = pgm_viewport_factory_create (factory, &vp2);
  fail_if (ret != PGM_ERROR_OK, "viewport_factory_create error");
  fail_unless (PGM_IS_VIEWPORT (vp2), "viewport_factory_create returns a bad "
               "viewport");
  ASSERT_OBJECT_REFCOUNT (vp2, "vp1", 1);
  ASSERT_OBJECT_REFCOUNT (factory, "factory", 3);

  /* Unref the viewports */
  gst_object_unref (vp1);
  ASSERT_OBJECT_REFCOUNT (factory, "factory", 2);
  gst_object_unref (vp2);
  ASSERT_OBJECT_REFCOUNT (factory, "factory", 1);

  gst_object_unref (factory);
  pgm_deinit ();
}
GST_END_TEST;

Suite *
pgm_viewport_factory_suite (void)
{
  Suite *s = suite_create ("PgmViewportFactory");
  TCase *tc_chain = tcase_create ("pgmviewportfactory tests");

  suite_add_tcase (s, tc_chain);
  tcase_add_test (tc_chain, test_factory_load);
  tcase_add_test (tc_chain, test_factory_desc);
  /* FIXME: disabled, see #105 */
  /* tcase_add_test (tc_chain, test_factory_make); */
  tcase_add_test (tc_chain, test_factory_create);

  return s;
}

GST_CHECK_MAIN (pgm_viewport_factory);
