/***************************************************************************
 *   Copyright (C) 2005-2008 Nicolas Hadacek <hadacek@kde.org>             *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 ***************************************************************************/
#include "process.h"

#include "unistd.h"
#if defined(Q_OS_UNIX)
#  include <signal.h>
#endif

#if defined(NO_KDE)
#  if QT_VERSION<0x040000
#    include <qprocess.h>
#  else
#    include <Qt3Support/Q3Process>
#  endif
#else
#  include <process_kde.h>
#endif

#include "purl.h"
#include "common/common/synchronous.h"

//----------------------------------------------------------------------------
Process::State Process::runSynchronously(Base &process, RunActions actions, uint timeout)
{
  Synchronous sync(timeout);
  QObject::connect(&process, SIGNAL(done(int)), &sync, SLOT(done()));
  QObject::connect(&process, SIGNAL(requestSynchronousStop()), &sync, SLOT(done()));
  if ( (actions & Start) && !process.start(0) ) return process.state();
  if ( !sync.enterLoop() ) process.timeoutSlot();
  return process.state();
}

//----------------------------------------------------------------------------
Process::Base::Base(QObject *parent, const char *name)
  : QObject(parent, name), _useShell(false), _state(Stopped)
{
  _process = new ProcessIO(this);
  connect(_process, SIGNAL(processExited()), SLOT(exitedSlot()));
  connect(_process, SIGNAL(readyReadStdout()), SLOT(readyReadStdoutSlot()));
  connect(_process, SIGNAL(readyReadStderr()), SLOT(readyReadStderrSlot()));
  _timer = new QTimer(this);
  connect(_timer, SIGNAL(timeout()), SLOT(timeoutSlot()));
}

Process::Base::~Base()
{
  _process->kill();
}

void Process::Base::setup(const QString &executable, const QStringList &options, bool withWine)
{
  _arguments.clear();
  _environment.clear();
  if (withWine) {
    _environment += "WINEDEBUG=-all";
    _arguments += "wine";
  }
  _arguments += executable;
  _arguments += options;
}

void Process::Base::clearOutputs()
{
  FOR_EACH(OutputType, type) _outputs[type] = QString::null;
}

bool Process::Base::start(uint timeout)
{
  _state = Stopped;
  _timer->stop();
  clearOutputs();
  QStringList args;
  if (_useShell) {
    args += "/bin/sh";
    args += "-c";
    args += _arguments.join(" ");
  } else args = _arguments;
  _process->setArguments(args);
  QStringList env;
  if ( !_environment.isEmpty() ) {
    for (uint i=0; environ[i]; i++) env += environ[i];
    env += _environment;
  }
  if ( !_process->start(env.isEmpty() ? 0 : &env) ) {
    _state = StartFailed;
    return false;
  }
  _state = Running;
  if (timeout) _timer->start(timeout);
  return true;
}

void Process::Base::timeoutSlot()
{
  _state = Timedout;
  _process->kill();
  emit timeout();
}

int Process::Base::exitCode() const
{
  return _process->exitStatus();
}

void Process::Base::exitedSlot()
{
  _state = Exited;
  _timer->stop();
  FOR_EACH(OutputType, type) readyRead(type);
  emit done(exitCode());
}

bool Process::Base::isRunning() const
{
  return _process->isRunning();
}

void Process::Base::writeToStdin(const QString &s)
{
  const char *cs = s.latin1();
  QByteArray a;
  a.duplicate(cs, strlen(cs));
  _process->writeToStdin(a);
}

void Process::Base::kill()
{
  _process->kill();
}

bool Process::Base::signal(int n)
{
#if defined(Q_OS_UNIX)
  return ( ::kill(_process->processIdentifier(), n)!=-1 );
#elif defined(Q_OS_WIN)
  // #### impossible to do ??
  return false;
#endif
}

void Process::Base::setWorkingDirectory(const PURL::Directory &dir)
{
  _process->setWorkingDirectory(dir.path());
}

void Process::Base::setUseShell(bool useShell)
{
  _useShell = useShell;
}

bool Process::Base::isFilteredLine(const QString &line)
{
  // "wine" returns all those "libGL warning" that mess up the output...
  return line.startsWith("libGL warning");
}

void Process::Base::readyRead(OutputType type)
{
  QString s;
  for (;;) {
    QByteArray a;
    switch (type.type()) {
      case OutputType::Stdout: a = _process->readStdout(); break;
      case OutputType::Stderr: a = _process->readStderr(); break;
      case OutputType::Nb_Types: Q_ASSERT(false); break;
    }
    if (a.count() == 0) break;
    s += QString::fromLatin1(a.data(), a.count());
  }
  if (s.length()) {
    emit received(type, s);
    emit dataReceived(type);
  }
}

//----------------------------------------------------------------------------
QString Process::StringOutput::allOutputs()
{
  QString s;
  FOR_EACH(OutputType, type) s += _outputs[type];
  return s;
}

//----------------------------------------------------------------------------
QStringList received(const QString &s, QString &buffer)
{
  buffer += s;
  buffer.remove('\r');
  QStringList lines = QStringList::split('\n', buffer, true);
  if ( lines.isEmpty() ) buffer = QString::null;
  else {
    buffer = lines.last();
    lines.pop_back();
  }
  return lines;
}

void Process::LineBase::received(OutputType type, const QString &s)
{
  QString& sout  = _outputs[type];
  QStringList lines = ::received(s, sout);
  for (QStringList::const_iterator it=lines.begin(); it!=lines.end(); ++it)
    if ( !isFilteredLine(*it) ) {
      addLine(type, *it);
    }
  if ( !_process->isRunning() && !isFilteredLine(sout) ) addLine(type, sout);
}

//----------------------------------------------------------------------------
QStringList Process::LineOutput::allOutputs()
{
  QStringList lines;
  FOR_EACH(OutputType, type) lines += _lines[type];
  return lines;
}

void Process::LineOutput::clearOutputs()
{
  Process::LineBase::clearOutputs();
  FOR_EACH(OutputType, type) _lines[type].clear();
}

