/***************************************************************************
 *            bridge.c
 *
 *  Workarounds to bridge the code between pilot-link 0.11.8 and 0.12
 *
 *  Tue Dec 13 11:38:41 2005
 *  Copyright  2005-2006  Neil Williams
 *  linux@codehelp.co.uk
 ****************************************************************************/
/*
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */

#include "config.h"
#include "qof-main.h"
#include "bridge.h"
#include "pilot-qof.h"
#include <ctype.h>

#ifndef COMPATIBILITY_MODE
/** use 0.12 code */

#include "pi-header.h"
#include "pi-socket.h"
#include "pi-dlp.h"
#include "pi-source.h"

static const char *env_pilotport = "PILOTPORT";

int plu_connect(char* plu_port, int plu_quiet)
{
	int sd = -1;
	int result;

	struct  SysInfo sys_info;

	/* Determine here for better user feedback on unset port */
	if (plu_port == NULL)
		plu_port = getenv(env_pilotport);
	if (plu_port == NULL) {
		fprintf(stderr, "\n   Unable to determine port to bind\n"
				"   Please use --help for more information\n\n");
		return -1;
	}

	if ((sd = pi_socket(PI_AF_PILOT,
			PI_SOCK_STREAM, PI_PF_DLP)) < 0) {
		fprintf(stderr, "\n   Unable to create socket '%s'\n", plu_port);
		return -1;
	}

	result = pi_bind(sd, plu_port);

	if (result < 0) {
		fprintf(stderr, "   Unable to bind to port: %s\n"
				"   Please use --help for more information\n\n",
				plu_port);
		return result;
	}

	if (!plu_quiet && isatty(fileno(stdout))) {
		printf("\n   Listening for incoming connection on %s... ",
			plu_port);
		fflush(stdout);
	}

	if (pi_listen(sd, 1) < 0) {
		fprintf(stderr, "\n   Error listening on %s\n", plu_port);
		pi_close(sd);
		return -1;
	}

	sd = pi_accept(sd, 0, 0);
	if (sd < 0) {
		fprintf(stderr, "\n   Error accepting data on %s\n", plu_port);
		pi_close(sd);
		return -1;
	}

	if (!plu_quiet && isatty(fileno(stdout))) {
		printf("connected!\n\n");
	}

	if (dlp_ReadSysInfo(sd, &sys_info) < 0) {
		fprintf(stderr, "\n   Error read system info on %s\n", plu_port);
		pi_close(sd);
		return -1;
	}

	dlp_OpenConduit(sd);
	return sd;
}

#endif

pi_buffer_t*
pi_buffer_new (size_t capacity)
{
	pi_buffer_t* buf;
	buf = (struct pi_buffer_t *) malloc (sizeof (struct pi_buffer_t));
	if (buf == NULL)
		return NULL;
	
	if (capacity <= 0)
	{
	/* allocating 0 byte is illegal - use a small value instead */
		 capacity = 16;
	}
	
	buf->data = (unsigned char *) malloc (capacity);
	if (buf->data == NULL) {
		free (buf);
		return NULL;
	}
	
	buf->allocated = capacity;
	buf->used = 0;
	return buf;
}

void
pi_buffer_free (pi_buffer_t* buf)
{
if (buf) {
	if (buf->data)
		free (buf->data);
	free (buf);
}
}


int plu_findcategory(const struct CategoryAppInfo *info, const char *name, int flags)
{
	int cat_index, match_category;

	match_category = -1;
	for (cat_index = 0; cat_index < 16; cat_index += 1) {
		if (info->name[cat_index][0]) {
			if (flags & PLU_CAT_CASE_INSENSITIVE) {
				if (strncasecmp(info->name[cat_index], name, 15) == 0) {
					match_category = cat_index;
					break;
				}
			} else {
				if (strncmp(info->name[cat_index],name,15) == 0) {
					match_category = cat_index;
					break;
				}
			}
		}
	}
	
	if ((match_category == -1)  && (flags & PLU_CAT_MATCH_NUMBERS)) {
		while (isspace(*name)) {
			name++;
		}
		if (isdigit(*name)) {
			match_category = atoi(name);
		}
	
		if ((match_category < 0) || (match_category > 15)) {
			match_category = -1;
		}
	}
	
	if (flags & PLU_CAT_WARN_UNKNOWN) {
		if (match_category == -1) {
			fprintf(stderr,"   WARNING: Unknown category '%s'%s.\n",
				name,
				(flags & PLU_CAT_DEFAULT_UNFILED) ? ", using 'Unfiled'" : "");
		}
	}
	
	if (flags & PLU_CAT_DEFAULT_UNFILED) {
		if (match_category == -1) {
			match_category = 0;
		}
	}
	
	return match_category;
}

