/***************************************************************************
 *            pilot-qof.c
 *
 *   Sat Feb  5 10:40:03 GMT 2005
 *  Copyright  2005  Neil Williams
 *  linux@codehelp.co.uk
 ****************************************************************************/
/*
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */
/** @addtogroup QOF
    @{ */
/** @addtogroup pilot
@{ */

/** @file pilot-qof.c
  @brief Executable interface to the QOF external framework
  @author Copyright (c) 2005-2006 Neil Williams <linux@codehelp.co.uk>
*/

#define _GNU_SOURCE
#include "config.h"
#include <stdio.h>
#include <stdlib.h>
#include <regex.h>

#include "pi-debug.h"
#include "pi-socket.h"
#include "pi-file.h"
#include "pi-header.h"
#include "pi-util.h"
#include "qof-main.h"
#include "pilot-qof.h"

/* used to print debug logs. */
static QofLogModule log_module = PQ_MOD_CLI;
/* pack routines for communication with pilot */
static GList *pilot_modules = NULL;

#define ARGUMENT_BAD_OPTION	17227

#define EXCLUDE_REPEATER_SQL "SELECT * from pilot_datebook " \
"where DATEBOOK_REPEATER == TRUE;"

#define ENUM_LIST_Q(_)  \
	_(qof_op_noop, = 0) \
	_(qof_op_offline, ) \
	_(qof_op_list,)     \
	_(qof_op_hotsync,)  \
	_(qof_op_category,) \
	_(qof_op_database,) \
	_(qof_op_timespec,) \
	_(qof_op_exclude,)  \
	_(qof_op_sql,)      \
	_(qof_op_sql_file,) \
	_(qof_op_write, )   \
	_(qof_op_upload, )  \
	_(qof_op_explain,)  \
	_(qof_op_vers,)     \
	_(qof_op_compress,) \
	_(qof_op_debug,)    \
	_(qof_op_inv_city,) \
	_(qof_op_inv_vendor,)

	DEFINE_ENUM(qof_op_type, ENUM_LIST_Q)

static pilot_qof_data*
pilot_qof_create(void)
{
	pilot_qof_data *context;

	context = g_new0(pilot_qof_data, 1);
	return context;
}

pilot_qof_data*
pilot_qof_init (void)
{
	qof_init();
	g_return_val_if_fail(AddressRegister (), NULL);
	g_return_val_if_fail(ExpensesRegister (), NULL);
	g_return_val_if_fail(DateBookRegister (), NULL);
	g_return_val_if_fail(ToDoRegister (), NULL);
	return pilot_qof_create();
}

void
pilot_qof_pack(QofEntity *ent, gpointer user_data)
{
	const gchar *category_name;
	const QofParam *param;
	int size;
	qof_pack pack_func;
	pilot_qof_data *context;
	const pilot_pack *p;

	context = (pilot_qof_data*)user_data;
	size = 0;
	g_return_if_fail(context != NULL);
	p = pilot_qof_pack_lookup(ent->e_type);
	if(!p) { return; }
	context->pi_buf = pi_buffer_new(PQ_DEF_BUFSZ);
	pack_func = p->pack_func;
	if(pack_func == NULL) 
	{
		context->qof.error = TRUE;
		return;
	}
	size = pack_func(ent, context);
	if(size == -1) { return; }
	param = qof_class_get_parameter(ent->e_type, CATEGORY_NAME);
	category_name = (const gchar*)param->param_getfcn(ent, param);
	context->ent_category = plu_findcategory(&context->pi_cat, 
		category_name, PLU_CAT_CASE_INSENSITIVE | PLU_CAT_DEFAULT_UNFILED);
	if(context->ent_category == 0)
	{
		PWARN (" Category: '%s' not found or not set, using 'Unfiled'",
			category_name);
	}
	if(dlp_WriteRecord(context->sd, context->db, 
		PQ_DLP_REC_ATTR, PQ_DLP_NEW_REC, context->ent_category, 
		(unsigned char*)context->pi_buf->data, context->pi_buf->used, 
		PQ_DLP_SET_ID) < 0)
	{
		PERR (" record could not be written");
		return;
	}
}

void
pilot_qof_unpack(QofEntity *ent, gpointer user_data)
{
	qof_pack unpack_func;
	const pilot_pack *p;
	pilot_qof_data *context;
	int result;

	context = (pilot_qof_data*)user_data;
	g_return_if_fail(context && ent);
	p = pilot_qof_pack_lookup(ent->e_type);
	g_return_if_fail(p);
	unpack_func = p->unpack_func;
	if(unpack_func == NULL) {
		context->qof.error = TRUE;
		PERR ("No unpack routine was defined for the %s object!",
			ent->e_type);
		return;
	}
	result = unpack_func(ent, context);
	if(result == -1) {
		qof_entity_release(ent);
		g_free(ent);
	}
}

void
pilot_app_unpack(QofIdTypeConst e_type, gpointer user_data)
{
	pilot_qof_data *context;
	qof_pack app_unpack;
	const pilot_pack *p;

	context = (pilot_qof_data*)user_data;
	g_return_if_fail(context != NULL);
	p = pilot_qof_pack_lookup(e_type);
	if(!p) { return; }
	app_unpack = p->app_info_unpack;
	if(app_unpack == NULL) {
		context->qof.error = TRUE;
		PERR (" no unpack routine for %s", e_type);
		return;
	}
	/* no entity available for the appInfo, pass NULL and work only in the context. */
	app_unpack(NULL, context);
}

static void
pilot_entity_free(QofEntity *ent, gpointer user_data)
{
	qof_pack free_pack_func;
	const pilot_pack *p;

	p = pilot_qof_pack_lookup(ent->e_type);
	if(!p) { return; }
	free_pack_func = p->free_pack_func;
	free_pack_func(ent, NULL);
}

static void
pilot_object_free(QofObject *obj, gpointer user_data)
{
	QofBook *book;

	book = (QofBook*)user_data;
	qof_object_foreach(obj->e_type, book, pilot_entity_free, NULL);
}

void
pilot_entity_finaliser(QofBook *book, gpointer key, gpointer data)
{
	qof_object_foreach_type(pilot_object_free, book);
}

gboolean
pilot_qof_pack_register(const pilot_pack *p)
{
	if (g_list_index (pilot_modules, (gpointer)p) == -1)
	{
		pilot_modules = g_list_prepend(pilot_modules, (gpointer)p);
	}
	else { return FALSE; }
	return TRUE;
}

const pilot_pack *
pilot_qof_pack_lookup(QofIdTypeConst object_type)
{
	GList      *iter;
	pilot_pack *p;

	if (!object_type) { return NULL; }
	for(iter = pilot_modules; iter; iter = iter->next)
	{
		p = iter->data;
		if(0 == safe_strcmp(p->e_type, object_type))
		{
			return p;
		}
	}
	return NULL;
}

static void
pilot_qof_free(pilot_qof_data *data)
{
	qof_main_free(&data->qof);
}

static void
write_ent_cb (QofEntity *ent, gpointer user_data)
{
	pilot_qof_data *context;

	context = (pilot_qof_data*)user_data;
	g_return_if_fail(context && ent);
	if(context->qof.error) { return; }
	context->pi_buf = pi_buffer_new(PQ_DEF_BUFSZ);
	pilot_qof_pack(ent, context);
}

static void
pilot_database_open(QofObject *obj, gpointer user_data)
{
	char *db_name;
	char *log;
	pilot_qof_data *context;
	const pilot_pack *p;
	int db, attr, category, i, len;
	QofBook *book;
	QofEntity *ent;
	QofInstance *inst;

	context = (pilot_qof_data*)user_data;
	if(context->qof.error) { return; }
	len = 0;
	i = 0;
	p = pilot_qof_pack_lookup(obj->e_type);
	if(!p) { return; }
	db_name = g_strdup(p->palm_db_name);
	if(db_name == NULL) { 
		context->qof.error = TRUE; 
		PERR (" object %s has no database name", obj->e_type);
		return; 
	}
	if(0 == safe_strcmp(obj->e_type, context->qof.exclude)) { return; }
	/* If no database specified, query all except any excluded database. */
	if((context->qof.database != NULL)&&
		(0 != safe_strcmp(obj->e_type, context->qof.database))) { return; }
	if (dlp_OpenDB(context->sd, PQ_DLP_CARD, 
			PI_DLP_ARG_FLAG_SHORT | PI_DLP_ARG_FLAG_LONG, 
			db_name, &db) < 0) 
	{
		PWARN (" Unable to open %s database on Palm.\n", db_name);
		log = g_strdup_printf(_("%s: Unable to open %s database on Palm.\n"), 
			PACKAGE, db_name);
		dlp_AddSyncLogEntry(context->sd, log);
		pi_close(context->sd);
		context->qof.error = TRUE;
		return;
	}
	context->db = db;
	context->app_buf = pi_buffer_new(PQ_DEF_BUFSZ);
	/* each database has it's own category list. */
	dlp_ReadAppBlock(context->sd, context->db, PQ_DLP_OFFSET, 
		(int)context->app_buf->allocated, context->app_buf);
	pilot_app_unpack(obj->e_type, context);
	/* Write each entity if an upload file was specified. */
	if(context->qof.input_file)
	{
		book = qof_session_get_book(context->qof.input_session);
		qof_object_foreach(obj->e_type, book, write_ent_cb, context);
	}
	context->pi_buf = pi_buffer_new(PQ_DEF_BUFSZ);
	book = qof_session_get_book(context->qof.input_session);
	for(i = 0; len >= 0; i++)
	{
		len = dlp_ReadRecordByIndex(context->sd, context->db, i, 
			context->pi_buf, PQ_DLP_RECORD, &attr, &category);
		/* attr holds archive - when deleted from Palm with option to archive on PC */
		if ((attr & dlpRecAttrDeleted) || (attr & dlpRecAttrArchived)) { continue; }
		/* category holds the index of the category in this database category list. */
		context->ent_category = category;
		/* Create new entity to hold the unpacked data. */
		inst = (QofInstance*)qof_object_new_instance(obj->e_type, book);
		g_return_if_fail(inst != NULL);
		ent = &inst->entity;
		pilot_qof_unpack(ent, context);
		if(context->qof.error == TRUE) { len=-1; break; }
	}
	pi_buffer_free(context->app_buf);
	pi_buffer_free(context->pi_buf);
	dlp_CloseDB(context->sd, context->db);
}

static void 
pilot_error (pilot_qof_data *context, const char* message)
{
#ifndef COMPATIBILITY_MODE
	PERR (" %s - Error code: %d", message, pi_error(context->sd));
#endif
	if(context->sd > 0) { pi_close(context->sd); }
	qof_session_end(context->qof.input_session);
	qof_session_end(context->qof.export_session);
	context->qof.error = TRUE;
}

static void
find_invoice_contact(QofEntity *ent, gpointer data)
{
	GSList *field_list, *category_param_list;
	QofQueryPredData *appt_pred, *exp_pred, *category_pred;
	pilot_qof_data *context;
	const QofParam *param;
	gchar *string;
	GList *results;
	QofBook *book;

	context = (pilot_qof_data*)data;
	g_return_if_fail(context && ent);
	param = NULL;
	string = NULL;
	results = NULL;
	category_param_list = NULL;
	if(0 != safe_strcmp(ent->e_type, PILOT_LINK_QOF_EXPENSES)
		&& (0 != safe_strcmp(ent->e_type, PILOT_LINK_QOF_DATEBOOK)))
	{
		return;
	}
	ENTER (" ent=%s", ent->e_type);
	book = qof_session_get_book(context->qof.input_session);
	qof_query_set_book(context->qof.query, book);
	field_list = NULL;
	/* remember: query runs in address using strings from expenses or datebook. */
	if(context->qof.category) {
		category_param_list = qof_query_build_param_list(CATEGORY_NAME, NULL);
		category_pred = qof_query_string_predicate(QOF_COMPARE_EQUAL, 
			context->qof.category, QOF_STRING_MATCH_CASEINSENSITIVE, FALSE);
		qof_query_add_term(context->qof.query, category_param_list, 
			category_pred, QOF_QUERY_AND);
	}
	/* lookup appointment or expenses strings */
	if(0 == safe_strcmp(ent->e_type, PILOT_LINK_QOF_EXPENSES))
	{
		if(context->invoice_vendor)
		{
			param = qof_class_get_parameter(ent->e_type, EXP_VENDOR);
			string = param->param_getfcn(ent, param);
			field_list = qof_query_build_param_list(ADDR_CITY, NULL);
		}
		if(context->invoice_city)
		{
			param = qof_class_get_parameter(ent->e_type, EXP_CITY);
			string = param->param_getfcn(ent, param);
			field_list = qof_query_build_param_list(ADDR_COMPANY, ADDR_TITLE, NULL);
		}
		if(string)
		{
			exp_pred = qof_query_string_predicate(QOF_COMPARE_EQUAL,
				string, QOF_STRING_MATCH_CASEINSENSITIVE, FALSE);
			qof_query_add_term(context->qof.query, field_list, 
				exp_pred, QOF_QUERY_AND);
			results = qof_query_run (context->qof.query);
			if(results != NULL) {
				qof_entity_copy_list(context->qof.export_session, results);
			}
		}
	}
	if(0 == safe_strcmp(ent->e_type, PILOT_LINK_QOF_DATEBOOK))
	{
		param = qof_class_get_parameter(ent->e_type, DATEBOOK_DESCRIPTION);
		string = param->param_getfcn(ent, param);
		field_list = qof_query_build_param_list(ADDR_COMPANY, 
			ADDR_CITY, ADDR_TITLE, NULL);
		if(string)
		{
			appt_pred = qof_query_string_predicate(QOF_COMPARE_EQUAL,
				string, QOF_STRING_MATCH_CASEINSENSITIVE, FALSE);
			qof_query_add_term(context->qof.query, field_list,
				appt_pred, QOF_QUERY_AND);
			results = qof_query_run (context->qof.query);
			if(results != NULL) {
				qof_entity_copy_list(context->qof.export_session, results);
			}
		}
	}
	/* copy matching contacts into book */
	if(context->qof.query) { qof_query_clear(context->qof.query); }
	LEAVE (" ");
}

static void
check_invoice_handler(pilot_qof_data *context)
{
	if(context->qof.min_ts.tv_sec == 0)
	{
		fprintf(stderr, _("%s: Error: Please specify a time period "
			"using -t."), PACKAGE);
		qof_session_end(context->qof.input_session);
		context->qof.error = TRUE;
		return;
	}
	if(context->qof.exclude)
	{
		g_free(context->qof.exclude);
		context->qof.exclude = NULL;
	}
	context->qof.exclude = g_strdup(PILOT_LINK_QOF_TODO);
	if(context->qof.database)
	{
		g_free(context->qof.database);
		context->qof.database = NULL;
	}
	if(context->qof.input_file)
	{
		g_free(context->qof.input_file);
		context->qof.input_file = NULL;
	}
	if(context->qof.sql_str)
	{
		g_free(context->qof.sql_str);
		context->qof.sql_str = NULL;
	}
	if(context->qof.sql_list)
	{
		g_list_free(context->qof.sql_list);
		context->qof.sql_list = NULL;
	}
}

void
qof_cmd_hotsync (pilot_qof_data *context)
{
	struct PilotUser QUser;
	QofBook *book;
	gchar *log_msg;

	if(0 == safe_strcmp(context->qof.exclude, context->qof.database)
		&&(context->qof.exclude != NULL))
	{
		fprintf(stderr, _("%s: Error: Cannot exclude database \"%s\" with option -e\n"
		"\tbecause option -d is set to the same database: \"%s\"\n"),
				PACKAGE, context->qof.exclude, context->qof.database);
		qof_session_end(context->qof.input_session);
		return;
	}
	if((context->invoice_city) ||(context->invoice_vendor)) 
	{ 
		check_invoice_handler(context);
	}
	if(context->qof.error) { return; }
	if(context->qof.input_file)
	{
		PINFO (" Trying to upload %s", context->qof.input_file);
		qof_session_begin(context->qof.input_session, 
			context->qof.input_file, TRUE, FALSE);
		qof_session_load(context->qof.input_session, NULL);
	}
	else { 
		qof_session_begin(context->qof.input_session, QOF_STDOUT, TRUE, FALSE); 
	}
	context->qof.export_session = qof_session_new();

	context->sd = plu_connect(context->port, context->quiet);

	if(context->sd < 0) {
		pilot_error(context, _("Unable to connect to the Palm"));
		return;
	}
	if (dlp_ReadUserInfo(context->sd, &QUser) < 0) {
		pilot_error (context, _("Unable to read Palm user info"));
		return;
	}
	context->qof.error = FALSE;
	qof_object_foreach_type(pilot_database_open, context);
	QUser.lastSyncPC = 0x00010000;
	QUser.lastSyncDate = QUser.successfulSyncDate = time(0);
	if(dlp_WriteUserInfo(context->sd, &QUser) < 0) {
		pilot_error(context, _("Unable to write user info"));
		return;
	}
	log_msg = g_strdup_printf(_("%s hotsync\n\n"
		"Thank you for using %s.\n"), PACKAGE, PACKAGE);
	dlp_AddSyncLogEntry(context->sd, log_msg);
	g_free(log_msg);
	dlp_EndOfSync(context->sd, 0);
	pi_close(context->sd);
	/* Now run the query, copy the objects, destroy the input session
	and write out export_session */
	if(context->qof.write_file) {
		qof_session_begin(context->qof.export_session,
			context->qof.write_file, TRUE, TRUE);
		qof_mod_compression (context->qof.gz_level, &context->qof);
	}
	else { qof_session_begin(context->qof.export_session, QOF_STDOUT, TRUE, FALSE); }
	/** \note if no query is given, ignore repeater clones?
	Nice idea, but in practice, difficult. It also makes recursive
	queries of the XML harder as the XML needs to understand datebook
	repeats. Instead, we simply ignore all repeater clones when it comes
	to write data to the Palm by a simple check in datebook_pack.
	*/
	qof_main_moderate_query(&context->qof);
	/* if invoice_hook, create a second query and lookup in contacts */
	if((context->invoice_city) || (context->invoice_vendor))
	{
		book = qof_session_get_book(context->qof.export_session);
		context->qof.query = qof_query_create_for(PILOT_LINK_QOF_ADDRESS);
		qof_object_foreach(PILOT_LINK_QOF_DATEBOOK, book, find_invoice_contact, context);
		qof_object_foreach(PILOT_LINK_QOF_EXPENSES, book, find_invoice_contact, context);
	}
	qof_session_save(context->qof.export_session, NULL);
	qof_main_show_error(context->qof.export_session);
	qof_session_end(context->qof.input_session);
	qof_session_end(context->qof.export_session);
}

static void
pq_invoice_xmlfile (pilot_qof_data *context)
{
	QofBook *book;

	ENTER (" ");
	qof_session_begin(context->qof.input_session, context->qof.filename, TRUE, FALSE);
	qof_session_load(context->qof.input_session, NULL);
	context->qof.export_session = qof_session_new();
	if(context->qof.write_file) {
		qof_session_begin(context->qof.export_session, 
			context->qof.write_file, TRUE, TRUE);
		qof_mod_compression (context->qof.gz_level, &context->qof);
	}
	else { qof_session_begin(context->qof.export_session, QOF_STDOUT, TRUE, TRUE); }
	qof_main_moderate_query(&context->qof);
	book = qof_session_get_book(context->qof.export_session);
	context->qof.query = qof_query_create_for(PILOT_LINK_QOF_ADDRESS);
	qof_object_foreach(PILOT_LINK_QOF_DATEBOOK, book, find_invoice_contact, context);
	qof_object_foreach(PILOT_LINK_QOF_EXPENSES, book, find_invoice_contact, context);
	qof_session_save(context->qof.export_session, NULL);
	qof_main_show_error(context->qof.export_session);
	qof_main_show_error(context->qof.input_session);
	qof_session_end(context->qof.input_session);
	qof_session_end(context->qof.export_session);
	LEAVE (" ");
}

int
main (int argc, const char *argv[])
{
	const char *help_header_text, *input_file;
	int plu_quiet;
	char *plu_port;
	pilot_qof_data *pilot_qof_context;
	gboolean debug_on;
	poptContext pc;
	int optc, gz_level;
	qof_op_type palm_command;
	QOF_OP_VARS

	struct poptOption options[] = {
		{ "port", 'p', POPT_ARG_STRING, &plu_port,  0 ,
 		 _("Use the device <port> to communicate with Palm"), "<port>"},
		ENABLE_QUIET_MODE
		{"hot-query", 'a', POPT_ARG_NONE, NULL, qof_op_hotsync,
		 _("Activate/HotSync and query the Palm."), NULL},
		{"upload", 'u', POPT_ARG_STRING, &input_file, qof_op_upload,
		 _("Upload data from <filename> to the Palm. Requires -a"), "filename"},
		{"invoice-vendor", 0, POPT_ARG_NONE, NULL, qof_op_inv_vendor,
		 _("Shorthand to relate an event or expense to a contact, by vendor. "
		  "Requires -t."), NULL},
		{"invoice-city", 0, POPT_ARG_NONE, NULL, qof_op_inv_city,
		 _("Shorthand to relate an event or expense to a contact, by city. "
		  "Requires -t."), NULL},
		QOF_CLI_OPTIONS
		POPT_TABLEEND
	};
	palm_command = qof_op_noop;
	debug_on = FALSE;
	QOF_OP_INIT;
	input_file = NULL;
	plu_quiet = 0;
	plu_port = NULL;

	help_header_text = _(
		"\n"
		"   Application interface between pilot-link and QOF - \n"
		"   the Query Object Framework.\n"
		"   Supports writing palm data to QSF XML offline storage and running\n"
		"   SQL-type queries on the live data or XML file.\n"
		"   See http://pilot-qof.sourceforge.net/\n"
		"   QSF XML can be imported into other QOF applications.\n\n"
		"   Use exactly one of -o -a -l --explain;\n"
		"   options are -c -t -w, -d or -e, -s or -f.\n"
	    "   option -u requires -a\n\n");

	#ifdef ENABLE_NLS
	setlocale (LC_ALL, "");
	bindtextdomain (GETTEXT_PACKAGE, LOCALE_DIR);
	bind_textdomain_codeset (GETTEXT_PACKAGE, "UTF-8");
	textdomain (GETTEXT_PACKAGE);
	#endif
	pc = poptGetContext (PACKAGE, argc, argv, options, 0);

	poptSetOtherOptionHelp (pc, help_header_text);

	if (argc < 2)
	{
		poptPrintUsage (pc, stderr, 0);
		return EXIT_FAILURE;
	}
	pilot_qof_context = pilot_qof_init();
	if(!pilot_qof_context) 
	{ 
		fprintf (stderr, _("%s: Failed to initialise the query object framework."),
			PACKAGE);
		return EXIT_FAILURE; 
	}
	while ((optc = poptGetNextOpt (pc)) >= 0)
	{
		switch (optc)
		{
			/* commands - mutually exclusive */
			case qof_op_offline:
			case qof_op_list:
			case qof_op_explain:
			case qof_op_hotsync:
			{
				if (qof_op_noop != palm_command)
				{
					fprintf (stderr, _("%s: ERROR: specify only one of "
					"-o, -a, -l or --explain.\n"), PACKAGE);
					poptPrintUsage(pc, stderr, 0);
					return EXIT_FAILURE;
				}
				palm_command = optc;
				break;
			}
			case qof_op_vers :
			{
				fprintf (stdout, _("\n This is %s v%s\n"), PACKAGE, VERSION);
				fprintf (stdout, _(" Query interface for Palm databases as objects.\n"));
				fprintf (stdout, "\n Copyright (c) 2005-2006 "
					"Neil Williams <linux@codehelp.co.uk>\n");
				fprintf (stdout, _(" For %s support, join the QOF-devel "
					"mailing list at\n"), PACKAGE);
				fprintf (stdout, 
					" http://lists.sourceforge.net/mailman/listinfo/qof-devel\n\n");
				/* Translators: Add or subtract dots to keep the translated
				lines aligned vertically */
				fprintf (stdout, _(" Build target............: %s\n"), HOST_OS);
				fprintf (stdout, _(" Build date..............: %s %s\n"), 
					__DATE__, __TIME__);
				fprintf (stdout, _(" Built for pilot-link ...: %s\n"),
					PILOT_LINK_SUPPORT);
				fprintf (stdout, _(" --debug logs to.........: %s\n\n"), PILOT_QOF_LOG);
				fprintf (stdout, _(" Please use --help for more detailed options.\n\n"));
				return EXIT_SUCCESS;
			}
			/* optional modifiers - store to act on later. */
			case qof_op_category:
			{
				qof_mod_category (category, &pilot_qof_context->qof);
				break;
			}
			case qof_op_database:
			{
				qof_mod_database (database, &pilot_qof_context->qof);
				break;
			}
			case qof_op_timespec:
			{
				qof_mod_timespec (date_time, &pilot_qof_context->qof);
				break;
			}
			case qof_op_exclude:
			{
				qof_mod_exclude (exclude, &pilot_qof_context->qof);
				break;
			}
			case qof_op_sql:
			{
				qof_mod_sql (sql_query, &pilot_qof_context->qof);
				break;
			}
			case qof_op_sql_file:
			{
				qof_mod_sql_file (sql_file, &pilot_qof_context->qof);
				break;
			}
			case qof_op_write:
			{
				qof_mod_write (write_file, &pilot_qof_context->qof);
				break;
			}
			case qof_op_upload :
			{
				if(palm_command != qof_op_hotsync)
				{
					fprintf (stderr, 
					_("%s: Error: Please specify -a if you use -u\n"), PACKAGE);
					poptPrintUsage(pc, stderr, 0);
					return EXIT_FAILURE;
				}
				pilot_qof_context->qof.input_file = g_strdup(input_file);
				break;
			}
			case qof_op_debug:
			{
				qof_log_init_filename(PILOT_QOF_LOG);
				qof_log_set_default(QOF_LOG_DETAIL);
				qof_log_set_level(PQ_MOD_CLI, QOF_LOG_DETAIL);
				qof_log_set_level(PQ_MOD_PILOT, QOF_LOG_DETAIL);
				qof_log_set_level(QOF_MAIN_CLI, QOF_LOG_DETAIL);
				qof_log_set_level(QOF_MOD_QSF, QOF_LOG_DETAIL);
				debug_on = TRUE;
				break;
			}
			case qof_op_compress:
			{
				pilot_qof_context->qof.gz_level = gz_level;
				break;
			}
			case qof_op_inv_vendor :
			{
				if(!pilot_qof_context->invoice_city)
				{
					pilot_qof_context->invoice_vendor = TRUE;
				}
				break;
			}
			case qof_op_inv_city :
			{
				pilot_qof_context->invoice_city = TRUE;
				pilot_qof_context->invoice_vendor = FALSE;
				break;
			}
			default:
			{
				fprintf (stderr, _("%s: ERROR: got option %d, arg %s\n"), 
				PACKAGE, optc, poptGetOptArg (pc));
				return EXIT_FAILURE;
			}
		}
	}
	if (optc < -1)
	{
		fprintf(stderr, "%s: %s %s\n\n", PACKAGE,
		poptBadOption(pc, POPT_BADOPTION_NOALIAS),
		poptStrerror(optc));
		poptPrintUsage(pc, stderr, 0);
		return EXIT_FAILURE;
	}
	/* If we get this far, we should have sensible options: start the work. */
	pilot_qof_context->qof.input_session = qof_session_new();
	switch (palm_command)
	{
		case qof_op_offline:
		{
			pilot_qof_context->qof.filename = g_strdup(filename);
			if((pilot_qof_context->invoice_city) || (pilot_qof_context->invoice_vendor))
			{
				check_invoice_handler(pilot_qof_context);
				pq_invoice_xmlfile (pilot_qof_context);
			}
			else
			{
				qof_cmd_xmlfile (&pilot_qof_context->qof);
			}
			break;
		}
		case qof_op_list:
		{
			qof_cmd_list ();
			break;
		}
		case qof_op_explain:
		{
			if(!pilot_qof_context->qof.database) 
			{ 
				fprintf (stderr, _("%s: Error: please specify which database "
				"you would like explained.\n\n"), PACKAGE);
				break;
			}
			qof_cmd_explain(&pilot_qof_context->qof);
			break;
		}
		case qof_op_hotsync:
		{
			pilot_qof_context->port = g_strdup(plu_port);
			pilot_qof_context->quiet = plu_quiet;
			qof_cmd_hotsync (pilot_qof_context);
			break;
		}
		default:
		{
			/* should be impossible */
			break;
		}
	}
	poptFreeContext(pc);
	pilot_qof_free(pilot_qof_context);
	if(debug_on) { qof_log_shutdown(); }
	qof_close();
	return EXIT_SUCCESS;
}

/** @} */
/** @} */
