/***************************************************************************
 *            qof-invoice.c
 *
 *  Sat 22 Nov 2008 23:02:03 GMT
 *  Copyright  2008  Neil Williams  <linux@codehelp.co.uk>
 ****************************************************************************/
/*
    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <stdlib.h>
#include <glib.h>
#include <glib/gprintf.h>
#include <qof.h>
#include "qof-main.h"
#include "qof-datebook.h"
#ifdef HAVE_QOFEXPENSES
#include <qof-expenses.h>
//#define PILOT_LINK_QOF_EXPENSES GPE_QOF_EXPENSES
#else
#define qof_currency_lookup pq_currency_lookup
#define QofCurrency PQCurrency
#endif
#include "pilot-expenses.h"
#include "pilot-expenses-p.h"
#include "qof-address.h"
#include "qof-invoice.h"

#define QOF_INVOICE_DESC "Pilot-link QOF Invoice"

static QofLogModule log_module = INVOICE_MOD;

static GList * date_list = NULL;
static GList * addr_list = NULL;
static GList * exp_list = NULL;

typedef struct
{
	QofInstance inst;
	gchar *category;
	const gchar *print_string;
	QofCurrency *currency;
	gdouble temp_amount;   /**<  The amount is set before the
		currency code is available so this acts as a buffer until both
		pieces of data are available. */
	gboolean reset_amount; /**< Denote whether the buffer should be used. */
	QofDate * inv_date, * pay_date;
	gchar * reference;
	gchar * note;
	gchar * description;
	gchar * city;
	gchar * vendor;
	/** the amount is a total - individual items will need
	to wait for later. */
	QofNumeric amount;
} QofInvoice;

static QofInvoice *
invoice_create (QofBook * book)
{
	QofInvoice *obj;
	QofCollection *coll;
	GList *all;

	obj = g_new0 (QofInvoice, 1);
	qof_instance_init (&obj->inst, PILOT_INVOICE, book);
	coll = qof_book_get_collection (book, PILOT_INVOICE);
	all = qof_collection_get_data (coll);
	all = g_list_prepend (all, obj);
	qof_collection_set_data (coll, all);
	obj->amount = qof_numeric_zero();
	/** \bug QOF bug - should be possible to have an empty date/time */
	obj->inv_date = qof_date_new_dmy(1,1,1970);
	obj->pay_date = qof_date_new_dmy(1,1,1970);
	obj->currency = g_new0(QofCurrency, 1);
	obj->currency->pq_code = -1;
	return obj;
}

static QofTime*
inv_getTime (QofInvoice * e)
{
	QofTime *qt;

	g_return_val_if_fail (e != NULL, NULL);
	ENTER (" ");
	/** \bug only enable once QSF can cope with nulls.
	if (!qof_date_valid(e->inv_date))
		return NULL;*/
	qt = qof_date_to_qtime (e->inv_date);
	LEAVE (" ");
	return qt;
}

static QofTime*
pay_getTime (QofInvoice * e)
{
	QofTime *qt;

	g_return_val_if_fail (e != NULL, NULL);
	/** \bug only enable once QSF can cope with nulls.
	if (!qof_date_valid(e->inv_date))
		return NULL;*/
	qt = qof_date_to_qtime (e->pay_date);
	return qt;
}

static gint
inv_getCurrency (QofInvoice * i)
{
	QofCurrency * c;

	g_return_val_if_fail (i != NULL, -1);
	if (!i->currency)
		return -1;
	c = i->currency;
	return c->pq_code;
}

static const gchar *
inv_getReference (QofInvoice * e)
{
	g_return_val_if_fail (e != NULL, NULL);
	return e->reference;
}

static const gchar *
inv_getDescription (QofInvoice * e)
{
	g_return_val_if_fail (e != NULL, NULL);
	return e->description;
}

static QofNumeric
inv_getAmount (QofInvoice * e)
{
	QofNumeric amount;

	amount = qof_numeric_zero ();
	g_return_val_if_fail (e != NULL, amount);
	if (qof_numeric_check (e->amount) == QOF_ERROR_OK)
		return e->amount;
	return qof_numeric_zero ();
}

static const gchar *
inv_getVendor (QofInvoice * e)
{
	g_return_val_if_fail (e != NULL, NULL);
	return e->vendor;
}

static const gchar *
inv_getCity (QofInvoice * e)
{
	g_return_val_if_fail (e != NULL, NULL);
	return e->city;
}

static const gchar *
inv_getNote (QofInvoice * e)
{
	g_return_val_if_fail (e != NULL, NULL);
	return e->note;
}

static const gchar *
inv_getCategory (QofInvoice * e)
{
	g_return_val_if_fail (e != NULL, NULL);
	return e->category;
}

static void
inv_setTime (QofInvoice * e, QofTime *h)
{
	g_return_if_fail (e != NULL);
	e->inv_date = qof_date_from_qtime (h);
}

static void
pay_setTime (QofInvoice * e, QofTime *h)
{
	g_return_if_fail (e != NULL);
	e->pay_date = qof_date_from_qtime (h);
}

static void
inv_setReference (QofInvoice * e, const gchar * type_string)
{
	g_return_if_fail (e != NULL);
	e->reference = g_strdup (type_string);
}

static void
inv_setCurrency (QofInvoice * i, gint code)
{
	g_return_if_fail (i != NULL);
	i->currency = qof_currency_lookup ((QofInstance*) i, code);
}

static void
inv_setDescription (QofInvoice * e, const gchar * desc_string)
{
	g_return_if_fail (e != NULL);
	e->description = g_strdup (desc_string);
}

static void
inv_setAmount (QofInvoice * e, QofNumeric h)
{
	g_return_if_fail (e != NULL);
	e->amount = h;
	e->temp_amount = qof_numeric_to_double (h);
	e->reset_amount = TRUE;
	/* if an amount can ever be set without a currency_code,
	   this needs to be reviewed. */
	/** \todo FIXME: INSERT handler can set one without the
	other. Need to use the pref.default_currency? */
/*	if (e->currency)
		inv_combine_currency_with_amount (e);*/
}

static void
inv_setVendor (QofInvoice * e, gchar * h)
{
	g_return_if_fail (e != NULL);
	e->vendor = g_strdup (qof_util_make_utf8 (h));
}

static void
inv_setCity (QofInvoice * e, gchar * h)
{
	g_return_if_fail (e != NULL);
	e->city = g_strdup (qof_util_make_utf8 (h));
}

static void
inv_setNote (QofInvoice * e, gchar * h)
{
	g_return_if_fail (e != NULL);
	e->note = g_strdup (qof_util_make_utf8 (h));
}

static void
inv_setCategory (QofInvoice * e, gchar * n)
{
	g_return_if_fail (e != NULL);
	e->category = g_strdup (qof_util_make_utf8 (n));
}

static QofCollection *
inv_getDates (QofInvoice * invoice)
{
	QofCollection *entry_coll;
	GList         *list;
	QofEntity     *entry;

	entry_coll = qof_collection_new(PILOT_LINK_QOF_DATEBOOK);
	for(list = date_list; list != NULL; list = list->next)
	{
		entry = QOF_ENTITY(list->data);
		qof_collection_add_entity(entry_coll, entry);
	}
	return entry_coll;
}

static QofCollection *
inv_getAddresses (QofInvoice * invoice)
{
	QofCollection *entry_coll;
	GList         *list;
	QofEntity     *entry;

	entry_coll = qof_collection_new(PILOT_LINK_QOF_ADDRESS);
	for(list = addr_list; list != NULL; list = list->next)
	{
		entry = QOF_ENTITY(list->data);
		qof_collection_add_entity(entry_coll, entry);
	}
	return entry_coll;
}

static QofCollection *
inv_getExpenses (QofInvoice * invoice)
{
	QofCollection *entry_coll;
	GList         *list;
	QofEntity     *entry;

	entry_coll = qof_collection_new(PILOT_LINK_QOF_EXPENSES);
	for(list = exp_list; list != NULL; list = list->next)
	{
		entry = QOF_ENTITY(list->data);
		qof_collection_add_entity(entry_coll, entry);
	}
	return entry_coll;
}

static void
inv_add_date_cb (QofEntity *ent, gpointer user_data)
{
	QofInvoice *invoice;

	invoice = (QofInvoice*)user_data;
	if(!invoice || !ent) { return; }
	date_list = g_list_append (date_list, ent);
}

static void
inv_add_addr_cb (QofEntity *ent, gpointer user_data)
{
	QofInvoice *invoice;

	invoice = (QofInvoice*)user_data;
	if(!invoice || !ent) { return; }
	addr_list = g_list_append (addr_list, ent);
}

static void
inv_add_exp_cb (QofEntity *ent, gpointer user_data)
{
	QofInvoice *invoice;

	invoice = (QofInvoice*)user_data;
	if(!invoice || !ent) { return; }
	exp_list = g_list_append (exp_list, ent);
}

static void 
inv_setDates (QofInvoice *invoice, QofCollection *entry_coll)
{
	if(!entry_coll) { return; }
	if(0 == safe_strcmp(qof_collection_get_type(entry_coll), PILOT_LINK_QOF_DATEBOOK))
	{
		qof_collection_foreach(entry_coll, inv_add_date_cb, invoice);
	}
}

static void 
inv_setAddresses (QofInvoice *invoice, QofCollection *entry_coll)
{
	if(!entry_coll) { return; }
	if(0 == safe_strcmp(qof_collection_get_type(entry_coll), PILOT_LINK_QOF_ADDRESS))
	{
		qof_collection_foreach(entry_coll, inv_add_addr_cb, invoice);
	}
}

static void 
inv_setExpenses (QofInvoice *invoice, QofCollection *entry_coll)
{
	if(!entry_coll) { return; }
	if(0 == safe_strcmp(qof_collection_get_type(entry_coll), PILOT_LINK_QOF_EXPENSES))
	{
		qof_collection_foreach(entry_coll, inv_add_exp_cb, invoice);
	}
}

static const gchar *
invoicePrintable (gpointer instance)
{
	QofInvoice *obj;

	obj = (QofInvoice *) instance;
	if (!obj)
		return NULL;
	if (inv_getDescription (obj))
		return g_strconcat (inv_getDescription (obj), " ",
			inv_getVendor (obj), " ", inv_getCity (obj), NULL);
	return NULL;
}

static QofObject expenses_object_def = {
  interface_version   :  QOF_OBJECT_VERSION,
  e_type              :  PILOT_INVOICE,
  type_label          :  QOF_INVOICE_DESC,
  create              :  (gpointer) invoice_create,
  book_begin          :  NULL,
  book_end            :  NULL,
  is_dirty            :  qof_collection_is_dirty,
  mark_clean          :  qof_collection_mark_clean,
  foreach             :  qof_collection_foreach,
  printable           :  invoicePrintable,
  version_cmp         :  (gint (*)(gpointer, gpointer)) qof_instance_version_cmp,
};

gboolean
InvoiceRegister (void)
{
	static QofParam params[] = {
		{INV_DATE, QOF_TYPE_TIME, (QofAccessFunc) inv_getTime,
				(QofSetterFunc) inv_setTime, NULL},
		{PAY_DATE, QOF_TYPE_TIME, (QofAccessFunc) pay_getTime,
				(QofSetterFunc) pay_setTime, NULL},
		{INV_REFERENCE, QOF_TYPE_STRING, (QofAccessFunc) inv_getReference,
				(QofSetterFunc) inv_setReference, NULL},
		{INV_CURRENCY, QOF_TYPE_INT32, (QofAccessFunc) inv_getCurrency,
				(QofSetterFunc) inv_setCurrency, NULL},
		{INV_AMOUNT, QOF_TYPE_NUMERIC, (QofAccessFunc) inv_getAmount,
				(QofSetterFunc) inv_setAmount, NULL},
		{INV_VENDOR, QOF_TYPE_STRING, (QofAccessFunc) inv_getVendor,
				(QofSetterFunc) inv_setVendor, NULL},
		{INV_CITY, QOF_TYPE_STRING, (QofAccessFunc) inv_getCity,
				(QofSetterFunc) inv_setCity, NULL},
		{INV_DESCRIPTION, QOF_TYPE_STRING, (QofAccessFunc) inv_getDescription,
				(QofSetterFunc) inv_setDescription, NULL},
		{INV_NOTE, QOF_TYPE_STRING, (QofAccessFunc) inv_getNote,
				(QofSetterFunc) inv_setNote, NULL},
		{INV_CATEGORY, QOF_TYPE_STRING, (QofAccessFunc) inv_getCategory,
				(QofSetterFunc) inv_setCategory, NULL},
		{INV_DATES,   QOF_TYPE_COLLECT, (QofAccessFunc)inv_getDates, 
				(QofSetterFunc)inv_setDates, NULL},
		{INV_ADDR,    QOF_TYPE_COLLECT, (QofAccessFunc)inv_getAddresses, 
				(QofSetterFunc)inv_setAddresses, NULL},
		{INV_EXPENSE, QOF_TYPE_COLLECT, (QofAccessFunc)inv_getExpenses, 
				(QofSetterFunc)inv_setExpenses, NULL},
		{INV_KVP, QOF_TYPE_KVP, (QofAccessFunc) qof_instance_get_slots,
				NULL, NULL},
		{QOF_PARAM_BOOK, QOF_ID_BOOK,
				(QofAccessFunc) qof_instance_get_book, NULL, NULL},
		{QOF_PARAM_GUID, QOF_TYPE_GUID,
				(QofAccessFunc) qof_instance_get_guid, NULL, NULL},
		{NULL, NULL, NULL, NULL, NULL},
	};

	qof_class_register (PILOT_INVOICE, NULL, params);

	return qof_object_register (&expenses_object_def);
}
