#include "scaledtoplevel.h"
#include <klocale.h>
#include <qdragobject.h>
#include <qwmatrix.h>
#include <qapplication.h>
#include <kwinmodule.h>


KIFScaledTopLevel::KIFScaledTopLevel(KIFImage *image, QWidget *parent,
                               const char *name)
    : QWidget(parent, name/* ,WResizeNoErase*/)
{
    setFocusPolicy(StrongFocus);

    pix.setOptimization(QPixmap::BestOptim);
    module = new KWinModule(this);
    setBackgroundMode(NoBackground);

    // calculate window frame & titlebar size. Logo is 340x226.
    /*
    KWin::Info info = KWin::info(winId());
    QRect fR = info.frameGeometry;
    frameW = fR.width()-340;
    frameH = fR.height()-226;
    qWarning("Frame: %dx%d", frameW, frameH);
    */
    frameW = 10;
    frameH = 20;

    imageBuffer = image;
    connect(imageBuffer, SIGNAL(updated()), this, SLOT(slotUpdateFromImage()));
    connect(imageBuffer, SIGNAL(invalidFile()), this, SLOT(slotInvalidFile()));

    connect(kifapp(), SIGNAL(hideAllWindows()), this,
            SLOT(slotHideAll()));
    connect(kifapp(), SIGNAL(showAllWindows()), this,
            SLOT(slotShowAll()));
}

KIFScaledTopLevel::~KIFScaledTopLevel()
{
    qWarning("In KIFScaledTopLevel destructor");
    delete module;
}

void KIFScaledTopLevel::slotUpdateFromImage()
{
    qWarning("In slotUpdateFromImage");
    scaleImage.reset();
    pix.convertFromImage(*imageBuffer->image());
    resizeWithAspect();
}

void KIFScaledTopLevel::slotInvalidFile()
{
    ;
}

void KIFScaledTopLevel::resizeEvent(QResizeEvent *ev)
{
    if(ev->size() != pix.size()){
        QImage scaleImage(*imageBuffer->image());
        if(scaleImage.depth() < 32)
            scaleImage = scaleImage.convertDepth(32);
        scaleImage = scaleImage.smoothScale(ev->size().width(),
                                            ev->size().height());
        pix.convertFromImage(scaleImage);
    }
}


void KIFScaledTopLevel::resizeWithAspect()
{
    if(imageBuffer->image()->isNull()){
        qWarning("Imagebuffer is empty");
        return;
    }
    else if(pix.width() != width() || pix.height() != height()){
        //QRect deskRect = QRect(0, 0, QApplication::desktop()->width(),
        //                       QApplication::desktop()->height());
        QRect deskRect = module->workArea();
        deskRect.setWidth(deskRect.width()-frameW-10);
        deskRect.setHeight(deskRect.height()-frameH-10);
        QRect r = QRect(x(), y(), pix.width(), pix.height());
        QSize oldSize = size(); 
        // may need to do this twice
        if(r.width() > deskRect.width() || r.height() > deskRect.height()){
            while(r.width() > deskRect.width() || r.height() > deskRect.height()){
                if(r.width() > deskRect.width()){
                    float percent = ((float)deskRect.width())/r.width();
                    r.setWidth((int)(r.width()*percent));
                    r.setHeight((int)(r.height()*percent));
                    qWarning("Scaling width");
                }
                if(r.height() > deskRect.height()){
                    float percent = ((float)deskRect.height())/r.height();
                    r.setWidth((int)(r.width()*percent));
                    r.setHeight((int)(r.height()*percent));
                    qWarning("Scaling height");
                }
            }
        }
        if(r.bottom() > deskRect.bottom() || r.right() > deskRect.right()){
            move(deskRect.x(), deskRect.y());
        }
        resize(r.size());
        if(oldSize == size())
            repaint();
    }
    else
        repaint(false);

}

void KIFScaledTopLevel::paintEvent(QPaintEvent *ev)
{

    QRect r = ev->rect();
    if(pix.isNull() || pix.mask()){
        QPainter p;
        p.begin(this);
        p.fillRect(r, Qt::white);
        p.end();
    }
    if(pix.isNull())
        return;
    if((r.height() > pix.height() || r.width() > pix.width()) &&!pix.mask()){
        // shouldn't happen
        QPainter p;
        p.begin(this);
        QRegion region(r);
        region = region.subtract(QRect(0, 0, pix.width(), pix.height()));
        p.setClipRegion(region);
        p.fillRect(r, Qt::white);
        p.setClipping(false);
        p.end();
    }
    bitBlt(this, r.topLeft(), &pix, r, Qt::CopyROP);
}

void KIFScaledTopLevel::closeEvent(QCloseEvent *ev)
{
    qWarning("In scaledTopLevel close()");
    scaleImage.reset();
    pix.resize(0, 0);
    ev->accept();
    emit finished();
}

QSize KIFScaledTopLevel::imageSize()
{
    return(pix.size());
}

void KIFScaledTopLevel::slotHideAll()
{
    visible = isVisible();
    if(visible)
        hide();
}

void KIFScaledTopLevel::slotShowAll()
{
    if(visible)
        show();
}

void KIFScaledTopLevel::keyPressEvent(QKeyEvent *ev)
{
    int key = ev->key();
    if(key == Key_Left)
        emit(prevList());
    else if(key == Key_Right || key == Key_Space)
        emit(nextList());
    else if(key == Key_Up)
        emit(prevImage());
    else if(key == Key_Down)
        emit(nextImage());
    else if(key == Key_Escape || key == Key_Enter || key == Key_Return)
        close();
}










