/* PKS proxy: parse a configuration file */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <string.h>
#include <errno.h>
#include <ctype.h>

#include "server.h"


enum conftype
{
  DT_INT, DT_STR, DT_LIST, DT_NONE
};

static struct conf
{
  const char *instr;
  enum conftype type;
  void *ptr;
  void *dflt;
}
instr [] = 
{
  { "keyservers", 		DT_LIST, 	&Keyservers,		NULL },
  { "postponed_interval", 	DT_INT,		&PostponedInterval,	0    },
  { "port",			DT_INT,		&Port,			NULL },
  { "spool_dir",		DT_STR,		&SpoolDir,		SPOOLDIR },
  { "recheck",			DT_INT,		&RecheckInterval,	NULL },
  { NULL,			DT_NONE,	NULL }
};

static int parse_list (const char *, void *);
static int parse_str  (const char *, void *);
static int parse_int  (const char *, void *);

void pkspxy_init_config (void)
{
  int i;

  for (i = 0; instr[i].instr; i++)
  {
    if (instr[i].type == DT_STR)
    {
      if (instr[i].dflt)
	*((char **) instr[i].ptr) = strdup (instr[i].dflt);
      else
	*((char **) instr[i].ptr) = NULL;
    }
  }
}

int pkspxy_parse_config (const char *fname)
{
  char buff[2048];
  char *s;
  FILE *fp;
  int i;
  int rv = 0;
  
  if (!(fp = fopen (fname, "r")))
  {
    fprintf (stderr, "%s: Can't read %s (%s).\n",
	     Progname, fname, strerror (errno));
    return -1;
  }

  while (fgets (buff, sizeof (buff), fp))
  {
    for (s = buff; *s && isspace (*s); s++)
      ;

    if (*s == '\r' || *s == '\n' || *s == '#')
      continue;

    if (Debug)
      fprintf (stderr, "%s: read config line: %s", Progname, s);

    if (!(s = strtok (s, " #\t\r\n")))
      continue;

    for (i = 0; instr[i].instr; i++)
    {
      if (!strcasecmp (instr[i].instr, s))
      {
	s = strtok (NULL, "#\r\n");
	switch (instr[i].type)
	{
	  case DT_LIST: rv = parse_list (s, instr[i].ptr); break;
	  case DT_STR:  rv = parse_str (s, instr[i].ptr); break;
	  case DT_INT: rv = parse_int (s, instr[i].ptr); break;
	  case DT_NONE: break;
	}
	break;
      }
    }

    if (rv != 0)
      break;
  }
  
  fclose (fp);
  
  return rv;
}
	  
static int parse_list (const char *str, void *ptr)
{
  struct pkspxy_strlist **lp = (struct pkspxy_strlist **) ptr;
  struct pkspxy_strlist *l;
  char *s;
  
  for (l = *lp; l; l = l->next)
    lp = &l->next;
  
  s = strtok (str, " \t");
  while (s)
  {
    l = calloc (1, sizeof (struct pkspxy_strlist));
    l->str = strdup (s);
    *lp = l;
    lp = &l->next;
    
    s = strtok (NULL, " \t");
  }

  return 0;
}

static int parse_str (const char *str, void *ptr)
{
  char **s = (char **) ptr;
  char *t = strtok (str, " \t\r\n#");

  if (!t)
  {
    fprintf (stderr, "Missing parameter.\n");
    return -1;
  }
  
  if (*s) free (*s);
  *s = strdup (str);
  return 0;
}

static int parse_int (const char *str, void *ptr)
{
  int *t = (int *) ptr;
  char *s = strtok (str, " \t\r\n#");
  if (!s)
  {
    fprintf (stderr, "Missing parameter.\n");
    return -1;
  }
  
  *t = atoi (s);
  return 0;
}

