;;; planner-gnus.el --- Gnus integration for the Emacs Planner

;; Copyright (C) 2001 John Wiegley
;; Copyright (C) 2003 Sandra Jean Chua

;; Author: John Wiegley <johnw@gnu.org>
;; Maintainer: Sacha Chua <sacha@free.net.ph>
;; Version: $Version$
;; Keywords: planner, gnus
;; URL: http://sacha.free.net.ph/notebook/wiki/PlannerMode.php#gnus

;; This file is free software; you can redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation; either version 2, or (at your option)
;; any later version.

;; This file is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.

;; You should have received a copy of the GNU General Public License
;; along with GNU Emacs; see the file COPYING.  If not, write to
;; the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
;; Boston, MA 02111-1307, USA.

;;; Commentary:

;; This file adds annotations for Gnus messages. ;You will then be
;; able to use M-x planner-create-task-from-buffer to create tasks
;; from Gnus summary or message buffers with the correct annotation.
;; If you add
;;
;;     (planner-gnus-insinuate)
;;
;; to your .emacs, you can also use 'n' to create a task from a buffer.

;;; Code:

(require 'planner)
(require 'gnus)

;;;###autoload
(defun planner-gnus-insinuate ()
  "Hook Planner into Gnus.

Adds special planner keybindings to the variable `gnus-summary-article-map'.
From the Gnus summary article view, you can type:

\\<gnus-summary-mode-map>\\[gnus-summary-article-map] n planner-task-from-gnus"
  (eval-after-load "gnus-sum"
    '(define-key gnus-summary-article-map "n"
       'planner-create-task-from-buffer)))

(require 'gnus-sum)

(defun planner-gnus-get-message-id ()
  "Return the message-id of the current message."
  (save-excursion
    (if (equal major-mode 'gnus-summary-mode)
        (mail-header-message-id (gnus-data-header
                                 (assq (gnus-summary-article-number)
                                       gnus-newsgroup-data)))
      ;; Refer to the article buffer
      (set-buffer (get-buffer gnus-article-buffer))
      (set-buffer gnus-original-article-buffer)
      (goto-char (point-min))
      (let ((case-fold-search t))
        (when (re-search-forward "^Message-ID:\\s-*\\(.+\\)"
                                 (point-max) t)
          (match-string 1))))))

(defun planner-gnus-get-address (&optional header)
  "Return the address of the sender of the current message.
If HEADER is \"To\", return the recipient instead."
  (save-excursion
    (set-buffer (get-buffer gnus-article-buffer))
    (set-buffer gnus-original-article-buffer)
    (goto-char (point-min))
    (let ((case-fold-search t))
      (when (re-search-forward
             (concat "^" (or header "From") ":\\s-*\\(.+\\)") (point-max) t)
        (match-string-no-properties 1)))))

(defun planner-gnus-annotation-from-summary ()
  "If called from a Gnus summary buffer, return an annotation.
Suitable for use in `planner-annotation-functions'."
  (when (equal major-mode 'gnus-summary-mode)
    (let ((headers (gnus-data-header (assq (gnus-summary-article-number)
                                           gnus-newsgroup-data))))
      (emacs-wiki-make-link
       (concat "gnus://" gnus-newsgroup-name "/"
               (planner-gnus-get-message-id))
       (concat "E-Mail "
               (if (and planner-ignored-from-addresses
                        (string-match planner-ignored-from-addresses
                                      (mail-header-from headers)))
                   ;; Mail from me, so use the To: instead
                   (concat "to " (planner-get-name-from-address
                                  (cdr (assq 'To
                                             (mail-header-extra headers)))))
                 ;; Mail to me, so use the From:
                 (concat "from " (planner-get-name-from-address
                                  (mail-header-from headers)))))))))

(defun planner-gnus-annotation-from-message ()
  "If called from a Gnus article, return an annotation.
Suitable for use in `planner-annotation-functions'."
  (when (equal major-mode 'gnus-article-mode)
    (let ((from (planner-gnus-get-address "From")))
      (emacs-wiki-make-link
       (concat "gnus://" gnus-newsgroup-name "/"
               (planner-gnus-get-message-id))
       (concat "E-Mail "
               (if (and planner-ignored-from-addresses
                        (string-match planner-ignored-from-addresses from))
                   (concat "to " (planner-get-name-from-address
                                  (planner-gnus-get-address "To")))
                 (concat "from " (planner-get-name-from-address
                                  from))))))))

;;;###autoload
(defun planner-gnus-annotation ()
  "Return an annotation from a Gnus summary or message buffer.
Suitable for use in `planner-annotation-functions'. If you
include this, you can omit `planner-gnus-annotation-from-summary'
and `planner-gnus-annotation-from-message'."
  (or (planner-gnus-annotation-from-summary)
      (planner-gnus-annotation-from-message)))

;;;###autoload
(defun planner-gnus-browse-url (url)
  "If this is a Gnus URL, jump to it."
  (when (string-match "^gnus://\\(.+\\)/\\(.+\\)" url)
    (let ((group (match-string 1 url))
          (article (match-string 2 url)))
      (gnus-fetch-group group 1)
      (or (gnus-summary-goto-article article nil t)
          (when (fboundp 'gnus-summary-insert-cached-articles)
            (gnus-summary-insert-cached-articles)
            (gnus-summary-goto-article article nil t))
          (message "Message could not be found."))
      t)))

(fset 'planner-get-from 'planner-gnus-get-address)
(fset 'planner-get-message-id 'planner-gnus-get-message-id)
(custom-add-option 'planner-annotation-functions
                   'planner-gnus-annotation)
(add-hook 'planner-annotation-functions 'planner-gnus-annotation)
(add-to-list 'emacs-wiki-url-protocols '("gnus" planner-gnus-browse-url nil))

(provide 'planner-gnus)

;;; planner-gnus.el ends here
