/*
 *   Copyright (C) 2007 Percy Leonhardt <percy@eris23.de>
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU Library General Public License version 2 as
 *   published by the Free Software Foundation
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details
 *
 *   You should have received a copy of the GNU Library General Public
 *   License along with this program; if not, write to the
 *   Free Software Foundation, Inc.,
 *   51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */

#include "network.h"

#include <QSpinBox>
#include <QPainter>
#include <QPaintEvent>
#include <QApplication>
#include <QStyleOptionGraphicsItem>

#include <KDebug>
#include <KLocale>
#include <KDialog>
#include <KSharedConfig>

#include <plasma/svg.h>

Network::Network(QObject *parent, const QVariantList &args)
    : Plasma::Applet(parent, args),
      m_rxPackets( 0 ),
      m_txPackets( 0 ),
      m_isWireless( false ),
      m_rxState( "offline" ),
      m_txState( "offline" ),
      m_quality( "0percent" ),
      m_dialog( 0 )
{
    setHasConfigurationInterface( true );

    KConfigGroup cg = config();
    m_interface = cg.readEntry( "interface", "eth0" );
    m_pixelSize = cg.readEntry( "size", 200 );

    m_traffic = new Plasma::Svg( "widgets/network_traffic", this );
    m_traffic->setContentType( Plasma::Svg::ImageSet );
    m_traffic->resize( m_pixelSize, m_pixelSize );

//     m_plotter = new Plasma::Svg( "widgets/plot-background", this );
//     m_plotter->setContentType( Plasma::Svg::SingleImage );
//     m_plotter->resize();

    setDrawStandardBackground( false );

    dataEngine( "network" )->connectSource( m_interface, this );
    constraintsUpdated();
}

Network::~Network()
{
    delete m_dialog;
}

QSizeF Network::contentSizeHint() const
{
    return m_size;
}

void Network::constraintsUpdated()
{
    prepareGeometryChange();
    m_size = QSizeF( m_pixelSize, m_pixelSize );
    update();
}

void Network::updated( const QString& source, const Plasma::DataEngine::Data &data )
{
    Q_UNUSED( source );

    m_isWireless = data["Wireless"].toBool();

    int rxPackets = data["RX packets"].toInt();
    int txPackets = data["TX packets"].toInt();
    int linkQuality = data["Link quality"].toInt();

    QString newRxState;
    QString newTxState;
    QString newQuality;
    if ( data["Running"] == false )
    {
        newRxState = "offline";
        newTxState = "offline";
        newQuality = "0percent";
    }
    else
    {
        if ( m_rxPackets < rxPackets &&
             m_txPackets < txPackets )
        {
            newRxState = "traffic";
            newTxState = "traffic";
        }
        else if ( m_rxPackets == rxPackets &&
                  m_txPackets == txPackets )
        {
            newRxState = "notraffic";
            newTxState = "notraffic";
        }
        else if ( m_rxPackets < rxPackets )
        {
            newRxState = "traffic";
            newTxState = "notraffic";
        }
        else
        {
            newRxState = "notraffic";
            newTxState = "traffic";
        }

        if ( linkQuality < 25 )
        {
            newQuality = "25percent";
        }
        else if ( linkQuality < 50 )
        {
            newQuality = "50percent";
        }
        else if ( linkQuality < 75 )
        {
            newQuality = "75percent";
        }
        else
        {
            newQuality = "100percent";
        }
    }

    m_rxPackets = data["RX packets"].toInt();
    m_txPackets = data["TX packets"].toInt();
    if ( m_rxState != newRxState ||
         m_txState != newTxState ||
         m_quality != newQuality )
    {
        m_rxState = newRxState;
        m_txState = newTxState;
        m_quality = newQuality;
        update();
    }
}

void Network::showConfigurationInterface()
{
    if (m_dialog == 0)
    {
        m_dialog = new KDialog;
        m_dialog->setCaption( i18n( "Configure Network Monitor" ) );
	ui.setupUi(m_dialog->mainWidget());
        m_dialog->setButtons( KDialog::Ok | KDialog::Cancel | KDialog::Apply );
        connect( m_dialog, SIGNAL(applyClicked()), this, SLOT(configAccepted()) );
        connect( m_dialog, SIGNAL(okClicked()), this, SLOT(configAccepted()) );
    }
    ui.interfaceEdit->setText( m_interface );
    ui.sizeSpin->setValue( m_pixelSize );

    m_dialog->show();
}

void Network::configAccepted()
{
    if ( m_interface != ui.interfaceEdit->text() )
    {
        dataEngine( "network" )->disconnectSource( m_interface, this );
        m_interface = ui.interfaceEdit->text();
        dataEngine( "network" )->connectSource( m_interface, this );
        m_rxState = "offline";
        m_txState = "offline";
    }

    m_pixelSize = ui.sizeSpin->value();
    m_traffic->resize( m_pixelSize, m_pixelSize );

    KConfigGroup cg = config();
    cg.writeEntry( "interface", m_interface );
    cg.writeEntry( "size", m_pixelSize );
    cg.config()->sync();

    constraintsUpdated();
}

void Network::paintInterface(QPainter *p, const QStyleOptionGraphicsItem *option, const QRect &contentsRect)
{
    Q_UNUSED(option);

    p->setRenderHint( QPainter::SmoothPixmapTransform );

    QRectF rect( 0, 0, 0, 0 );
    float stateSize = m_pixelSize * 0.75;
    float qualitySize = m_pixelSize * 0.25;

    if ( m_isWireless == true )
    {
        // paint quality level
        int w = m_traffic->elementSize( m_quality ).width();
        int h = m_traffic->elementSize( m_quality ).height();
        int wMax = m_traffic->elementSize( "100percent" ).width();
        int hMax = m_traffic->elementSize( "100percent" ).height();
        float hScaled = ( ( float ) h / hMax ) * ( m_pixelSize * 0.5 );
        rect.setTopLeft( QPointF( stateSize, ( m_pixelSize * 0.5 ) - ( hScaled / 2 ) ) );
        rect.setSize( QSizeF( ( ( float ) w / wMax ) * qualitySize, hScaled ) );
        if ( m_traffic->elementExists( m_quality ) )
        {
            m_traffic->paint( p, rect, m_quality );
        }
    }
    else
    {
        // paint back monitor
        rect.setTopLeft( QPointF( contentsRect.width() - stateSize, 0 ) );
        rect.setSize( QSizeF( stateSize, stateSize ) );
        if ( m_traffic->elementExists( m_rxState ) )
        {
            m_traffic->paint( p, rect, m_rxState );
        }
    }
    // paint the front monitor
    rect.setTopLeft( QPointF( 0, contentsRect.height() - stateSize ) );
    rect.setSize( QSizeF( stateSize, stateSize ) );
    if ( m_traffic->elementExists( m_txState ) )
    {
        m_traffic->paint( p, rect, m_txState );
    }

}

#include "network.moc"
